import { getSettings } from '@/services/SettingsService'
import { getWarehousesPriorityList } from '@/support/globalSettingsHelpers'
import { prepareDateTimeStringToFormat } from '@/support/timeHelper'

const KEYS = {
  ACCOUNTING_TRANSACTIONS: [
    'nominal_code_mapping_sales_orders', 'nominal_code_mapping_cost_of_goods_sold',
    'nominal_code_mapping_inventory_control', 'nominal_code_mapping_accrued_purchases'
  ]
}

const prepareDateTimeValue = (value) => prepareDateTimeStringToFormat(value, 'UTC')

export const SettingsModule = {
  namespaced: true,
  state: {
    settings: [],
    loading: false
  },
  getters: {
    getSettings: state => state.settings,
    getDateFormat: state => state.settings.find(s => s.key === 'default_date_format') || { value: 'MMMM Do, YYYY' },
    getDateFormatValue: (_, getters) => getters.getDateFormat?.value || 'MMMM Do, YYYY',
    getDateTimeFormatValue: (_, getters) => {
      const dateFormat = getters.getDateFormatValue.includes('Do')
        ? getters.getDateFormatValue.replace('Do', 'D')
        : getters.getDateFormatValue

      return `${dateFormat}, HH:mm:ss`
    },
    getTimezone: state => state.settings.find(s => s.key === 'default_timezone') || null,
    getTimezoneValue: (_, getters) => getters.getTimezone?.value || getters.getTimezone?.default_value || null,
    getInventoryStartDate: state => {
      if (!state.settings || state.settings.length === 0) return null

      const found = state.settings.find(setting => setting.key === 'inventory_start_date')
      return found ? prepareDateTimeValue(found.value) : null
    },
    getDefaultWarehouseId: state => {
      if (state.settings.length === 0) return null
      const found = state.settings.find(setting => setting.key === 'sales_order_default_warehouse')
      return found ? found.value : null
    },
    getSalesCreditsDefaultWarehouse: state => {
      if (!state.settings || state.settings.length === 0) return null
      return state.settings.find(setting => setting.key === 'sales_credit_default_warehouse') || null
    },
    getDefaultWeightUnit: state => {
      if (!state.settings || state.settings.length === 0) return null
      const found = state.settings.find(setting => setting.key === 'default_weight_unit')
      return found ? found.value : null
    },
    getDefaultDimensionUnit: state => {
      if (!state.settings || state.settings.length === 0) return null
      const found = state.settings.find(setting => setting.key === 'default_dimension_unit')
      return found ? found.value : null
    },
    getLoading: state => state.loading,
    getSettingByKey: state => (key) => state.settings.find(s => s.key === key),
    isLoadedSettings: (_, getters) => getters.getSettings.length !== 0,
    getWarehousePriorityList: (_, getters) => {
      return getWarehousesPriorityList(getters.getSettings)
    },
    getAccountingTransactionsSettings: state => state.settings.filter(item => KEYS.ACCOUNTING_TRANSACTIONS.includes(item?.key)) || [],
    getFinancialAlertsEnabled: state => {
      if (!state.settings || state.settings.length === 0) return null
      const found = state.settings.find(setting => setting.key === 'financial_alerts_enabled')
      return found ? found.value : null
    }
  },
  mutations: {
    SET_SETTINGS (state, settings) {
      state.settings = settings
    },
    SET_LOADING (state, value) {
      state.loading = value
    }
  },
  actions: {
    async loadSettings ({ commit }) {
      commit('SET_LOADING', true)
      try {
        const response = await getSettings()
        commit('SET_SETTINGS', response.data)

        return Promise.resolve(response)
      } catch (e) {
        throw e
      } finally {
        commit('SET_LOADING', false)
      }
    }
  }
}
