import Vue from 'vue'
import {addPayment, fetchSalesCredits, updatePayment} from '@/services/orders/SalesCreditService'
import {isEqual} from 'lodash'
import wasArrayChanged from '@/support/wasArrayChanged'
import exactMath from 'exact-math'

export const SalesCreditModule = {
  namespaced: true,
  state: {
    itemsList: [],
    tableSpecifications: {},
    maxPages: 0,
    meta: {},
    loading: false,
    itemByIdLoading: false,

    selectedItem: null,
    drawerMode: null,// view, edit, create
    salesCreditDrawer: {
      show: false,
      hideNavigator: false,
      initialItemId: false
    },

    editedCredit: {
      store: null,
      warehouse: null,
      customer: null,
      currency: null,
      items: [],
      taxes: []
    },

    selectedIsTaxIncluded: false,
    selectedTaxRate: {
      tax_rate_id: null,
      tax_rate: null,
    },
    selectedTaxes: [],
    initialTaxes: [],

    customers: [],

    fetchedSelectValues: false,

    itemFields: [
      'item_discount',
      'item_name',
      'item_nominal_code',
      'item_price',
      'item_quantity',
      'tax_rate_id',
      'tax_rate',
    ],

    paymentModal: {
      show: false,
      deleting: false,
      payment: null,
      isRefund: false,
      actionName: '',
    },
    globalSettings: null
  },
  getters: {
    getShowBarcode: state => {
      if (!state.globalSettings) return false
      const barcodeObj = state.globalSettings.find(i => i.key === 'sales_credit_show_barcode_in_frontend')
      if (!barcodeObj) return false
      return barcodeObj.value
    },
    getItemsList: state => state.itemsList,
    getTableSpecifications: state => state.tableSpecifications,
    getMaxPages: state => state.maxPages,
    getMeta: state => state.meta,
    getItemByIdLoading: state => state.itemByIdLoading,
    getLoading: state => state.loading,

    getSelectedItem: state => state.selectedItem,
    getSelectedItemIndex: state => {
      if (!state.selectedItem) return 0
      return state.itemsList.findIndex(i => i.id === state.selectedItem.id)
    },
    getDrawerMode: state => state.drawerMode,
    getStoreChanged: state => {
      const initialValue = (state.selectedItem && state.selectedItem.store && state.selectedItem.store.id) || '-'
      const localValue = (state.editedCredit && state.editedCredit.store && state.editedCredit.store.id) || '-'
      return localValue !== initialValue
    },
    getCustomerChanged: state => {
      const initialValue = (state.selectedItem && state.selectedItem.customer_name && state.selectedItem.customer_name.id) || '-'
      const localValue = (state.editedCredit && state.editedCredit.customer && state.editedCredit.customer.id) || '-'
      return localValue !== initialValue
    },
    getCurrencyChanged: state => {
      const initialValue = (state.selectedItem?.currency_code) || '-'
      const localValue = (state.editedCredit?.currency?.code) || '-'
      return localValue !== initialValue
    },
    getWarehouseChanged: state => {
      const initialValue = (state.selectedItem && state.selectedItem.to_warehouse && state.selectedItem.to_warehouse.id) || '-'
      const localValue = (state.editedCredit && state.editedCredit.warehouse && state.editedCredit.warehouse.id) || '-'
      return localValue !== initialValue
    },
    getItemsChanged: state => {
      if (!state.selectedItem) return false
      const localItems = state.editedCredit.items
      // items added or removed
      if (localItems.length !== state.selectedItem.item_info.length) return true

      // no items - nothing to track
      if (!state.selectedItem.item_info.length) return false

      // changes inside of items
      let initialObj = {}
      let localObj = {}

      for (let i = 0; i < state.selectedItem.item_info.length; i++) {
        const initialItem = state.selectedItem.item_info[i]
        const localItem = localItems[i]

        state.itemFields.forEach(field => {
          initialObj[field] = initialItem[field] || ''
          localObj[field] = localItem[field] || ''
        })

        if (!isEqual(initialObj, localObj)) {
          return true
        }
      }
      return false
    },
    getSelectedTaxes: state => state.selectedTaxes,
    getTaxesChanged: state => {
      if (!state.selectedItem) return false

      const localItems = state.selectedTaxes || []
      const initialItems = state.initialTaxes || []

      return wasArrayChanged(localItems, initialItems)
    },
    getSomethingChanged: (_, getters) => {
      return getters.getStoreChanged || getters.getCustomerChanged || getters.getWarehouseChanged ||
        getters.getItemsChanged || getters.getIsTaxIncludedChanged || getters.getIsGeneralTaxRateChanged ||
        getters.getCurrencyChanged
    },
    getIsGeneralTaxRateChanged : (state,getters) => {
      return state.selectedItem?.tax_rate_id !== getters.getSelectedTaxRate.tax_rate_id
    },

    getSalesCreditDrawer: state => state.salesCreditDrawer,

    getEditedCredit: state => state.editedCredit,
    getCustomers: state => state.customers,
    getFetchedSelectValues: state => state.fetchedSelectValues,
    getSelectedCreditPayments: state => state.selectedItem?.payments || [],
    getPaymentModal: state => state.paymentModal,
    getSelectedIsTaxIncluded: state => state.selectedIsTaxIncluded,
    getIsTaxIncludedChanged: state => {
      if (!state.selectedItem) return false
      return !!state.selectedItem?.is_tax_included !== state.selectedIsTaxIncluded
    },
    getIsSetGeneralTaxRate: (state, getters) => {
      return state.selectedTaxRate.tax_rate_id !== null
    },
    getSelectedTaxRate: state => state.selectedTaxRate,
    getAmountPayments: (_,getters) => {
      if (!getters.getSelectedCreditPayments) return 0

      return getters.getSelectedCreditPayments.reduce((acc, payment) => {
        const amount = parseFloat(payment.amount) 
        acc = exactMath.add(acc, amount)
        return acc
      }, 0)
    },
    getTotal: (_,getters) => {
      if (!getters.getSelectedItem) return 0
      return !getters.getSelectedItem.is_tax_included ?
        exactMath.add(parseFloat(getters.getSelectedItem.total), parseFloat(getters.getSelectedItem.calculated_tax_total)):
        getters.getSelectedItem.total
    },
    getDue: (_,getters) => exactMath.sub(getters.getTotal, getters.getAmountPayments),
  },
  mutations: {
    SET_ITEMS_LIST (state, value) {
      state.itemsList = value
    },
    SET_LOADING (state, value) {
      state.loading = value
    },
    SET_ITEM_BY_ID_LOADING (state, value) {
      state.itemByIdLoading = value
    },
    SET_TABLE_SPECIFICATIONS (state, value) {
      state.tableSpecifications = value
    },
    SET_MAX_PAGES (state, value) {
      state.maxPages = value
    },
    SET_META (state, value) {
      state.meta = value
    },

    SET_SELECTED_ITEM (state, value) {
      state.selectedItem = value
    },
    SET_DRAWER_MODE (state, value) {
      state.drawerMode = value
    },
    SET_SALES_CREDIT_DRAWER_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.salesCreditDrawer, field, value)
    },
    SET_EDITED_CREDIT (state, value) {
      state.editedCredit = value
    },
    SET_EDITED_CREDIT_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.editedCredit, field, value)
    },
    SET_CUSTOMERS (state, value) {
      state.customers = value
    },
    SET_FETCHED_SELECT_VALUES (state, value) {
      state.fetchedSelectValues = value
    },
    SET_SELECTED_CREDIT_PAYMENTS (state, value) {
      Vue.prototype.$set(state.selectedItem, 'payments', value)
    },
    SET_PAYMENT_MODAL_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.paymentModal, field, value)
    },
    SET_GLOBAL_SETTINGS (state, value) {
      state.globalSettings = value
    },
    SET_EDITED_TAXES (state, value) {
      Vue.prototype.$set(state.editedCredit, 'taxes', value)
    },
    REMOVE_TAX_BY_INDEX (state, index) {
      state.editedCredit.taxes.splice(index, 1)
    },
    EDIT_TAX_BY_INDEX (state, { index, value }) {
      Vue.prototype.$set(state.editedCredit.taxes, index, value)
    },
    SET_SELECTED_IS_TAX_INCLUDED (state, value) {
      state.selectedIsTaxIncluded = value
    },
    SET_SELECTED_GENERAL_TAX_RATE (state, value) {
      state.selectedTaxRate = value
    },
    SET_INITIAL_TAXES (state, value) {
      state.initialTaxes = value
    },
    SET_SELECTED_TAXES (state, value) {
      state.selectedTaxes = value
    },
  },
  actions: {
    async getItemsRequest ({ commit }, query) {
      try {
        commit('SET_LOADING', true)
        const response = await fetchSalesCredits(query)

        commit('SET_ITEMS_LIST', response.data)
        if (response.table_specifications) commit('SET_TABLE_SPECIFICATIONS', response.table_specifications)
      } catch (e) {
        return Promise.reject(e)
      } finally {
        commit('SET_LOADING', false)
      }
    },
    async getMetaRequest ({ commit }, query) {
      try {
        const response = await fetchSalesCredits(query)
        if (response.meta && response.meta.last_page) commit('SET_MAX_PAGES', response.meta.last_page)
        if (response.meta) commit('SET_META', response.meta)
      } catch (e) {
        return Promise.reject(e)
      }
    },
    async loadTableSpecifications ({ commit }, query) {
      try {
        const response = await fetchSalesCredits(query)

        if (response.data?.table_specifications) commit('SET_TABLE_SPECIFICATIONS', response.data.table_specifications)
        return Promise.resolve()
      } catch (e) {
        throw e
      }
    },
    async addPaymentToSelectedCredit ({ state, commit }, payload) {
      if (!state.selectedItem) return
      try {
        const response = await addPayment(state.selectedItem.id, payload)
        // Add payment to payments
        commit('SET_SELECTED_ITEM', response.data)
        commit('SET_SELECTED_CREDIT_PAYMENTS', response.data.payments)
      } catch (e) {
        return Promise.reject(e)
      }
    },
    async updateSelectedCreditPayment ({state,commit}, payload) {
      if (!state.selectedItem) return
      try {
        const {data} = await updatePayment(state.selectedItem.id, payload.id, payload.payment)
        // Add payment to payments
        commit('SET_SELECTED_ITEM', data)
        commit('SET_PAYMENT_MODAL_FIELD', {field: 'payment', value: null})
        commit('SET_SELECTED_CREDIT_PAYMENTS', data.payments)
      } catch (e) {
        return Promise.reject(e)
      }
    }
  }
}
