import { getAllReportTemplates, createTemplate, updateTemplate } from '@/services/ReportTemplatesService'
import Vue from 'vue'

export const ReportsModule = {
  namespaced: true,
  state: {
    templates: [],
    timeIntervals: [
      'Day', 'Week', 'Month', 'Quarter', 'Year'
    ],
    groupBy: [
      { name: 'None', value: null }, { name: 'Brand', value: 'brand_id' }, { name: 'Supplier', value: 'supplier_id' }, { name: 'Sales Channel', value: 'sales_channel_id' }, { name: 'Store', value: 'store_id' }, { name: 'SKU', value: 'product_id' }
    ],
    metrics: [
      'Sales', 'Profit', 'Units sold'
    ],
    templatesLoading: false,
    editingTemplate: {
      metric: 'Sales',
      timeInterval: 'Day',
      filters: [],
      name: '',
      groupBy: null,
      startDate: null,
      endDate: null
    }
  },
  getters: {
    getTemplates: state => state.templates,
    getTimeIntervals: state => state.timeIntervals,
    getGroupBy: state => state.groupBy,
    getTemplatesLoading: state => state.templatesLoading,
    getEditingTemplate: state => state.editingTemplate,
    getMetrics: state => state.metrics
  },
  mutations: {
    SET_TEMPLATES (state, value) {
      state.templates = value
    },
    SET_SELECTED_TEMPLATE (state, template) {
      state.editingTemplate = template
    },
    SET_TEMPLATES_LOADING (state, status) {
      state.templatesLoading = status
    },
    SET_SELECTED_TEMPLATE_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.editingTemplate, field, value)
    }
  },
  actions: {
    /**
       * Loads the saved report templates
       * @param {*} commit
       */
    async loadTemplates ({ commit }) {
      commit('SET_TEMPLATES_LOADING', true)
      try {
        const response = await getAllReportTemplates({ model: 'reports' })
        commit('SET_TEMPLATES', response)
      } catch (e) {
        return Promise.reject(e)
      } finally {
        commit('SET_TEMPLATES_LOADING', false)
      }
    },
    /**
     * Creates a new template
     * @param {*} param0
     */
    async createTemplate ({ commit, state }) {
      try {
        const response = await createTemplate({
          model: 'reports',
          name: state.editingTemplate.name,
          content: {
            metric: state.editingTemplate.metric,
            timeInterval: state.editingTemplate.timeInterval,
            startDate: state.editingTemplate.startDate,
            endDate: state.editingTemplate.endDate,
            filters: state.editingTemplate.filters.map(row => ({
              conjunction: row.conjunction,
              field: row.field,
              operator: row.operator,
              value: row.value
            })) || [],
            groupBy: state.editingTemplate.groupBy
          }
        })
        // Add the template to the templates
        const templates = state.templates
        templates.push(response.data)
        commit('SET_TEMPLATES', templates)

        if (templates.length === 1) {
          commit('SET_SELECTED_TEMPLATE', response.data.content)
        }
      } catch (e) {
        return Promise.reject(e)
      }
    },
    async updateTemplate ({ commit, state }, reportId) {
      try {
        const response = await updateTemplate(reportId, {
          model: 'reports',
          name: state.editingTemplate.name,
          content: {
            metric: state.editingTemplate.metric,
            timeInterval: state.editingTemplate.timeInterval,
            startDate: state.editingTemplate.startDate,
            endDate: state.editingTemplate.endDate,
            filters: state.editingTemplate.filters.map(row => ({
              conjunction: row.conjunction,
              field: row.field,
              operator: row.operator,
              value: row.value
            })),
            groupBy: state.editingTemplate.groupBy
          }
        })
        // Update the template in the templates
        const templates = state.templates
        const index = templates.findIndex(item => item.id === reportId)
        if (index !== -1) {
          templates[index] = response.data
          commit('SET_TEMPLATES', templates)
        }
      } catch (e) {
        return Promise.reject(e)
      }
    }
  }
}
