import {fetchAllProductListings} from '@/services/ProductListingsService'
import Vue from 'vue'
import {isEqual} from 'lodash'

export const ProductListingsModule = {
  namespaced: true,
  state: {
    productListings: {
      itemsList: [],
      tableSpecifications: {},
      maxPages: 0,
      meta: {}
    },
    loading: false,

    selectedItem: {},
    drawerMode: 'view',
    productListingsDrawer: {
      show: false,
      hideNavigator: false,
      initialItemId: null
    },

    editedItem: {
      inheritPriceSettings: null,
      masterOfPrice: null,
      pricingTier: null,

      inheritInventorySettings: null,
      masterOfStock: null,
      selectedWarehouses: [],
      maxRuleType: null,
      maxRuleTypeValue: null,
      minRuleType: null,
      minRuleTypeValue: null,
      substractBufferStock: null,
      fulfillmentLatency: null
    },
    editedInventoryLocations: [],
    initialInventoryLocations: [],
    dashboardData: {
      loading: false,
      analytics: [],
      chart: []
    },
    showInventory: true
  },
  getters: {
    getItemsList: state => state.productListings.itemsList,
    getTableSpecifications: state => state.productListings.tableSpecifications,
    getMaxPages: state => state.productListings.maxPages,
    getMeta: state => state.productListings.meta,
    getSelectedItem: state => state.selectedItem,
    getSelectedItemIndex: state => {
      if (!state.selectedItem || !state.productListings.itemsList || !state.productListings.itemsList.length) { return -1 }
      return state.productListings.itemsList.findIndex(i => i.id === state.selectedItem.id)
    },
    getDrawerMode: state => state.drawerMode,
    getProductListingsDrawer: state => state.productListingsDrawer,

    getPricingTierChanged: state => {
      if (!state.editedItem.pricingTier) return false
      return state.editedItem.pricingTier.id !== state.selectedItem.pricing_tier_id
    },
    getInheritPriceSettingsChanged: state => {
      if (typeof state.editedItem.inheritPriceSettings !== 'boolean') return false
      return state.editedItem.inheritPriceSettings !== state.selectedItem.price_inherit
    },
    getMasterOfPriceChanged: state => {
      if (!state.editedItem.masterOfPrice) return false
      return state.editedItem.masterOfPrice.id !== state.selectedItem.master_of_price
    },
    getPricingChanged: (state, getters) => {
      if (!state.selectedItem) return false

      return getters.getInheritPriceSettingsChanged ||
        getters.getMasterOfPriceChanged ||
        getters.getPricingTierChanged
    },

    getInheritInventorySettingsChanged: state => {
      if (typeof state.editedItem.inheritInventorySettings !== 'boolean') return false
      return state.editedItem.inheritInventorySettings !== state.selectedItem.stock_inherit
    },
    getMasterOfStockChanged: state => {
      if (!state.editedItem.masterOfStock) return false
      let masterOfStockId
      switch (state.editedItem.masterOfStock) {
        case 'sku.io':
          masterOfStockId = 'sku.io'
          break
        case 'Amazon US (contact support)':
          masterOfStockId = 'sales_channel'
          break
        case 'Neither':
          masterOfStockId = 'neither'
          break
      }

      return masterOfStockId !== state.selectedItem.master_of_stock
    },
    getSelectedWarehousesChanged: state => {
      if (!state.editedItem.selectedWarehouses || !state.selectedItem.inventory_rules) return false
      const idsAfter = state.editedItem.selectedWarehouses.map(i => i.id)
      if (!state.selectedItem.inventory_rules && idsAfter) {
        return !!idsAfter.length
      }
      const idsBefore = state.selectedItem.inventory_rules.selectedWarehouses && state.selectedItem.inventory_rules.selectedWarehouses.map(i => i.id)
      return !isEqual(idsBefore, idsAfter)
    },
    getMaxRuleTypeChanged: state => {
      if (!state.editedItem.maxRuleType || !state.selectedItem.inventory_rules) return false
      if (state.selectedItem.inventory_rules) {
        return state.editedItem.maxRuleType !== state.selectedItem.inventory_rules.maxRuleType
      }
      return !!state.editedItem.maxRuleType
    },
    getMaxRuleTypeValueChanged: state => {
      if (!state.selectedItem.inventory_rules) return false
      if (state.editedItem.maxRuleTypeValue || parseInt(state.editedItem.maxRuleTypeValue, 10) === 0) {
        if (state.selectedItem.inventory_rules) {
          return state.editedItem.maxRuleTypeValue !== state.selectedItem.inventory_rules.maxRuleTypeValue
        }
        return !!state.editedItem.maxRuleTypeValue
      }
      return false
    },
    getMinRuleTypeChanged: state => {
      if (!state.editedItem.minRuleType || !state.selectedItem.inventory_rules) return false
      if (state.selectedItem.inventory_rules) {
        return state.editedItem.minRuleType !== state.selectedItem.inventory_rules.minRuleType
      }
      return !!state.editedItem.minRuleType
    },

    getMinRuleTypeValueChanged: state => {
      if (!state.editedItem.minRuleType || !state.selectedItem.inventory_rules) return false
      if (state.selectedItem.inventory_rules) {
        return state.editedItem.minRuleTypeValue !== state.selectedItem.inventory_rules.minRuleTypeValue
      }
      return !!state.editedItem.minRuleTypeValue
    },
    getSubstractBufferStockChanged: state => {
      if (!state.selectedItem || !state.selectedItem.inventory_rules) return false
      if (state.editedItem.substractBufferStock || parseInt(state.editedItem.substractBufferStock, 10) === 0) {
        if (state.selectedItem.inventory_rules) {
          return state.editedItem.substractBufferStock !== state.selectedItem.inventory_rules.substractBufferStock
        }
        return !!state.editedItem.substractBufferStock
      }
      return false
    },
    getFulfillmentLatencyChanged: state => {
      if (!state.selectedItem || !state.selectedItem.inventory_rules) return false
      if (state.editedItem.fulfillmentLatency || parseInt(state.editedItem.fulfillmentLatency, 10) === 0) {
        if (state.selectedItem.inventory_rules) {
          return state.editedItem.fulfillmentLatency !== state.selectedItem.inventory_rules.fulfillmentLatency
        }
        return !!state.editedItem.fulfillmentLatency
      }
      return false
    },
    getInventoryChanged: (state, getters) => {
      if (!state.selectedItem) return false

      return getters.getMasterOfStockChanged || getters.getSelectedWarehousesChanged || getters.getMaxRuleTypeChanged ||
        getters.getMaxRuleTypeValueChanged || getters.getMinRuleTypeChanged || getters.getMinRuleTypeValueChanged ||
        getters.getSubstractBufferStockChanged || getters.getFulfillmentLatencyChanged || getters.getInheritInventorySettingsChanged
    },

    getSomethingChanged: (state, getters) => {
      return getters.getInventoryChanged || getters.getPricingChanged
    },
    isInventoryChanged: (_, getters) => {
      return !isEqual(getters.getInitialInventoryLocations, getters.getEditedInventoryLocations) || getters.getInheritInventorySettingsChanged
    },
    getEditedItem: state => state.editedItem,
    getShowInventory: state => state.showInventory,
    getLoading: state => state.loading,
    getEditedInventoryLocations: state => state.editedInventoryLocations,
    getInitialInventoryLocations: state => state.initialInventoryLocations,
    getDashboardAnalytics: state => state.dashboardData.analytics,
    getDashboardLoading: state => state.dashboardData.loading,
    getDashboardChartData: state => state.dashboardData.chart
  },
  mutations: {
    SET_PRODUCT_LISTINGS_ITEMS_LIST (state, value) {
      Vue.prototype.$set(state.productListings, 'itemsList', value)
    },
    SET_PRODUCT_LISTINGS_TABLE_SPECIFICATIONS (state, value) {
      Vue.prototype.$set(state.productListings, 'tableSpecifications', value)
    },
    SET_PRODUCT_LISTINGS_MAX_PAGES (state, value) {
      Vue.prototype.$set(state.productListings, 'maxPages', value)
    },
    SET_PRODUCT_LISTINGS_META (state, value) {
      Vue.prototype.$set(state.productListings, 'meta', value)
    },
    SET_SELECTED_PRODUCT_LISTING (state, productListing) {
      state.selectedItem = productListing
    },
    SET_DRAWER_MODE (state, value) {
      state.drawerMode = value
    },
    SET_PRODUCT_LISTING_DRAWER_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.productListingsDrawer, field, value)
    },
    SET_EDITED_ITEM (state, value) {
      state.editedItem = value
    },
    SET_EDITED_ITEM_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.editedItem, field, value)
    },
    SET_SHOW_INVENTORY (state, value) {
      state.showInventory = value
    },
    RESET_EDITED_ITEM (state) {
      state.editedItem = {
        inheritPriceSettings: null,
        masterOfPrice: null,
        pricingTier: null,
      }
    },
    SET_LOADING (state, value) {
      state.loading = value
    },
    SET_EDITED_INVENTORY_LOCATIONS (state, value) {
      state.editedInventoryLocations = value
    },
    SET_INITIAL_INVENTORY_LOCATIONS (state, value) {
      state.initialInventoryLocations = value
    },
    SET_DASHBOARD_DATA (state, { field, value }) {
      Vue.prototype.$set(state.dashboardData, field, value)
    }
  },
  actions: {
    async getItemsRequest ({ commit }, { productId, query }) {
      try {
        commit('SET_LOADING', true)
        const response = await fetchAllProductListings({ productId, query })

        commit('SET_PRODUCT_LISTINGS_ITEMS_LIST', response.data)
        if (response.table_specifications) commit('SET_PRODUCT_LISTINGS_TABLE_SPECIFICATIONS', response.table_specifications)
      } catch (e) {
        throw e
      } finally {
        commit('SET_LOADING', false)
      }
    },
    async getMetaRequest ({ commit, state }, { productId, query }) {
      try {
        const response = await fetchAllProductListings({ productId, query })
        if (response.meta && response.meta.last_page) commit('SET_PRODUCT_LISTINGS_MAX_PAGES', response.meta.last_page)
        if (response.meta) commit('SET_PRODUCT_LISTINGS_META', response.meta)
      } catch (e) {
        throw e
      }
    },
    async loadTableSpecifications ({ commit }, query) {
      try {
        const response = await fetchAllProductListings(query)

        if (response.data?.table_specifications) commit('SET_PRODUCT_LISTINGS_TABLE_SPECIFICATIONS', response.data.table_specifications)
        return Promise.resolve()
      } catch (e) {
        throw e
      }
    }
  }
}
