import {getAllNominalCodes, mappingNominalCodes, mappingNominalCodesFromXero} from '@/services/NominalCodesService'

export const NominalCodes = {
  namespaced: true,
  state: {
    itemsList: [],
    mappings: [],
    tableSpecifications: {},
    maxPages: 0,
    meta: {},
    loading: false,
    nominalCodes: []
  },
  getters: {
    getItemsList: state => state.itemsList,
    getTableSpecifications: state => state.tableSpecifications,
    getMaxPages: state => state.maxPages,
    getMeta: state => state.meta,
    getLoading: state => state.loading,
    getMappingsList: state => state.mappings,
    getDefaultNominalCode: (_, getters) => getters.getDefaultNominalCodeByKey('nominal_code_mapping_sales_orders'),
    getDefaultNominalCodeByKey: (_, getters) => (key) => {
      if (!getters.getMappingsList || !getters.getMappingsList.length) return ''
      const defaultNominalCode = getters.getNominalCodeByKey(key)
      return defaultNominalCode ? defaultNominalCode.name : ''
    },
    getGroupedNominalCodes: (_, getters) => {
      return getters.getItemsList.reduce((groups, item) => {
        if (!groups.includes(item.type.trim())) {
          groups.push(item.type.trim())
          groups.push(item)
        } else {
          const indexOfType = groups.indexOf(item.type.trim())
          groups.splice(indexOfType + 1, 0, item)
        }
        return groups
      }, [])
    },
    getFormattedNominalCodes: (_, getters) => {
      return getters.getItemsList.map(item => ({ ...item, name_with_code: `${item.code} - ${item.name}` }))
    },
    getNominalCodeByKey: (_, getters) => (key) => {
      if (!getters.getMappingsList || !getters.getMappingsList.length) return null
      // sales order has default nominal code, which is set in settings
      const salesOrdersDefaultMapping = getters.getMappingsList.find(i => i.key === key)
      // find default nominal code within all nominal codes
      let defaultNominalCode
      if (salesOrdersDefaultMapping && getters.getItemsList && getters.getItemsList.length) {
        defaultNominalCode = getters.getItemsList.find(i => i.id === salesOrdersDefaultMapping.value)
      }

      return defaultNominalCode
    },
    getDefaultSalesNominalCode: (_, getters) => getters.getNominalCodeByKey('nominal_code_mapping_sales_orders'),
    getNominalCodes: state => state.nominalCodes,
    getDefaultPurchaseNominalCode: (_, getters) => getters.getDefaultNominalCodeByKey('nominal_code_mapping_purchase_orders')
  },
  mutations: {
    SET_ITEMS_LIST (state, value) {
      state.itemsList = value
    },
    SET_TABLE_SPECIFICATIONS (state, value) {
      state.tableSpecifications = value
    },
    SET_MAX_PAGES (state, value) {
      state.maxPages = value
    },
    SET_META (state, value) {
      state.meta = value
    },
    SET_LOADING (state, value) {
      state.loading = value
    },
    SET_MAPPINGS (state, value) {
      state.mappings = value
    },
    SET_NOMINAL_CODES (state, value) {
      state.nominalCodes = value
    }
  },
  actions: {
    async getItemsRequest ({ commit }, query) {
      try {
        commit('SET_LOADING', true)
        const fetchingByPaymentType = query.fetchByPaymentType || false
        delete query.fetchByPaymentType
        const response = await getAllNominalCodes(query)

        const responsee = await mappingNominalCodes(fetchingByPaymentType)
        if (responsee?.data) commit('SET_MAPPINGS', responsee.data)

        if (response?.data) commit('SET_ITEMS_LIST', response.data)
        if (response.table_specifications) commit('SET_TABLE_SPECIFICATIONS', response.table_specifications)
      } catch (e) {
        throw e
      } finally {
        commit('SET_LOADING', false)
      }
    },
    async getMetaRequest ({ commit }, query) {
      try {
        const response = await getAllNominalCodes(query)
        if (response.meta && response.meta.last_page) commit('SET_MAX_PAGES', response.meta.last_page)
        if (response.meta) commit('SET_META', response.meta)
      } catch (e) {
        throw e
      }
    },
    async loadTableSpecifications ({ commit }, query) {
      try {
        const response = await getAllNominalCodes(query)

        if (response.data?.table_specifications) commit('SET_TABLE_SPECIFICATIONS', response.data.table_specifications)
        return Promise.resolve()
      } catch (e) {
        throw e
      }
    },
    async fetchMappings ({ commit }) {
      try {
        const response = await mappingNominalCodes()
        commit('SET_MAPPINGS', response.data)
      } catch (e) {
        throw e
      }
    },
    async getMappingsFromXero ({ commit }) {
      try {
        const response = await mappingNominalCodesFromXero()
        commit('SET_MAPPINGS', response.data)
      } catch (e) {
        throw e
      } finally {
        commit('SET_LOADING', false)
      }
    }
  }
}
