import {
  fetchInventoryAssemblies,
  getInventoryAssemblyById,
  openInventoryAssembly
  , updateInventoryAssembly } from '@/services/InventoryAssembliesService'
import Vue from 'vue'
import { isEmpty, cloneDeep } from 'lodash'
import { difference } from '@/support/helpers'
import moment from 'moment'

export const InventoryAssembliesModule = {
  namespaced: true,
  state: {
    inventoryAssemblies: {
      itemsList: [],
      tableSpecifications: {},
      maxPages: 0,
      meta: {},
      loading: false
    },
    itemsList: [],
    selectedItem: null,
    showCreateAssemblyModal: false,
    drawerMode: null, // view, create, edit
    inventoryAssemblyEntityDrawer: {
      show: false,
      hideNavigator: false,
      initialItemId: false,
      ignoreItemResetOnLaunch: false,
    },
    inventoryAssemblyLoading: false,
    editingAssembly: {},
    globalSettings: null,
    stockData: [],
  },
  getters: {
    getShowBarcode: state => {
      if (!state.globalSettings) return false
      const barcodeObj = state.globalSettings.find(i => i.key === 'sales_credit_show_barcode_in_frontend')
      if (!barcodeObj) return false
      return barcodeObj.value
    },
    getItemsList: state => state.inventoryAssemblies.itemsList,
    getTableSpecifications: state => state.inventoryAssemblies.tableSpecifications,
    getMaxPages: state => state.inventoryAssemblies.maxPages,
    getMeta: state => state.inventoryAssemblies.meta,
    getShowCreateAssemblyModal: state => state.showCreateAssemblyModal,
    getSelectedItem: state => state.selectedItem,
    getSelectedItemIndex: state => {
      if (!state.selectedItem || !state.inventoryAssemblies.itemsList.length) { return -1 }
      return state.inventoryAssemblies.itemsList.findIndex(i => i.id === state.selectedItem.id)
    },
    getEditingAssembly: state => state.editingAssembly,
    getEditingWarehouse: state => {
      return state.editingAssembly.warehouse
    },
    getDrawerMode: state => state.drawerMode,
    getInventoryAssemblyEntityDrawer: state => state.inventoryAssemblyEntityDrawer,
    getInventoryAssemblyLoading: state => state.inventoryAssemblyLoading,
    getSomethingChanged: (_, getters) => {
      return !isEmpty(getters.getEditingDifference)
    },
    getEditingDifference: state => {
      if (!state.editingAssembly.id || !state.selectedItem.id) return {}
      const diff = difference(state.selectedItem, state.editingAssembly)
      return diff
    },
    getAssembliesItemsLoading: state => state.inventoryAssemblies.loading,
    getStockData: state => state.stockData,
    isViewDrawerMode: (_,getters) => getters.getDrawerMode === 'view',
    isCreateDrawerMode: (_,getters) => getters.getDrawerMode === 'create',
    getProducts: (_,getters) => getters.getSelectedItem.items.filter(item => item.product_type === 'component') || [],
    getKit: (_,getters) => getters.getSelectedItem?.items.find(item => item.product_type === 'kit') || {},
  },
  mutations: {
    SET_GLOBAL_SETTINGS (state, value) {
      state.globalSettings = value
    },
    SET_INVENTORY_ASSEMBLIES_ITEMS_LIST (state, value) {
      Vue.prototype.$set(state.inventoryAssemblies, 'itemsList', value)
    },
    SET_INVENTORY_ASSEMBLIES_TABLE_SPECIFICATIONS (state, value) {
      Vue.prototype.$set(state.inventoryAssemblies, 'tableSpecifications', value)
    },
    SET_INVENTORY_ASSEMBLIES_MAX_PAGES (state, value) {
      Vue.prototype.$set(state.inventoryAssemblies, 'maxPages', value)
    },
    SET_INVENTORY_ASSEMBLIES_META (state, value) {
      Vue.prototype.$set(state.inventoryAssemblies, 'meta', value)
    },
    SET_SELECTED_INVENTORY_ASSEMBLY (state, assembly) {
      state.selectedItem = assembly
    },
    SET_EDITING_INVENTORY_ASSEMBLY (state, assembly) {
      state.editingAssembly = assembly
    },
    SET_SHOW_CREATE_ASSEMBLY_MODAL (state, value) {
      state.showCreateAssemblyModal = value
    },
    SET_DRAWER_MODE (state, value) {
      state.drawerMode = value
    },
    SET_INVENTORY_ASSEMBLY_ENTITY_DRAWER_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.inventoryAssemblyEntityDrawer, field, value)
    },
    SET_INVENTORY_ASSEMBLY_LOADING (state, loading) {
      state.inventoryAssemblyLoading = loading
    },
    SET_EDITING_ASSEMBLY_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.editingAssembly, field, value)
    },
    SET_EDITING_ASSEMBLY (state, value) {
      state.editingAssembly = value
    },
    SET_ASSEMBLIES_ITEMS_LOADING (state, value) {
      Vue.prototype.$set(state.inventoryAssemblies, 'loading', value)
    },
    SET_ITEMS_LIST (state, value) {
      state.itemsList = value
    },
    SET_STOCK_DATA (state, value) {
      state.stockData = value
    },
  },
  actions: {
    initiateSelectedItemEdit ({ state, commit }) {
      if (!state.selectedItem || !state.selectedItem.id) return
      commit('SET_EDITING_INVENTORY_ASSEMBLY', cloneDeep(state.selectedItem))
    },

    async setSelectedInventoryAssembly ({ commit }, ID) {
      if (!ID) return
      try {
        commit('SET_INVENTORY_ASSEMBLY_LOADING', true)
        const response = await getInventoryAssemblyById(ID)
  
        // Re-write dates fields for comparisons
        if (response.data.action_date) { response.data.action_date = moment(response.data.action_date).format('YYYY-MM-DD') }
  
        commit('SET_SELECTED_INVENTORY_ASSEMBLY', response.data)
  
      } catch (e) {
        return Promise.reject(e)
      } finally {
        commit('SET_INVENTORY_ASSEMBLY_LOADING', false)
      }
    },

    async updateSelectedAssembly ({ state, commit, getters }, payload) {
      try {
        const response = await updateInventoryAssembly(state.selectedItem.id, payload)
        // Update the selected item
        commit('SET_SELECTED_INVENTORY_ASSEMBLY', response.data)
        // Also update the value in the list
        Vue.prototype.$set(state.inventoryAssemblies.itemsList, getters.getSelectedItemIndex, response.data)
      } catch (e) {
        return Promise.reject(e)
      }
    },

    async openSelectedAssembly ({ state, commit, getters }, payload) {
      if (!state.selectedItem.id) return

      try {
        const response = await openInventoryAssembly(state.selectedItem.id, payload)

        // Update the selected item
        commit('SET_SELECTED_INVENTORY_ASSEMBLY', response.data)
        // Also update the value in the list
        Vue.prototype.$set(state.inventoryAssemblies.itemsList, getters.getSelectedItemIndex, response.data)
      } catch (e) {
        return Promise.reject(e)
      }
    },

    // inventory assembly requests
    async getItemsRequest ({ commit }, query) {
      try {
        commit('SET_ASSEMBLIES_ITEMS_LOADING', true)
        const response = await fetchInventoryAssemblies(query)

        // sort items
        let assemblyList = response.data.map(i => {
          if (i.items && i.items.length) {
            i.items = i.items.sort((a, b) => {
              if (a.product_type && a.product_type === 'bundle') return -1
              return 1
            })
          }
          return i
        })
        commit('SET_INVENTORY_ASSEMBLIES_ITEMS_LIST', assemblyList)
        commit('SET_ITEMS_LIST', assemblyList)
        if (response.table_specifications) commit('SET_INVENTORY_ASSEMBLIES_TABLE_SPECIFICATIONS', response.table_specifications)
      } catch (e) {
        throw e
      } finally {
        commit('SET_ASSEMBLIES_ITEMS_LOADING', false)
      }
    },
    async getMetaRequest ({ commit, state }, query) {
      try {
        const response = await fetchInventoryAssemblies(query)
        if (response.meta && response.meta.last_page) commit('SET_INVENTORY_ASSEMBLIES_MAX_PAGES', response.meta.last_page)
        if (response.meta) commit('SET_INVENTORY_ASSEMBLIES_META', response.meta)
      } catch (e) {
        throw e
      }
    },
    async loadTableSpecifications ({ commit }, query) {
      try {
        const response = await fetchInventoryAssemblies(query)

        if (response.data?.table_specifications) commit('SET_INVENTORY_ASSEMBLIES_TABLE_SPECIFICATIONS', response.data.table_specifications)
        return Promise.resolve()
      } catch (e) {
        throw e
      }
    },
  }
}
