import {getAllCustomers} from '@/services/orders/CustomersService'
import Vue from 'vue'
import {isEqual} from 'lodash'

export const CustomersModule = {
  namespaced: true,
  state: {
    itemsList: [],
    tableSpecifications: {},
    maxPages: 0,
    meta: {},
    drawerMode: null, // dashboard and edit
    itemByIdLoading: false,
    selectedItem: null,
    loading: false,

    editedCustomer: {
      name: '',
      email: '',
      phone: '',
      company_name: '',
      default_pricing_tier: null,
      default_store: null,
      default_shipping_method_domestic: null,
      default_shipping_method_international: null,
      default_warehouse: null
    },

    fetchedSelectValues: false,
    uniqueEmailError: false,

    salesOrdersSettings: {
      pricingTier: null,
      store: null,
      shippingMethodDomestic: null,
      shippingMethodInternational: null,
      warehouse: null
    },

    customerDrawer: {
      show: false,
      hideNavigator: false,
      initialItemId: false
    },
    dashboardData: {
      loading: false,
      analytics: [],
      chart: [],
    }
  },
  getters: {
    getItemsList: state => state.itemsList,
    getTableSpecifications: state => state.tableSpecifications,
    getMaxPages: state => state.maxPages,
    getMeta: state => state.meta,
    getDrawerMode: state => state.drawerMode,
    getItemByIdLoading: state => state.itemByIdLoading,
    getSelectedItem: state => state.selectedItem,
    getSelectedItemIndex: state => {
      if (!state.selectedItem) return 0
      return state.itemsList.findIndex(i => i.id === state.selectedItem.id)
    },
    getLoading: state => state.loading,
    getFields: state => {
      let fields = []
      for (let key in state.editedCustomer) {
        fields.push(key)
      }

      return fields
    },
    getEditedCustomer: state => state.editedCustomer,

    getCustomerName: state => state.editedCustomer.name,
    getCustomerPhone: state => state.editedCustomer.phone,
    getCustomerCompany: state => state.editedCustomer.company_name,
    getCustomerEmail: state => state.editedCustomer.email,

    getDefaultPricingTier: state => state.editedCustomer.default_pricing_tier,
    getDefaultStore: state => state.editedCustomer.default_store,
    getDefaultShippingDomestic: state => state.editedCustomer.default_shipping_method_domestic,
    getDefaultShippingInternational: state => state.editedCustomer.default_shipping_method_international,
    getDefaultWarehouse: state => state.editedCustomer.default_warehouse,

    getCustomerChanged: (state, getters) => {
      if (!state.selectedItem || !state.editedCustomer) return false
      let initialCustomer = {}
      let editedCustomer = {}

      getters.getFields.forEach(field => {
        initialCustomer[field] = state.selectedItem[field] || ''
        editedCustomer[field] = state.editedCustomer[field] || ''
      })

      return !isEqual(initialCustomer, editedCustomer)
    },

    getSomethingChanged: (state, getters) => {
      return getters.getCustomerChanged
    },

    getFetchedSelectValues: state => state.fetchedSelectValues,
    getUniqueEmailError: state => state.uniqueEmailError,

    getSalesOrdersSettings: state => state.salesOrdersSettings,

    getCustomerDrawer: state => state.customerDrawer,

    getDashboardAnalytics: state => state.dashboardData.analytics,
    getDashboardLoading: state => state.dashboardData.loading,
    getDashboardChartData: state => state.dashboardData.chart,
  },
  mutations: {
    SET_ITEMS_LIST (state, value) {
      state.itemsList = value
    },
    SET_TABLE_SPECIFICATIONS (state, value) {
      state.tableSpecifications = value
    },
    SET_MAX_PAGES (state, value) {
      state.maxPages = value
    },
    SET_META (state, value) {
      state.meta = value
    },
    SET_DRAWER_MODE (state, value) {
      state.drawerMode = value
    },
    SET_ITEM_BY_ID_LOADING (state, value) {
      state.itemByIdLoading = value
    },
    SET_SELECTED_ITEM (state, value) {
      state.selectedItem = value
    },
    SET_CUSTOMER_ADDRESSES (state, addresses) {
      Vue.prototype.$set(state.selectedItem, 'addresses', addresses)
    },
    DELETE_ADDRESS_BY_ID (state, id) {
      const deleteIndex = state.selectedItem.addresses.findIndex(a => a.id === id)

      if (deleteIndex) {
        state.selectedItem.addresses.splice(deleteIndex, 1)
      }
    },
    UPDATE_ADDRESS (state, address) {
      const updateIndex = state.selectedItem.addresses.findIndex(a => a.id === address.id)

      if (updateIndex || updateIndex === 0) {
        Vue.prototype.$set(state.selectedItem.addresses, updateIndex, address)
      }
    },
    ADD_ADDRESS (state, address) {
      state.selectedItem.addresses.push(address)
    },
    SET_DEFAULT_ADDRESS (state, { address, type }) {
      // clone addresses
      let addresses = JSON.parse(JSON.stringify(state.selectedItem.addresses))
      // set all addresses as not default
      addresses = addresses.map(a => {
        a[`is_default_${type}`] = false
        return a
      })
      // find address
      const targetAddress = addresses.find(a => a.id === address.id)
      // set it as default
      targetAddress[`is_default_${type}`] = true
      // assign addresses to customer
      Vue.prototype.$set(state.selectedItem, 'addresses', addresses)
    },
    SET_LOADING (state, value) {
      state.loading = value
    },
    SET_CUSTOMER_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.editedCustomer, field, value)
    },

    RESET_EDITED_CUSTOMER (state) {
      state.editedCustomer = {
        name: '',
        email: '',
        phone: '',
        company_name: '',
        default_pricing_tier: null,
        default_store: null,
        default_shipping_method_domestic: null,
        default_shipping_method_international: null,
        default_warehouse: null
      }
    },

    SET_FETCHED_SELECT_VALUES (state, value) {
      state.fetchedSelectValues = value
    },
    SET_UNIQUE_EMAIL_ERROR (state, value) {
      state.uniqueEmailError = value
    },

    SET_SALES_ORDERS_SETTINGS (state, value) {
      state.salesOrdersSettings = value
    },

    SET_CUSTOMER_DRAWER_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.customerDrawer, field, value)
    },
    SET_DASHBOARD_DATA (state, {field,value}) {
      Vue.prototype.$set(state.dashboardData, field, value)
    }
  },
  actions: {
    async getItemsRequest ({ commit }, query) {
      try {
        commit('SET_LOADING', true)
        const response = await getAllCustomers(query)

        commit('SET_ITEMS_LIST', response.data)
        if (response.table_specifications) commit('SET_TABLE_SPECIFICATIONS', response.table_specifications)
      } catch (e) {
        return Promise.reject(e)
      } finally {
        commit('SET_LOADING', false)
      }
    },
    async getMetaRequest ({ commit }, query) {
      try {
        const response = await getAllCustomers(query)

        if (response.meta && response.meta.last_page) commit('SET_MAX_PAGES', response.meta.last_page)
        if (response.meta) commit('SET_META', response.meta)
      } catch (e) {
        return Promise.reject(e)
      }
    },
    async loadTableSpecifications ({ commit }, query) {
      try {
        const response = await getAllCustomers(query)

        if (response.data?.table_specifications) commit('SET_TABLE_SPECIFICATIONS', response.data.table_specifications)
        return Promise.resolve()
      } catch (e) {
        throw e
      }
    }
  }
}
