import { fetchTransactions, fetchTransactionById, syncExternal } from '@/services/AccountingService'
import { lookupAccountingTransactions } from '@/services/LookupService'
import { getTableMetaRequest, getTableSpecificationsRequest, getItemsRequest } from '@/support/tableApiRequestHelpers'
import moment from 'moment-timezone'

export const AccountingTransactionsModule = {
  namespaced: true,
  state: () => ({
    selectedItem: null,
    itemsList: [],
    tableSpecifications: {},
    maxPages: 0,
    meta: {},
    loading: false,
    notificationId: null,
    syncing: false,
    loadingItem: false
  }),
  getters: {
    getItemsList: state => state.itemsList,
    getTableSpecifications: state => state.tableSpecifications,
    getMaxPages: state => state.maxPages,
    getMeta: state => state.meta,
    getLoading: state => state.loading,
    isLinkableHeader: (_) => (value) => ['type', 'parent_type'].includes(value),
    getSelectedItem: state => state.selectedItem,
    getSelectedItemIndex: state => {
      if (!state.selectedItem) return 0
      return state.itemsList.findIndex(i => i.id === state.selectedItem.id)
    },
    getNotificationId: state => state.notificationId,
    isActiveForSync: (state, getters, rootState) => (transaction_date) => {
      const startDate = rootState?.AccountingModule?.integration?.integration_settings?.settings?.sync_start_date ?? null

      return startDate && transaction_date ? moment(transaction_date).isBefore(startDate) : false
    },
    isSyncing: state => state.syncing,
    isLoadingItem: state => state.loadingItem
  },
  mutations: {
    SET_ITEMS_LIST (state, value) {
      state.itemsList = value
    },
    SET_TABLE_SPECIFICATIONS (state, value) {
      state.tableSpecifications = value
    },
    SET_MAX_PAGES (state, value) {
      state.maxPages = value
    },
    SET_META (state, value) {
      state.meta = value
    },
    SET_LOADING (state, value) {
      state.loading = value
    },
    SET_SELECTED_ITEM (state, value) {
      state.selectedItem = value
    },
    SET_NOTIFICATION_ID (state, value) {
      state.notificationId = value
    },
    SET_SYNCING (state, value) {
      state.syncing = value
    },
    SET_LOADING_ITEM (state, value) {
      state.loadingItem = value
    }
  },
  actions: {
    async getItemsRequest ({ commit }, query) {
      return getItemsRequest(query, commit, fetchTransactions)
    },
    async getMetaRequest ({ commit }, query) {
      return getTableMetaRequest(query, commit, fetchTransactions)
    },
    async loadTableSpecifications ({ commit }, query) {
      return getTableSpecificationsRequest(query, commit, fetchTransactions)
    },
    async fetchLookupValues ({ commit }, { value, header, page = 0, limit = 10 }) {
      try {
        return await lookupAccountingTransactions({ value, header, page, limit })
      } catch (e) {

      }
    },
    async fetchItemById ({ commit }, id) {
      commit('SET_LOADING_ITEM', true)
      try {
        const { data } = await fetchTransactionById(id)
        commit('SET_SELECTED_ITEM', data)
      } catch (e) {
        throw (e)
      } finally {
        commit('SET_LOADING_ITEM', false)
      }
    },
    async syncTransaction ({ commit, getters }) {
      commit('SET_SYNCING', true)
      try {
        await syncExternal({ ids: [getters.getSelectedItem.id] })
      } catch (e) {
        throw (e)
      } finally {
        commit('SET_SYNCING', false)
      }
    }
  }
}
