import axiosConfig from '../config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'

const baseURL = '/stock-takes'

export const getAllStockTakes = async (options = {}) => {
  try {
    const { data } = await axiosConfig.get(baseURL, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getStockTakeById = async (id) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${id}`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Create stock take
export const createStockTake = async (payload) => {
  try {
    const { data } = await axiosConfig.post(baseURL, payload)
    return data.data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Update stock take
export const updateStockTake = async (id, payload) => {
  try {
    const { data } = await axiosConfig.put(`${baseURL}/${id}`, payload)
    return data.data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Initiates the stock take
export const initiateStockTake = async (id, payload = {}) => {
  const url = `${baseURL}/${id}/initiate`

  try {
    const { data } = await axiosConfig.post(url, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Revert to draft
export const revertStockTakeToDraftMode = async (id) => {
  const url = `${baseURL}/${id}/revert-to-draft`

  try {
    const { data } = await axiosConfig.post(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Finalizes the stock take
export const finalizeStockTake = async (id) => {
  const url = `${baseURL}/${id}/finalize`

  try {
    const { data } = await axiosConfig.post(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveStockTakeByID = async (id) => {
  const url = `${baseURL}/${id}/archive`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveStockTakeById = async (id) => {
  const url = `${baseURL}/${id}/unarchived`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveStockTakes = async (sendData) => {
  const url = `${baseURL}/archive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveStockTakes = async (sendData) => {
  const url = `${baseURL}/unarchive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteStockTakeByID = async (id) => {
  const url = `${baseURL}/${id}`

  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteStockTakes = async (sendData) => {
  try {
    const { data } = await axiosConfig.delete(baseURL, { params: sendData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const isDeletableStockTakes = async (ids) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/deletable`, { ids })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const translateCsvListForStockTake = async (id, file) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${id}/import-lines`, { file })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}
