import axiosConfig from '../config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'
import {getAxiosRequest, postAxiosRequest} from '@/support/apiRequestHelpers'

const baseURL = '/reporting'

const getUrl = (path = '') => {
  return path ? `${baseURL}/${path}` : baseURL
}

/**
 * Gets the report for a given product
 * based on the provided dates
 *
 * @param {Number} productId
 * @param {Date} from
 * @param {Date} to
 * @param {Object} cancelToken (axios.CancelToken)
 */
export const getForProduct = async (productId, from, to, cancelToken = null) => {
  const options = {
    filters: JSON.stringify({
      conjunction: 'and',
      filterSet: [
        {
          column: 'reporting_date',
          operator: 'isWithin',
          value: { from: from, to: to }
        },
        {
          column: 'product_id',
          operator: '=',
          value: productId
        }
      ]
    }),
    cancelToken
  }

  return getAllReportingProducts(options, 'reporting_date')
}

/**
 * Gets the top selling products
 *
 * @param {Date} from
 * @param {Date} to
 */
export const getTopSellingProducts = async (from, to) => {
  const filters = {
    filterSet: [
      {
        column: 'reporting_date',
        operator: 'isWithin',
        value: { from: from, to: to }
      }
    ]
  }

  const url = `${baseURL}/top-selling-products?filters=${JSON.stringify(filters)}`

  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getMostProfitableProducts = async (from, to) => {
  const filters = {
    filterSet: [
      {
        column: 'reporting_date',
        operator: 'isWithin',
        value: { from: from, to: to }
      }
    ]
  }

  const url = `${baseURL}/most-profitable-products?filters=${JSON.stringify(filters)}`

  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

/**
 * Gets reporting orders
 *
 * @param {Date} from
 * @param {Date} to
 * @param {Number} limit
 */
export const getReportingOrders = async (from, to, limit = -1) => {
  const options = {
    filters: JSON.stringify({
      conjunction: 'and',
      filterSet: [
        {
          column: 'order_date',
          operator: 'isWithin',
          value: { from: from, to: to }
        }
      ]
    }),
    limit
  }

  return getAllReportingOrders(options, 'order_date')
}

/**
 * Fetches reporting orders for sales channel
 *
 * @param {*} channelId
 * @param {*} from
 * @param {*} to
 * @param {*} limit
 *
 */
export const getReportingOrdersForSalesChannel = async (channelId, from, to, limit = -1) => {
  const options = {
    filters: JSON.stringify({
      conjunction: 'and',
      filterSet: [
        {
          column: 'order_date',
          operator: 'isWithin',
          value: { from: from, to: to }
        },
        {
          column: 'sales_channel_id',
          operator: '=',
          value: channelId
        }
      ]
    }),
    limit: limit
  }

  return getAllReportingOrders(options, 'order_date')
}

/**
 * Fetches reporting orders for product listing
 * @param {*} listingId
 * @param {*} from
 * @param {*} to
 * @param {*} limit
 */
export const getReportingOrdersForListing = async (listingId, from, to, limit = 10) => {
  const options = {
    filters: JSON.stringify({
      conjunction: 'and',
      filterSet: [
        {
          column: 'order_date',
          operator: 'isWithin',
          value: { from: from, to: to }
        },
        {
          column: 'product_listing_id',
          operator: '=',
          value: listingId
        }
      ]
    }),
    limit: limit
  }

  return getAllReportingOrders(options, 'order_date')
}

/**
 * Gets reporting orders for customer
 * @param {*} customerId
 * @param {*} from
 * @param {*} to
 * @param {*} limit
 */
export const getReportingOrdersForCustomer = async (customerId, from, to, limit = 3) => {
  const options = {
    filters: JSON.stringify({
      conjunction: 'and',
      filterSet: [
        {
          column: 'order_date',
          operator: 'isWithin',
          value: { from: from, to: to }
        },
        {
          column: 'customer_id',
          operator: '=',
          value: customerId
        }
      ]
    }),
    limit: limit
  }

  return getAllReportingOrders(options)
}

export const getReportingOrdersForSupplier = async (customerId, from, to, limit = 3) => {
  const options = {
    filters: JSON.stringify({
      conjunction: 'and',
      filterSet: [
        {
          column: 'order_date',
          operator: 'isWithin',
          value: { from: from, to: to }
        },
        {
          column: 'supplier_id',
          operator: '=',
          value: customerId
        }
      ]
    }),
    limit: limit
  }

  return getAllReportingOrders(options, 'order_date')
}

export const fetchReportingOrders = async (filters = {}, limit = -1) => {
  const options = {
    filters: JSON.stringify(filters),
    limit: limit
  }
  return getAllReportingOrders(options, 'order_date')
}

/**
 * Gets reporting orders
 * @param {Object} options
 * @param {string} groupBy
 */
export const getAllReportingOrders = async (options = {}, groupBy = null) => {
  // Set limit to -1 if limit isn't set in the options.
  // This will fetch all resources
  if (!options.limit) options.limit = -1
  if (groupBy) options.group_by = groupBy
  try {
    const { data } = await axiosConfig.get(`${baseURL}/orders`, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

/**
 * Gets reporting products
 * @param {*} options
 * @param {*} groupBy
 */
export const getAllReportingProducts = async (options = {}, groupBy = null) => {
  // Set limit to -1 if limit isn't set in the options.
  // This will fetch all resources
  if (!options.limit) options.limit = -1
  if (groupBy) options.group_by = groupBy
  try {
    const { data } = await axiosConfig.get(`${baseURL}/products`, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getPurchasesSummaryByPeriod = async (filters) => {
  return getAxiosRequest(`${baseURL}/purchases/summary-by-period`, { params: filters })
}

export const getAccountingIncomeStatetment = async (params) => {
  return getAxiosRequest(getUrl('accounting/income-statement'), { params })
}
export const getShipBySchedule = async (params) => getAxiosRequest(getUrl('ship-by-schedule', { params }))

export const getFinancialsByBrand = async (filters) => {
  return getAxiosRequest(getUrl('financials-by-brand'), { params: filters })
}

export const getFinancialsBySupplier = async (filters) => {
  return getAxiosRequest(getUrl('financials-by-supplier'), { params: filters })
}

export const getFinancialsBySku = async (filters) => {
  return getAxiosRequest(getUrl('financials-by-sku'), { params: filters })
}

export const getFinancialsBySalesChannel = async (filters) => {
  return getAxiosRequest(getUrl('financials-by-sales-channel'), { params: filters })
}

export const getFinancialsByProductType = async (filters) => {
  return getAxiosRequest(getUrl('financials-by-product-type'), { params: filters })
}

export const getProductSalesChannelCoverage = (sendData) => postAxiosRequest(`${baseURL}/sales-channel-coverage?`, sendData)

