import axiosConfig from '@/config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'
import { getAxiosRequest } from '@/support/apiRequestHelpers'
import { isFailedRequest } from '@/support/integrations/integrationInstanceHelpers'

const baseURL = '/qbo'

export const fetchTransactions = async (options = {}) => {
  try {
    const { data } = await axiosConfig.get(baseURL, buildUrlParams(options))
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const syncTransactionById = async (id) => {
  const url = `${baseURL}/${id}/sync`
  try {
    const { data } = await axiosConfig.post(url)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const syncMultipleTransactions = async (sendData) => {
  const url = `${baseURL}/sync-multiple`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const syncTransactions = async (instanceId) => {
  const url = `${baseURL}/${instanceId}/sync-transactions`
  try {
    const { data } = await axiosConfig.post(url)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const checkIfQuickBooksIsEnabled = async () => {
  const url = `${baseURL}/check-if-quickBooks-is-enabled`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const downloadTaxRates = async (instanceId) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${instanceId}/tax-rates/refresh`)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    console.log(e)
    return Promise.reject(e)
  }
}

export const getQuickBooksTaxRates = async (instanceId) => {
  const url = `${baseURL}/${instanceId}/tax-rates`
  try {
    const { data } = await axiosConfig.get(url)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    console.log(e)
    return Promise.reject(e)
  }
}

export const getQuickBooksTaxRatesMappings = async () => {
  const url = `${baseURL}/tax-mappings`
  try {
    const data = await axiosConfig.get(url)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    console.log(e)
    return Promise.reject(e)
  }
}

export const mapQuickBooksTaxRates = async (sendData) => {
  const url = `${baseURL}/map-tax-rates`
  try {
    const { data } = await axiosConfig.put(url, sendData)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    console.log(e)
    return Promise.reject(e)
  }
}

export const deleteTaxRateMapping = async (mappingId) => {
  const url = `${baseURL}/tax-rate-mapping/${mappingId}`
  try {
    const { data } = await axiosConfig.delete(url)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    console.log(e)
    return Promise.reject(e)
  }
}

// Get Mapping Nominal Codes From QuickBooks
export const downloadAccounts = async () => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/accounts/download`)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Create QuickBooks SKU Nominal Codes
export const createSkuNominalCodes = async (instanceId, options) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${instanceId}/accounts/create-nominal-codes`, options);
    if (isFailedRequest(data)) return Promise.reject({ response: { data } });
    return data;
  } catch (e) {
    return Promise.reject(e);
  }
};

export const getAccounts = async (instanceId, options) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${instanceId}/accounts`, buildUrlParams(options))
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// get Tax Rates
export const fetchTaxRatesForMapping = async () => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}?included=["name","rate"]&limit=-1`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const syncContacts = async (instanceId) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${instanceId}/vendors/sync`)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// authorize
export const authQuickBooks = async (instanceId) => getAxiosRequest(`${baseURL}/${instanceId}/get-redirect-url`)
