import axiosConfig from '../config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'
import { getAxiosRequest } from '@/support/apiRequestHelpers'

const baseURL = '/products'

// Get All Products
export const getAllProducts = async (options = {}) => {
  try {
    const { data } = await axiosConfig.get(baseURL, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Get Product By ID
export const getProductByID = async (productID, cancelToken = null) => {
  const url = `${baseURL}/${productID}`

  try {
    const { data } = await axiosConfig.get(url, { cancelToken })
    return data.data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Get Product By ID with supplier variants
export const getProductByIDWithSupplierVariants = async (productID, supplierID, cancelToken = null) => {
  const url = `${baseURL}/${productID}?supplier_id=${supplierID}`

  try {
    const { data } = await axiosConfig.get(url, { cancelToken })
    return data.data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Get Product By SKU
export const getProductBySKU = async (productSKU, cancelToken = null) => {
  productSKU = encodeURIComponent(productSKU)
  const url = `${baseURL}/by-sku?sku=${productSKU}`

  try {
    const { data } = await axiosConfig.get(url, { cancelToken })
    return data.data
  } catch (e) {
  }
}

// Create Product
export const createProduct = async (productData) => {
  try {
    const { data } = await axiosConfig.post(baseURL, productData)
    return data.data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Update Product By ID
export const updateProductByID = async (productID, productData) => {
  const url = `${baseURL}/${productID}`

  try {
    const { data } = await axiosConfig.put(url, productData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Archive Product By ID
export const archiveProductByID = async (productID) => {
  const url = `${baseURL}/${productID}/archive`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveProductByID = async (productID) => {
  const url = `${baseURL}/${productID}/unarchived`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Archive Products By IDs or filter
export const archiveProducts = async (sendData) => {
  const url = `${baseURL}/archive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Unarchive Products By IDs or filter
export const unarchiveProducts = async (sendData) => {
  const url = `${baseURL}/unarchive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Delete Product By ID
export const deleteProductByID = async (productID) => {
  const url = `${baseURL}/${productID}`

  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Delete Products By IDs or filter
export const deleteProducts = async (sendData) => {
  try {
    const { data } = await axiosConfig.delete(baseURL, { params: sendData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const isDeletableProducts = async (ids) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/is-deletable`, { ids })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const makeDefaultSupplier = async (productID, supplierID) => {
  const url = `${baseURL}/${productID}/set-default-supplier/${supplierID}`
  return axiosConfig.put(url)
}

export const getProductSuppliers = async (productID) => {
  return getAxiosRequest(`${baseURL}/${productID}/suppliers`)
}

// Get Product Pricing Tiers
export const getProductPricingTiers = async (options = {}) => {
  const url = `/product-pricing-tiers`
  try {
    const { data } = await axiosConfig.get(url, buildUrlParams(options, true))
    return data.data
  } catch (e) {
    throw e
  }
}

// Get Product Images
export const getProductImages = async (productID, cancelToken = null) => {
  const url = `${baseURL}/${productID}/images`

  try {
    const { data } = await axiosConfig.get(url, { cancelToken })
    return data.data
  } catch (e) {
    throw e
  }
}

// Create Product image
export const createProductImage = async (productID, productImagePayload) => {
  const url = `${baseURL}/${productID}/images`

  try {
    const { data } = await axiosConfig.post(url, productImagePayload)
    return data.data
  } catch (e) {
    throw e
  }
}

// Update Product Images in Bulk
export const updateProductBulkImages = async (productID, payload) => {
  const url = `${baseURL}/${productID}/bulk-images`

  try {
    const { data } = await axiosConfig.post(url, { images: payload })
    return data
  } catch (e) {
    throw e
  }
}

// Update Product Image
export const updateProductImage = async (productImageID, productImagePayload) => {
  const url = `/product-images/${productImageID}`

  try {
    const { data } = await axiosConfig.put(url, productImagePayload)
    return data.data
  } catch (e) {
    throw e
  }
}

// Delete Product Image
export const deleteProductImage = async (productImageID) => {
  const url = `/product-images/${productImageID}`

  try {
    const { data } = await axiosConfig.delete(url)
    return data.data
  } catch (e) {
    throw e
  }
}

// Get Potential Bundles
export const getPotentialBundles = async (productID) => {
  const url = `${baseURL}/${productID}/potential-bundles`

  try {
    const { data } = await axiosConfig.get(url)
    return data.data
  } catch (e) {
    throw e
  }
}

// Get Product Inventory
export const getProductInventory = async (productID, noFba = false, noSuppliers = false) => {
  const url = `${baseURL}/${productID}/inventory`

  try {
    const { data } = await axiosConfig.get(url, { params: { no_fba: noFba, no_suppliers: noSuppliers } })
    return data.data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getProductInventoryFba = async (warehouseId, productID) => {
  const url = `${baseURL}/${warehouseId}/${productID}/inventory-fba`

  try {
    const { data } = await axiosConfig.get(url)
    return data.data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Get Product Initial Inventory
export const getProductInitialInventory = async (productID) => {
  const url = `${baseURL}/${productID}/initial-inventory`

  try {
    const { data } = await axiosConfig.get(url)
    return data.data
  } catch (e) {
    throw e
  }
}

// Get attributes group for product
export const getProductAttributeGroups = async (productId) => await getAxiosRequest(`${baseURL}/${productId}/attributes-grouped`, productId)

export const getProductFifoLayers = async (productId) => {
  const url = `${baseURL}/${productId}/fifo-layers`

  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const previewImportData = async (sendData) => {
  const url = `${baseURL}/import/preview`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const importData = async (sendData) => {
  const url = `${baseURL}/import`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchProductVariationsAttributes = async (id) => {
  const url = `${baseURL}/${id}/variations/attributes`

  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const sharedAttributesForProductList = async (ids) => {
  const url = `${baseURL}/attributes`

  try {
    const { data } = await axiosConfig.get(url, { params: { product_ids: JSON.stringify(ids) } })
    return data.data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const createStockTake = async (ids, warehouseId) => {
  const url = `/stock-takes/bulk-insert`

  try {
    const { data } = await axiosConfig.post(url,
      { ids: ids, warehouse_id: warehouseId })
    return data.data
  } catch (e) {
    return Promise.reject(e)
  }
}
