import axiosConfig from '../config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'
import {postAxiosRequest} from '@/support/apiRequestHelpers'

const baseURL = '/integrations'
const salesChannels = 'sales-channels'

export const INTEGRATION_VERACORE_NAME = 'Veracore'

export const fetchIntegrations = async (all = false) => {
  try {
    const { data } = await axiosConfig.get(all ? `${baseURL}/all` : baseURL)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getIntegrationInstanceById = async (id) => {
  const url = `/integration-instances/${id}`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getIntegrationsByName = async (name) => {
  const url = `/integrations`
  const options = {
    filters: {
      conjunction: 'and',
      filterSet: [{ column: 'name', operator: '=', value: name }]
    }
  }
  try {
    const { data } = await axiosConfig.get(url, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getIntegrationsByType = async (type) => {
  const url = `/integrations`
  const options = {
    filters: {
      conjunction: 'and',
      filterSet: [{ column: 'integration_type', operator: '=', value: type }]
    }
  }
  try {
    const { data } = await axiosConfig.get(url, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const editIntegrationInstance = async (id, sendData) => {
  const url = `/integration-instances/${id}`
  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteIntegrationInstance = async (id) => {
  const url = `/integration-instances/${id}`
  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchListings = async (options = {}) => {
  const url = `/integration-instances/${options.integrationInstanceId}/listings`
  try {
    const { data } = await axiosConfig.get(url, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteAllFba = async (instanceId) => {
  const url = `/integration-instances/${instanceId}/fba`
  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchListingById = async (instancdId, listingId) => {
  const url = `/integration-instances/${instancdId}/listings/${listingId}`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const mapListings = async (id, sendData) => {
  const url = `/integration-instances/${id}/map-listing`
  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unmapListings = async (id, sendData) => {
  const url = `/integration-instances/${id}/unmap-listing`
  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const downloadListings = async (instanceid) => {
  const url = `/integration-instances/${instanceid}/listings/download`
  try {
    const { data } = await axiosConfig.post(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteListingById = async (instanceid, listingId) => {
  const url = `/integration-instances/${instanceid}/listings/${listingId}`
  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteListings = async (instanceid, sendData) => {
  const url = `/integration-instances/${instanceid}/listings`
  try {
    const { data } = await axiosConfig.delete(url, { params: sendData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const isDeletableListing = async (instanceid, ids) => {
  const url = `/integration-instances/${instanceid}/listings/deletable`
  try {
    const { data } = await axiosConfig.post(url, { ids })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchCommunications = async (options = {}) => {
  const url = `/integration-instances/${options.integrationInstanceId}/communications`
  try {
    const { data } = await axiosConfig.get(url, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchCommunicationErrors = async (options = {}) => {
  const url = `/integration-instances/${options.integrationInstanceId}/errors`
  try {
    const { data } = await axiosConfig.get(url, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const legacyDownloadOrders = async (id, sendData) => {
  const url = `/integration-instances/${id}/download-orders`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const refreshOrder = async (path, integrationInstanceId, request) => {
  const url = `/${path}/${integrationInstanceId}/orders/refreshOrder`
  try {
    const { data } = await axiosConfig.post(url, request)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const refreshOrders = async (path, integrationInstanceId) => {
  const url = `/${path}/${integrationInstanceId}/orders/refresh`
  try {
    const { data } = await axiosConfig.post(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const downloadMagentoOrders = async (id, sendData) => {
  const url = `/magento/${id}/download-orders`
  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const downloadPreAuditTrailOrders = async (id, status) => {
  const url = `/shopify/${id}/download-pre-audit-trail-orders`
  try {
    const { data } = await axiosConfig.get(url, { params: { status } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const downloadInboundShipments = async (id, sendData) => {
  const url = `/integration-instances/${id}/download-inbound-shipments`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const initIntegrationInstance = async (sendData) => {
  const url = `/integration-instances`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchInstanceResources = async (model, options = {}) => {
  const url = `/integration-instances/${options.integrationInstanceId}/${model}`

  try {
    const { data } = await axiosConfig.get(url, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchInstanceResourceById = async (model, instanceId, modelId) => {
  const url = `/integration-instances/${instanceId}/${model}/${modelId}`

  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchInstanceOrders = async (options = {}) => {
  const url = `/integration-instances/${options.integrationInstanceId}/orders`

  try {
    const { data } = await axiosConfig.get(url, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchInstanceOrderById = async (instanceId, orderId) => {
  const url = `/integration-instances/${instanceId}/orders/${orderId}`

  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveInstanceResource = async (model, instanceId, orderId) => {
  const url = `/integration-instances/${instanceId}/${model}/${orderId}/archive`
  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveInstanceResource = async (model, instanceId, orderId) => {
  const url = `/integration-instances/${instanceId}/${model}/${orderId}/unarchived`
  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveInstanceResources = async (model, instanceId, ids) => {
  const url = `/integration-instances/${instanceId}/${model}/archive`
  try {
    const { data } = await axiosConfig.put(url, ids)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveInstanceResources = async (model, instanceId, ids) => {
  const url = `/integration-instances/${instanceId}/${model}/unarchived`
  try {
    const { data } = await axiosConfig.put(url, ids)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveInstanceOrder = async (instanceId, orderId) => {
  const url = `/integration-instances/${instanceId}/orders/${orderId}/archive`
  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveInstanceOrder = async (instanceId, orderId) => {
  const url = `/integration-instances/${instanceId}/orders/${orderId}/unarchived`
  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveInstanceOrders = async (instanceId, ids) => {
  const url = `/integration-instances/${instanceId}/orders/archive`
  try {
    const { data } = await axiosConfig.put(url, ids)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveInstanceOrders = async (instanceId, ids) => {
  const url = `/integration-instances/${instanceId}/orders/unarchived`
  try {
    const { data } = await axiosConfig.put(url, ids)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchInstanceShippingMethods = async (instanceId, options = {}) => {
  const url = `/integration-instances/${instanceId}/shipping-services`
  try {
    const { data } = await axiosConfig.get(url, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}
// Sales Channels
export const fetchSalesChannelShippingMethods = async (id, options = {}) => {
  const url = `/sales-channels/${id}/shipping-methods`
  try {
    const { data } = await axiosConfig.get(url, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const mapInstanceShippingMethods = async (instanceId, sendData) => {
  const url = `/integration-instances/${instanceId}/map-shipping-methods`
  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const mapSalesChannelShippingMethods = async (channelId, sendData) => {
  const url = `/sales-channels/${channelId}/map-shipping-methods`
  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchSalesChannelPaymentMethods = async (channelId, params = {}) => {
  const url = `/${salesChannels}/${channelId}/payment-methods`
  try {
    const { data } = await axiosConfig.get(url, { params })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const mapSalesChannelPaymentMethods = async (channelId, sendData) => {
  const url = `/${salesChannels}/${channelId}/map-payment-methods`
  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteSalesChannelPaymentMethodMapping = async (channelId) => {
  const url = `/${salesChannels}/payment-method-mapping/${channelId}`
  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteSalesChannelShippingMethodMapping = async (channelId) => {
  const url = `/${salesChannels}/shipping-method-mapping/${channelId}`
  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const lookupSalesChannelPaymentMethods = async (channelId, params = {}) => {
  const url = `/lookup/${salesChannels}/${channelId}/payment-methods`
  try {
    const { data } = await axiosConfig.get(url, { params, cancelToken: params.cancelToken })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchShippingProviderShippingMethods = async (instanceId, options = {}) => {
  const url = `/shipping-providers/${instanceId}/shipping-methods`
  try {
    const { data } = await axiosConfig.get(url, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const mapShippingProviderShippingMethods = async (salesChannelId, sendData) => {
  const url = `/shipping-providers/${salesChannelId}/map-shipping-methods`
  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unmapSalesChannelShippingMethod = async (salesChannelId, salesChannelMethodName) => {
  const url = `/sales-channels/${salesChannelId}/unmap-shipping-method/${salesChannelMethodName}`
  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteShippingProviderShippingMethod = async (instanceId, mappingId) => {
  const url = `/shipping-providers/${instanceId}/destroy-shipping-method/${mappingId}`
  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const askMapShippingMethodsCount = async (channelId, shippingMethod) => {
  const url = `/sales-channels/${channelId}/shipping-method-count`
  try {
    const { data } = await axiosConfig.get(url, { params: { shipping_method: shippingMethod } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const createOrderFromInstanceOrder = async (instanceId, sendData) => {
  const url = `/integration-instances/${instanceId}/orders/create-sku`
  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const importListingMappings = async (instanceId, sendData) => {
  const url = `/integration-instances/${instanceId}/import-mappings`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const syncShippingMethods = async (instanceId) => {
  const url = `shipstation/${instanceId}/download-shipping-services?limit=-1`
  try {
    const { data } = await axiosConfig.post(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const createProductsFromListings = async (instanceId, sendObj, filters = {}) => {
  const url = `integration-instances/${instanceId}/listings/create-sku-products`
  let params = {}
  if (filters && filters.conjunction) params.filters = JSON.stringify(filters)

  try {
    const { data } = await axiosConfig.post(url, sendObj, {
      params
    })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchAvailableMappingCount = async (instanceId, filters) => {
  const url = `integration-instances/${instanceId}/listings/mapping-info`
  let params = {}
  if (filters && filters.conjunction) params.filters = JSON.stringify(filters)
  try {
    const { data } = await axiosConfig.get(url, { params })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const checkLastMapping = async (instanceId = 1) => {
  let filters = {
    conjunction: 'and',
    filterSet: [{
      column: 'integration_instance_id',
      operator: '=',
      value: Number(instanceId)
    }]
  }
  const url = `import/listings/last-mappings`
  try {
    const { data } = await axiosConfig.get(url, {
      params: {
        filters: JSON.stringify(filters)
      }
    })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const requestFbaReports = async (instanceId) => {
  const url = `integration-instances/${instanceId}/request-fba-reports`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const updateShipstationOrder = async (orderFulfillmentId) => {
  const url = `shipstation/update-order/${orderFulfillmentId}`
  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const updateVeracoreOrder = async (orderFulfillmentId) => {
  const url = `veracore/update-order/${orderFulfillmentId}`
  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const updateStarshipitOrder = async (orderFulfillmentId) => {
  const url = `starshipit/update-order/${orderFulfillmentId}`
  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const requestInventoryPreview = async (instanceId, locationId) => {
  const url = `integration-instances/${instanceId}/sync-inventory/preview`
  const params = { location_id: locationId }
  try {
    const { data } = await axiosConfig.get(url, { params })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const downloadShopifyLocations = async (instanceId) => {
  const url = `/shopify/${instanceId}/locations`
  try {
    const { data } = await axiosConfig.post(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const downloadMagentoStores = async (id) => {
  const url = `/integration-instances/${id}/magento/download-stores`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getMagentoStores = async (id) => {
  const url = `/integration-instances/${id}/magento/stores`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getMagentoMappings = async (id) => {
  const url = `/integration-instances/${id}/magento/store/mapping`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const mapMagentoStores = async (instanceId, sendData) => {
  const url = `/integration-instances/${instanceId}/magento/store/mapping`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteMagentoMapping = async (instanceId) => {
  const url = `/integration-instances/${instanceId}/magento/store/mapping`
  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const downloadMagentoInventorySources = async (instanceId) => {
  const url = `/magento/${instanceId}/inventory-sources`
  try {
    const { data } = await axiosConfig.post(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const saveMappings = async (instanceId, sendData) => postAxiosRequest(`/import/${instanceId}/listings/mappings`, sendData)

export const importSalesChannelProductMappings = async (integrationName, instanceId, sendData) => postAxiosRequest(`${integrationName}/${instanceId}/products/import-mappings`, sendData)
