import { localStorageService, KEYS } from '../support/localStorage'
import axiosConfig, { axiosClean, axiosWeb } from '../config/axiosConfig' // Use a version of axios with no error interceptors
import Base64 from '@/support/base64.js'

const baseURL = '/auth'
const isExpireKey = KEYS.IS_EXPIRE_TOKEN
const redirectToKey = KEYS.REDIRECT_TO

// User is logged?
export const userCredentials = () => {
  return localStorageService.getToken() ?? null
}

// Login User against API
export const loginUser = async (loginData) => {
  const url = `${baseURL}/login`
  localStorageService.removeToken()

  try {
    const { data } = await axiosClean.post(url, loginData)
    if (data.hasOwnProperty('access_token')) {
      localStorageService.setToken(data)
      localStorageService.set(isExpireKey, 0)
      localStorageService.remove(redirectToKey)
      return data
    }
  } catch (e) {
    return Promise.reject(e)
  }
}

// Get authenticated user
export const getAuthUser = async () => {
  try {
    const response = await axiosConfig.get(`${baseURL}/profile`)
    return response?.data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Delete the user's access token
export const deleteAccessToken = async () => {
  const url = `${baseURL}/token`

  try {
    await axiosConfig.delete(url)
    localStorageService.removeToken()
    localStorageService.set(isExpireKey, 0)
    localStorageService.remove(redirectToKey)
  } catch (e) {
    return Promise.reject(e)
  }
}

export const logoutUser = async () => {
  try {
    // First, we delete the user's access token
    await deleteAccessToken()

    // Next, we send a request to log out the user.
    await axiosWeb.post('logout')
  } catch (e) {
    return Promise.reject(e)
  } finally {
    setTimeout(() => (redirectToLogin(0, false)), 0)
  }
}

export const redirectToLogin = (isExpire = 1, isSaveRedirectTo = true) => {
  localStorageService.set(isExpireKey, isExpire)
  localStorageService.removeToken()

  localStorageService.set(redirectToKey, isSaveRedirectTo ? top.location.href : '/dashboard')

  if (top.location.pathname !== '/login') {
    top.location.href = '/login'
  }
}

export const prependLogin = (inputUrl) => {
  const hasScheme = url => {
    let split = url.split('/')
    return split.length > 1 && split[1] === ''
  }

  let url = inputUrl
  let response = ''

  if (hasScheme(url)) {
    let parts = url.split('//')
    let schema = parts[0]
    response = schema + '//'
    parts.shift()
    url = parts.join('//')
    let path = url.split('/')
    let server = path[0]
    path.shift()
    response = response + server
    if (path[0] !== 'login') {
      response = response + '/login/'
      url = path.join('/')
      response = response + Base64.encode(url)
    } else {
      response = inputUrl
    }
  } else {
    if (url.startsWith('/')) {
      if (url.startsWith('/login/')) {
        response = url
      } else {
        response = '/login/' + Base64.encode(url)
      }
    } else {
      if (url.startsWith('login/')) {
        response = url
      } else {
        response = 'login/' + Base64.encode(url)
      }
    }
  }

  return response
}

top.prependLogin = prependLogin
