import { mapGetters, mapMutations, mapActions } from 'vuex'
import { createTableView, updateTableView, deleteTableView, setDefaultTableView, getAllTableViews } from '@/services/TableViewsService'
import axios from 'axios'
import store from '@/store/store.js'
import { SKUTableModule } from '@/plugins/SKUTable/store'
import { getParsedQuery, getRulesetFromUrl, addGivenQueryNameToUrl, removeGivenQueryNameFromUrl } from '@/support/paramsBuilderMagic'
import showErrors from '@/support/showErrors'

const CancelToken = axios.CancelToken
let cancelReloadRequest = () => false

export default ({
  useTableViews,
  storeNamespace = 'SKUTableModule',
  permanentlyIncludedValues = [],
  disableSaveAndReadFromUrl = false,
  skipMetatotalsLoading = false,
  isLazyLoading = false
}) => {
  return {
    data () {
      return {
        // table init
        headers: [],
        defaultHeaders: [],
        loading: false,
        // table views
        tableViews: [],
        saveViewLoading: false,
        saveAsNewLoading: false,
        selectedView: null,
        editFavouriteLoading: false,

        getRuleset: null,
        viewId: '',
        included: [],

        namespace: this.instance && this.instance.layerUid ? storeNamespace + '-' + this.instance.layerUid : storeNamespace
      }
    },

    beforeCreate () {
      let instance = this.$options.propsData && this.$options.propsData.instance
      let namespace = instance && instance.layerUid ? storeNamespace + '-' + instance.layerUid : storeNamespace

      this.$options.computed = {
        ...this.$options.computed,
        ...mapGetters(namespace, [
          'getSearchQuery',
          'getPageLimit',
          'getFilteringCriteria',
          'getSortingCriteria',
          'getCurrentPage',
          'getHasCustomDefaultView',
          'getTableViews',
          'getShowArchivedItems',
          'headersIncludingEmptyGroupsAndChildren',
          'getSelectedView',
          'getTableDensity',
          'selectedViewChanged',
          'getIncluded',
          'getTableHeaders',
          'getTableCompact',
        ]),
        localIncluded () {
          let included = this.getIncluded
          // only add permanently included if included has some values because there are cases when we want to send included as empty array
          // to fetch fields whith flag 'default_visible' set to true
          if (permanentlyIncludedValues.length && included.length) {
            permanentlyIncludedValues.forEach(permanentField => {
              let alreadyThere = included.find(field => field === permanentField)
              if (!alreadyThere) {
                included.push(permanentField)
              }
            })
          }
          return included
        }
      }

      this.$options.methods = {
        ...this.$options.methods,
        ...mapMutations(namespace, [
          'SET_SELECTED_ALL_ROWS',
          'SET_SELECTED_RAWS',
          'SET_CURRENT_PAGE',
          'SET_TABLE_VIEWS',
          'SET_FILTERING_CRITERIA',
          'SET_SORTING_CRITERIA',
          'SET_VIEW_RELOAD_IS_BLOCKED',
          'SET_FILTERS_RELOAD_IS_BLOCKED',
          'SET_SORTINGS_RELOAD_IS_BLOCKED',
          'SET_TOTALS_LOADING',
          'SET_SHOW_ARCHIVED_ITEMS',
          'SET_TABLE_DENSITY',
          'SET_SEARCH_QUERY',
          'SET_TABLE_CONFIGURATIONS_LOADING',
        ]),
        ...mapActions(namespace, ['setSelectedView', 'setCurrentPage'])
      }
    },
    created () {
      if (this.namespace && !store.state[this.namespace]) {
        // register unique store
        store.registerModule(this.namespace, SKUTableModule)
      } else if (!store.state.SKUTableModule) {
        // register shared store if not registered yet
        store.registerModule('SKUTableModule', SKUTableModule)
      }
    },
    mounted () {
      if (!isLazyLoading) this.initDataTable()
      this.$confirm.$on('confirm-new-default', this.setDefault)

      this.$drawer.$on('prevPage', this.checkNamespacePrevPage)
      this.$drawer.$on('nextPage', this.checkNamespaceNextPage)
      this.$drawer.$on('reloadData', this.checkNamespacereloadData)
    },
    beforeDestroy () {
      this.$confirm.$off('confirm-new-default', this.setDefault)

      this.$drawer.$off('prevPage', this.checkNamespacePrevPage)
      this.$drawer.$off('nextPage', this.checkNamespaceNextPage)
      this.$drawer.$off('reloadData', this.checkNamespacereloadData)

      let namespace = this.instance && this.instance.layerUid ? storeNamespace + '-' + this.instance.layerUid : storeNamespace
      removeGivenQueryNameFromUrl(namespace)
    },
    computed: {
      getStringTruncateLength () {
        return this.getTableCompact ? 50 : 10000
      },
    },
    methods: {
      addCancellationToken (config) {
        // create cancel token for metaTotals request
        config.cancelToken = new CancelToken(function executor (c) {
          // An executor function receives a cancel function as a parameter
          cancelReloadRequest = c
        })

        return config
      },
      getItemsRequest () {
        console.error('The getItemsRequest method must be provided from the outside component, which is using the tableInitMixin!')
      },
      getMetaRequest () {
        console.error('The getMetaRequest method must be provided from the outside component, which is using the tableInitMixin!')
      },
      // table init
      async initDataTable () {
        /**
         * Table may have table views or have only 1 view
         * Table may have custom default view or initial default view
         * Table view includes values like: filter/sort/density/archive/columns/query
         * Meaning that the user can change those parameters of the table and save to 'view'
         *
         * Table init flow:
         * Check if multiple table views are enabled
         *  if yes:
         *    fetch table views
         *    check if custom default view is set
         *      if yes:
         *        set default view as a selected view
         *          change of selected view triggers table reload, but in this case we pass a flag to cancel this
         *        assign included parameter: specify which columns to fetch from the database
         *  if no:
         *    table columns will be taken from default table_specifications
         *    add flag to only fetch default visible columns
         * parse url to check if particular table view is set
         *  if yes:
         *    set table view (rewrite selected table view)
         *    add view filters/sort/archived parameter to the initial request
         * parse url to check if additional filtering/sorting is required
         *  if yes:
         *    set table filter/query/sort/archived/density (rewrite selected table view values)
         *    add filters/sort/archived parameters to the initial request
         * request initial items and default headers
         * set default headers (a.k.a. columns) from table_specifications
         * again check if custom default view is set
         *  if yes:
         *    table headers (a.k.a columns) were already set alongside with selected table view
         *  if no:
         *    set default headers as table headers
        **/
        try {
          this.loading = true
          if (useTableViews) {
            if (!this.tableViews.length) await this.fetchTableViews()
            this.checkDefaultView()
          }

          // let getRuleset, viewId
          if (!disableSaveAndReadFromUrl) {
            // Get Ruleset when loading
            let rulesetData = getRulesetFromUrl.call(this, this.namespace)

            this.getRuleset = rulesetData.getRuleset
            this.viewId = rulesetData.viewId
          }

          if (this.viewId) {
            const foundView = this.getTableViews.find(v => v.id.toString() === this.viewId)
            if (foundView) {
              // required to skip whole data reload from getSelectedView watcher
              this.SET_VIEW_RELOAD_IS_BLOCKED(true)

              await this.setSelectedView(foundView)
            }
          }

          if (this.getRuleset) {
            this.assignViewValues(this.getRuleset)
          }

          await this.requestInitialItems()
        } catch (e) {
          if (!axios.isCancel(e)) showErrors.call(this, e)
        } finally {
          this.loading = false
        }
      },
      async requestInitialItems () {
        try {
          let requestObj = {
            limit: this.getPageLimit,
            specifications: '2',
            included: JSON.stringify(this.localIncluded),
            visibleOnly: this.getHasCustomDefaultView ? 0 : 1
          }

          // existing edge case:
          // if table has custom default view with specific filter/sort
          // and then the user deletes filter/sort
          // and loads the page with that url
          // the table would still have filter/sort, because it would look into the view filter/sort
          if (this.getRuleset) {
            // set filter/sort from url, but if url value is empty (because it's an initial load of the custom default view)
            // then set the values from the view
            if (this.getRuleset.filters.filterSet.length) {
              requestObj.filters = JSON.stringify(this.getRuleset.filters)
            } else {
              requestObj.filters = JSON.stringify(this.getSelectedView.content.filters)
            }
            if (this.getRuleset.sortObjs.length) {
              requestObj.sortObjs = JSON.stringify(this.getRuleset.sortObjs)
            } else {
              requestObj.sortObjs = JSON.stringify(this.getSelectedView.content.sort)
            }
            if (this.getRuleset.archived || this.getRuleset.archived === 0) requestObj.archived = this.getRuleset.archived
            if (this.getRuleset.query) requestObj.query = this.getRuleset.query
          } else if (this.getHasCustomDefaultView || this.viewId) {
            if (this.getSelectedView.content && this.getSelectedView.content.filters && this.getSelectedView.content.sort) {
              requestObj.filters = JSON.stringify(this.getSelectedView.content.filters)
              requestObj.sortObjs = JSON.stringify(this.getSelectedView.content.sort)
              requestObj.archived = this.getSelectedView.content.showArchivedItems ? 1 : 0
              requestObj.query = this.getSelectedView.content.query || ''
            }
          }

          requestObj = this.addCancellationToken(requestObj)
          await this.getItemsRequest(requestObj)

          this.assignHeaders(this.viewId,this.getTableSpecifications)

          if (!disableSaveAndReadFromUrl) {
            this.assignValuesToUrl()
          }

          this.loading = false

          // fetch meta.total after general data, to make initial request faster
          if (!skipMetatotalsLoading) this.fetchMetaTotals(requestObj)
        } catch (e) {
          if (!axios.isCancel(e)) showErrors.call(this, e)
        } finally {
          this.loading = false
        }
      },
      async checkDefaultView () {
        if (this.getHasCustomDefaultView) {
          let initialView = this.tableViews[0]

          // required to skip whole data reload from getSelectedView watcher
          this.SET_VIEW_RELOAD_IS_BLOCKED(true)

          await this.setSelectedView(initialView)
        }
      },
      assignViewValues (getRuleset) {
        if (getRuleset.filters.filterSet.length) {
          // block watchers reloads
          this.SET_FILTERS_RELOAD_IS_BLOCKED(true)
          this.SET_FILTERING_CRITERIA(getRuleset.filters)
        }
        if (getRuleset.sortObjs.length) {
          // block watchers reloads
          this.SET_SORTINGS_RELOAD_IS_BLOCKED(true)
          this.SET_SORTING_CRITERIA(getRuleset.sortObjs)
        }
        if (getRuleset.archived || getRuleset.archived === '0') this.SET_SHOW_ARCHIVED_ITEMS(getRuleset.archived !== '0')
        if (getRuleset.density) this.SET_TABLE_DENSITY(getRuleset.density)
        if (getRuleset.query) this.SET_SEARCH_QUERY(getRuleset.query)
      },
      assignHeaders (viewId, specifications) {
        // warning: getTableSpecifications must be provided from the outside component, which is using the mixin
        const defaultHeaders = this.prepareHeaders(specifications)

        if (this.getSelectedView?.id === -1) {
          this.$set(this, 'headers', JSON.parse(JSON.stringify(defaultHeaders)))
          this.$set(this, 'defaultHeaders', JSON.parse(JSON.stringify(defaultHeaders)))
        } else if ((useTableViews && this.getHasCustomDefaultView) || (viewId)) {
          // if there's custom view, then header are expected to be set at this point
          this.$set(this, 'defaultHeaders', JSON.parse(JSON.stringify(defaultHeaders)))
        }
      },
      checkNamespacereloadData ({ options, namespace }) {
        if (namespace === this.namespace) {
          this.reloadData(options)
        }
      },
      async loadSpecificationsAndAssignHeaders () {
        try {
          this.SET_TABLE_CONFIGURATIONS_LOADING(true)
          let config = {
            specifications: '1'
          }
          await this.loadTableSpecifications(config)
          this.assignHeaders(this.viewId,this.getTableSpecifications)
        } catch (e) {
          if (!axios.isCancel(e)) showErrors.call(this, e)
        } finally {
          this.SET_TABLE_CONFIGURATIONS_LOADING(false)
        }
      },
      async reloadData ({ query, limit, filters, sortObjs, page, archived, density, isPageChange, isResetPages, visibleOnly, viewId } = {}) {
        // cancel previous request if pending
        if (cancelReloadRequest) cancelReloadRequest('cancel previous table request')
        if (!this.getTableSpecifications || !this.getTableSpecifications.columns) {
          // this if-case is for a special situation when an initial request wasn't finished
          // the initial request can be 'not finished' when user changes the table view right after the
          // page has been loaded. the initial request has more into it comparing to a simple reloadData request
          // that is why we check if !this.getTableSpecifications is still empty - this serves as a flag
          this.getRuleset = null
          if (this.getSelectedView.id === -1) {
            // reset to default view values
            this.viewId = ''
          } else {
            this.viewId = this.getSelectedView.id
          }
          this.requestInitialItems()
        } else {
          this.SET_SELECTED_ALL_ROWS(false)
          this.SET_SELECTED_RAWS({})

          if (isResetPages && this.getCurrentPage !== 1) {
            await this.setCurrentPage(1)
          }
          let config = {}
          try {
            this.loading = true

            config = {
              query: query || this.getSearchQuery,
              limit: limit || this.getPageLimit,
              filters: filters || JSON.stringify(this.getFilteringCriteria),
              sortObjs: sortObjs || JSON.stringify(this.getSortingCriteria),
              page: page || this.getCurrentPage,
              archived: archived || this.getShowArchivedItems ? 1 : 0,
              visibleOnly: visibleOnly || 0,
              included: JSON.stringify(this.localIncluded)
            }

            config = this.addCancellationToken(config)
            await this.getItemsRequest(config)
            cancelReloadRequest = () => false
            if (!disableSaveAndReadFromUrl) {
              this.assignValuesToUrl({ query, limit, filters, sortObjs, archived, density, viewId })
            }

            this.loading = false

            // fetch totals in the end, we ondly want to show the loader for initial request, not for the totals
            if (!isPageChange && !skipMetatotalsLoading) await this.fetchMetaTotals(config)
          } catch (e) {
            if (!axios.isCancel(e)) showErrors.call(this, e)
          } finally {
            this.loading = false
          }
        }
      },
      assignValuesToUrl ({ query, limit, filters, sortObjs, archived, density, viewId } = {}) {
        // Parse Query
        const parsedQuery = getParsedQuery({
          query: query || this.getSearchQuery,
          limit: limit || this.getPageLimit,
          filters: filters || this.getFilteringCriteria,
          sortObjs: sortObjs || this.getSortingCriteria,
          archived: archived || this.getShowArchivedItems ? 1 : 0,
          density: density || this.getTableDensity,
          viewId: viewId || this.getSelectedView.id
        })

        if (parsedQuery && parsedQuery !== this.$route.query[this.namespace]) {
          addGivenQueryNameToUrl(this.namespace, parsedQuery)
        }
      },
      checkNamespacePrevPage ({ namespace, callback }) {
        if (namespace === this.namespace) this.prevPage(callback)
      },
      checkNamespaceNextPage ({ namespace, callback }) {
        if (namespace === this.namespace) this.nextPage(callback)
      },
      async nextPage (callback) {
        await this.reloadData({ page: this.getCurrentPage + 1 })
        this.SET_CURRENT_PAGE(this.getCurrentPage + 1)

        const newItem = this.getItemsList[0]
        if (callback) callback(newItem)
      },

      async prevPage (callback) {
        await this.reloadData({ page: this.getCurrentPage - 1 })
        if (this.getCurrentPage <= 1) return // Don't go to invalid pages
        this.SET_CURRENT_PAGE(this.getCurrentPage - 1)

        const newIndex = this.getPageLimit - 1
        const newItem = this.getItemsList[newIndex]
        if (callback) callback(newItem)
      },
      async fetchMetaTotals (config = {}) {
        config.total = 1
        config.specifications = '0'
        try {
          this.SET_TOTALS_LOADING(true)
          await this.getMetaRequest(config)
        } catch (e) {
          if (!axios.isCancel(e)) showErrors.call(this, e)
        } finally {
          this.SET_TOTALS_LOADING(false)
        }
      },
      // table views
      async fetchTableViews () {
        try {
          this.tableViews = await getAllTableViews({ model: useTableViews })
          return Promise.resolve()
        } catch (e) {
          return Promise.reject(e)
        }
      },
      async saveTableView (view) {
        if (this.saveViewLoading) return
        this.saveViewLoading = true

        try {
          const response = await updateTableView(view.id, { ...view })
          const updatedView = response.data

          this.fetchTableViews()

          this.selectedView = updatedView

          this.$notification.$emit('success', { html: 'Table-view updated successfully!' })
        } catch (e) {
          this.$notification.$emit('danger', { html: 'Something went wrong!' })
        } finally {
          this.saveViewLoading = false
        }
      },
      async saveTableViewAsNew (view) {
        if (this.saveAsNewLoading) return
        this.saveAsNewLoading = true

        try {
          view.model = useTableViews
          view.is_favorite = true
          view.is_default = false

          await createTableView(view)

          this.tableViews = await getAllTableViews({ model: useTableViews })

          let id = this.tableViews[this.tableViews.length - 1].id
          view.id = id

          delete view.value // delete value: 'default' for new table view

          // reset to trigger reactive update if the same view is selected
          this.selectedView = null
          this.$nextTick(() => {
            this.selectedView = view
          })
          // push its id to url
          addGivenQueryNameToUrl(this.namespace, `v:${view.id}`)
        } catch (e) {
          if (!axios.isCancel(e)) showErrors.call(this, e)
        } finally {
          this.saveAsNewLoading = false
        }
      },
      async onViewDeleted ({ id }) {
        try {
          await deleteTableView(id)
          await this.fetchTableViews()
        } catch (e) {
          if (!axios.isCancel(e)) showErrors.call(this, e)
        }
      },
      async setFavourite ({ view, isFavorite }) {
        if (this.editFavouriteLoading) return

        this.editFavouriteLoading = true
        try {
          let clonedView = JSON.parse(JSON.stringify(view))
          clonedView.is_favorite = isFavorite
          await updateTableView(clonedView.id, clonedView)
          this.updateTableViews()
        } catch (e) {
          this.$notification.$emit('danger')
        } finally {
          this.editFavouriteLoading = false
        }
      },
      confirmNewDefault (view) {
        this.$confirm.$emit('open', {
          source: 'new-default',
          message: `
            <div class="p-3 pb-5 rounded-sm bg-orange-200">You have selected <strong>${view.name}</strong> as the default table view.</div>
          `,
          data: view
        })
      },
      async setDefault (view) {
        if (this.loading) return

        try {
          this.loading = true
          await setDefaultTableView(view.id)
          this.updateTableViews()
        } catch (e) {
          this.$notification.$emit('danger')
        } finally {
          this.loading = false
        }
      },
      async unsetDefault (view) {
        if (this.loading) return

        try {
          this.loading = true
          const sendData = {
            name: view.name,
            model: view.model,
            content: view.content,
            is_default: false
          }
          await updateTableView(view.id, sendData)
          this.updateTableViews()
        } catch (e) {
          this.$notification.$emit('danger')
        } finally {
          this.loading = false
        }
      },
      async updateTableViews () {
        try {
          const views = await getAllTableViews({ model: useTableViews })
          this.SET_TABLE_VIEWS(views)

          await this.setSelectedView(this.getTableViews[0])
          return Promise.resolve()
        } catch (e) {
          return Promise.reject(e)
        }
      },
      prepareHeaders (specifications) {
        if (!specifications?.columns) return []

        return specifications?.columns?.map(c => {
          let obj = {
            value: c.data_name,
            name: c.column_label,
            order: c.default_order,
            visible: c.default_visible,
            sortable: c.sortable,
            filterable: c.filterable,
            group: c.group,
            type: c.type,
            groupData: c.group_data,
            isGroupBase: c.data_name === c.group_data, // agreement that data_name === group_data for groupBase column (fat column)
            onlyGrouped: c.only_grouped ? c.only_grouped : false,
            hidden: c.hidden
          }
          if (c.data_id) obj.data_id = c.data_id
          if (c.data_type) obj.data_type = c.data_type
          return obj
        }).sort((a, b) => a.order > b.order ? 1 : -1) || []
      },
    }
  }
}
