import {dateRequestFormat, showAlert} from '@/support/helpers'
import showErrors from '@/support/showErrors'
import axiosConfig from '@/config/axiosConfig'
import {checkIsDateString} from '@/support/timeHelper'

export default {
  computed: {
    customTableFilteringCriteria () {
      return []
    },
    urlParams () {
      return {}
    }
  },
  methods: {
    // Common method to prepare export parameters
    prepareExportParams (resource) {
      const included = this.includedForExport && this.includedForExport.length
        ? this.includedForExport
        : this.getVisibleHeaders.map((header) => header.value)

      const requiredFields = this.getRequiredImportFields ?? []
      requiredFields.forEach(field => {
        if (!included.includes(field)) {
          included.push(field)
        }
      })

      const query = this.getSearchQuery
      const sortObjs = JSON.stringify(this.getSortingCriteria)
      let filters = JSON.parse(JSON.stringify(this.getFilteringCriteria))
      if (this.customTableFilteringCriteria.length) {
        filters.filterSet = [...this.getFilteringCriteria.filterSet, ...this.customTableFilteringCriteria]
      }
      const page = this.getCurrentPage
      const archived = this.getShowArchivedItems ? 1 : 0

      let serializedFilters = this.serializeFilters(filters)

      return {
        format: 'csv',
        filters: serializedFilters,
        sortObjs,
        page,
        query,
        table_specifications: 0,
        archived,
        visible_only: 0,
        included: JSON.stringify(included),
        ...this.urlParams
      }
    },

    // Common method for serializing filters
    serializeFilters (filters) {
      if (filters?.filterSet) {
        return JSON.stringify({
          ...filters,
          filterSet: filters.filterSet.map(item => this.serializeFilterItem(item))
        })
      }
      return JSON.stringify(filters)
    },

    // Helper method for serializing a single filter item
    serializeFilterItem (item) {
      let convertedValue = item.value
      if (item?.value && typeof item.value === 'object') {
        convertedValue = item.value
      } else if (item?.value && checkIsDateString(item.value)) {
        convertedValue = dateRequestFormat(item.value)
      } else {
        convertedValue = encodeURIComponent(item.value)
      }
      return { ...item, value: convertedValue }
    },

    // Common method for handling the export logic
    async handleExportLogic (params, apiUrl, webUrl) {
      try {
        params.total = 1
        const response = await axiosConfig.get(apiUrl, { params })
        const total = response.data.data.total
        params.total = 0

        if (total > 20000) {
          showAlert(this, { type: 'success', html: 'Your file will be emailed to you when ready.' })
          await axiosConfig.get(apiUrl, { params: { ...params, total: 0 } })
        } else {
          const queryString = new URLSearchParams(params).toString()
          const customUrlParams = new URLSearchParams(this.urlParams).toString()
          const baseUrl = window.location.origin
          window.open(`${baseUrl}/${webUrl}?${queryString}&${customUrlParams}`)
        }
      } catch (e) {
        showErrors(e)
      }
    },

    async handleExportForIntegrations (integrationName, integrationInstanceId, resource) {
      if (!this.validateExportItems()) return

      const params = this.prepareExportParams(resource)
      const apiUrl = `${integrationName}/${integrationInstanceId}/${resource}/export`
      const webUrl = `export/${integrationName}/${integrationInstanceId}/${resource}`

      await this.handleExportLogic(params, apiUrl, webUrl)
    },

    async handleExportCSV (resource) {
      if (!this.validateExportItems()) return

      const params = this.prepareExportParams(resource)
      const apiUrl = `/export/${resource}`
      const webUrl = `export/${resource}`
      await this.handleExportLogic(params, apiUrl, webUrl)
    },

    validateExportItems () {
      if (!this.getItemsList.length) {
        showAlert(this, { type: 'danger', html: 'No matching data for export.' })
        return false
      }
      return true
    },
    addGroupChildren (groupName, visible) {
      visible.push(
        ...this.getGroupChildren(groupName).map(field => field.value)
      )
      return visible
    }
  }
}
