import { parseOptionValues } from '../support/attributesHelper'
import { getProductAttributeGroups } from '../services/ProductsService'
import moment from 'moment-timezone'
import { toArray } from 'lodash'

export default {

  methods: {
    prepareAttributePayload (payload) {
      // Only strings can be digital assets
      if (payload.type !== 'string') {
        delete payload.display_options.is_link
      }

      // Only longtexts have raw content
      if (payload.type !== 'longtext') {
        delete payload.display_options.is_html
      }

      if (!payload.display_options.has_options) {
        // Remove display options and values
        delete payload.display_options.allow_custom_value
        delete payload.display_options.type
        delete payload.option_values
      } else {
        // Add option values
        payload.option_values = parseOptionValues(payload.option_values)
      }

      return payload
    },

    isValidationApplicable (type) {
      return ['string', 'longtext', 'numeric', 'integer'].includes(type)
    },

    isTextValidation (type) {
      return ['string', 'longtext'].includes(type)
    },

    isNumberValidation (type) {
      return ['integer', 'numeric'].includes(type)
    },

    async fetchProductAttributes (productId) {
      const {data} = await getProductAttributeGroups(productId)
      data.grouped = toArray(data.groups)
      delete data.groups // Remove groups key

      // Handle date values for the right format
      data.direct.map(attribute => {
        if (attribute.type === 'date' && attribute.value === null) {
          attribute.value = moment(attribute.value).format('YYYY-MM-DD')
        }
        if (attribute.value === null) {
          attribute.value = '' // Necessary for tracking edit changes
        }
        if (!attribute.validation) {
          attribute.validation = {}
        }
        return attribute
      })

      data.direct = this.sortAttributes(data.direct)

      data.grouped.map(grouped => {
        const mapped = grouped.attributes.map(attribute => {
          if (attribute.type === 'date' && attribute.value !== null) {
            attribute.value = moment(attribute.value).format('YYYY-MM-DD')
          }
          return attribute
        })

        return this.sortAttributes(mapped)
      })

      return data
    },

    sortAttributes (attributes) {
      return attributes.sort((a, b) => {
        if ((a.display_options && b.display_options) &&
         a.display_options.sort_order !== b.display_options.sort_order) { return b.display_options.sort_order - a.display_options.sort_order }

        // Fall back to sort on name
        const aName = a.name.toLowerCase()
        const bName = b.name.toLowerCase()

        if (aName < bName) return -1
        else if (aName > bName) return 1
        return 0
      })
    }
  }

}
