import Vue from 'vue'
import App from './App.vue'
import router from './router'
import store from './store/store'
import SKUTable from '@/plugins/SKUTable'
import SKUGlobalSearch from './plugins/SKUGlobalSearch/SKUGlobalSearch'
import SKUGlobalCLI from './plugins/SKUGlobalCLI/SKUGlobalCLI'
import VueClipboard from 'vue-clipboard2'
import vuetify from './plugins/vuetify'
import VueScrollTo from 'vue-scrollto'
import formatDigit from './support/formatDigit'
import Vuelidate from 'vuelidate'
import '@/support/makeDialogsDraggable'
import {
  capitalize,
  extractWarehouseName,
  getImageUrl,
  booleanToString,
  toHumanFormat,
  formatMoney
} from './support/helpers'
import moment from 'moment-timezone'
import VueDraggableResizable from 'vue-draggable-resizable'
import SKULoader from '@/components/globals/SKULoader'
import SKUTag from '@/components/globals/SKUTag'
import VueMask from 'v-mask'
import { toUserTimezone } from './support/timeHelper'
import { localStorageService } from '@/support/localStorage'
import addCommas from '@/support/addCommas'
import '@mdi/font/css/materialdesignicons.css'

// require styles

import '@/assets/css/tailwind.css'
import 'vue2-datepicker/index.css'
import { getSettings, fetchAppSetting } from '@/services/SettingsService'

import '@/assets/include-fontawesome.js'
import LayerManager from '@/support/LayerManager'
import UpdateManager from '@/support/UpdateManager'
// import Echo from 'laravel-echo'
import axiosConfig from '@/config/axiosConfig'

const WSS_DEFAULT_PORT = 6001

// Install Plugins
Vue.use(Vuelidate)
Vue.use(SKUGlobalSearch, store)
Vue.use(SKUGlobalCLI, store)
Vue.use(SKUTable, store)
Vue.use(VueClipboard)
Vue.use(VueMask)
Vue.use(LayerManager, { router })
Vue.use(UpdateManager)

Vue.prototype.$notification = new Vue()
Vue.prototype.$drawer = new Vue()
Vue.prototype.$confirm = new Vue()
Vue.prototype.$update = new Vue()

Vue.prototype.window = window
Vue.prototype.devUrl = 'https://dev.sku.io'
Vue.prototype.stagingUrl = 'https://staging.sku.io'

Vue.prototype.moment = moment

Vue.component('VueDraggableResizable', VueDraggableResizable)
Vue.component('SKULoader', SKULoader)
Vue.component('SKUTag', SKUTag)

// Directives
Vue.use(VueScrollTo, {
  container: 'body',
  duration: 500,
  easing: 'ease',
  offset: 0,
  force: true,
  cancelable: true,
  onStart: false,
  onDone: false,
  onCancel: false,
  x: false,
  y: true
})

Vue.config.productionTip = false
Vue.config.devtools = true

// Prototypes
Vue.prototype.skuLocation = window.location
Vue.prototype.formatDigit = formatDigit
Vue.prototype.addCommas = addCommas
Vue.prototype.getImageUrl = getImageUrl
Vue.prototype.numeral = require('numeral')
Vue.prototype.extractWarehouseName = extractWarehouseName
Vue.prototype.Pusher = require('pusher-js')
// TODO: Disabling due to performance, replace with laravel reverb before turning back on
// Vue.prototype.Echo = new Echo({
//   broadcaster: 'pusher',
//   key: 'local',
//   wsHost: window.location.hostname === 'localhost' ? '127.0.0.1' : window.location.hostname,
//   wsPort: WSS_DEFAULT_PORT,
//   wssPort: WSS_DEFAULT_PORT,
//   forceTLS: true,
//   disableStats: true
// })
Vue.prototype.axios = axiosConfig

// Filters
Vue.filter('capitalize', capitalize)
Vue.filter('toUserTimezone', toUserTimezone)
Vue.filter('booleanToString', booleanToString)
Vue.filter('toHumanFormat', toHumanFormat)
Vue.filter('formatMoney', formatMoney)

const fetchSettings = async () => {
  // set default timezone
  store.commit('SettingsModule/SET_LOADING', true)
  try {
    let { data } = await getSettings()
    if (!data) {
      console.warn('Settings not found.')
      return
    }
    store.commit('SettingsModule/SET_SETTINGS', data)
    const poSettings = data.filter(row => row.key.search('purchase_order_') !== -1)
    store.commit('SettingsPurchaseOrders/SET_SETTINGS', poSettings)
    const timezoneObject = data.find(s => s.key === 'default_timezone')
    if (timezoneObject) {
      const timezone = timezoneObject.value || timezoneObject.default_value
      Vue.prototype.moment.tz.setDefault(timezone)
    }
  } catch (e) {} finally {
    store.commit('SettingsModule/SET_LOADING', false)
  }
}

const setupAppSetting = async () => {
  const { data } = await fetchAppSetting()
  const port = data.websockets.hasOwnProperty('port') ? data.websockets.port : WSS_DEFAULT_PORT
  if (port) Vue.prototype.Echo.options = { ...Vue.prototype.Echo.options, wsPort: port, wssPort: port }
}

const VueInstance = () => {
  new Vue({
    vuetify,
    router,
    store,
    async created () {
      if (!localStorageService.getToken()) return
      fetchSettings()
      // TODO: Disabling for now due to websockets server not being run consistently and performance concerns
      //  Replace with laravel reverb before turning back on
      // setupAppSetting()
    },
    mounted () {
      // Add event listener to be able to open global search
      document.addEventListener('keydown', (e) => {
        const isGlobalSearchOpenable = store.getters['SKUGlobalSearchModule/canGlobalSearch']
        const isGlobalCLIOpenable = store.getters['SKUGlobalCLIModule/canGlobalCLI']
        const escKey = e.code === 'Escape'
        const tagNames = ['INPUT', 'TEXTAREA']
        const isInputFocused = tagNames.includes(document.activeElement.tagName) || document.activeElement.getAttribute('contenteditable')
        if (e.key === '/' && isGlobalSearchOpenable && !isInputFocused) {
          setTimeout(() => this.$openGlobalSearch(), 0)
        } else if (e.key === ':' && isGlobalCLIOpenable && !isInputFocused) {
          setTimeout(() => this.$openGlobalCLI(), 0)
        } else {
          if (escKey) {
            this.$closeGlobalSearch()
            this.$closeGlobalCLI()
          }
        }
      })
    },
    render: h => h(App)
  }).$mount('#app')
}

store.dispatch('UsersModule/requestAuthUser').then((result) => {
  VueInstance()
}).catch((err) => {
  VueInstance()
})
