#!/bin/bash

DEPLOY_NAME=$1

if [[
    -z "$1" ||
    "$1" == "-h" ||
    "$1" == "--help"
  ]]
  then
    echo Run a configured deploy.
    echo
    echo Usage:
    echo        $0 \<deploy_name\>
    echo
    echo Will execute a deploy as configured in the deploy.json
    echo
    echo Example deploy:
    echo
    echo "       \"curvy\": {"
    echo "           \"domain\": \"curvy.sku.io\","
    echo "           \"env\": \"production\","
    echo "           \"host\": \"34.123.222.111\","
    echo "           \"deploy_user\": \"kalvin\","
    echo "           \"web_user\": \"www-data\","
    echo "           \"install_path\": \"/var/www\""
    echo "       },"
    echo
    exit 1
fi



APP_ENV="`php utils/configure ./deploy.json --key=${DEPLOY_NAME}.env`"

if (($?))
  then
    echo "Environment not set."
    echo "Configure it with:"
    echo ""
    echo "    php utils/configure ${DIR}/deploy.json --key=${DEPLOY_NAME}.env --value=<env>"
    echo ""
    exit 2
fi

WWW="`php utils/configure ./deploy.json --key=${DEPLOY_NAME}.install_path`"

if (($?))
  then
    echo "Install path not set.  Using default: /var/www"
    echo "Configure it with:"
    echo ""
    echo "    php utils/configure ${DIR}/deploy.json --key=${DEPLOY_NAME}.install_path --value=<path>"
    echo ""
    WWW="/var/www"
fi

DOMAIN="`php utils/configure ./deploy.json --key=${DEPLOY_NAME}.domain`"

if (($?))
  then
    echo "Domain not set."
    echo "Configure it with:"
    echo ""
    echo "    php utils/configure ${DIR}/deploy.json --key=${DEPLOY_NAME}.domain --value=<domain>"
    echo ""
    exit 3
fi

DEPLOY_HOST="`php utils/configure ./deploy.json --key=${DEPLOY_NAME}.host`"

if (($?))
  then
    echo "Host to intall into not set."
    echo "Configure it with:"
    echo ""
    echo "    php utils/configure ${DIR}/deploy.json --key=${DEPLOY_NAME}.host --value=<host>"
    echo ""
    exit 4
fi

DEPLOY_USER="`php utils/configure ./deploy.json --key=${DEPLOY_NAME}.deploy_user`"

if (($?))
  then
    echo "Deploy user not set.  Using default: sku"
    echo "Configure it with:"
    echo ""
    echo "    php utils/configure ${DIR}/deploy.json --key=${DEPLOY_NAME}.deploy_user --value=<user>"
    echo ""
    DEPLOY_USER="sku"
fi

WEB_USER="`php utils/configure ./deploy.json --key=${DEPLOY_NAME}.web_user`"

if (($?))
  then
    echo "Web user not set.  Using default: www-data"
    echo "Configure it with:"
    echo ""
    echo "    php utils/configure ${DIR}/deploy.json --key=${DEPLOY_NAME}.web_user --value=<user>"
    echo ""
    WEB_USER="sku"
fi

check_status() {
    if [ $? -ne 0 ]; then
        site_up
        fail "Remote deployment failed"
        exit 128
    fi
}

fail() {
    echo "[FATAL]" $1
    exit 1
}

BUILD_NUM="$BITBUCKET_BUILD_NUMBER"

if [ "$BUILD_NUM" == "" ]; then
    BUILD_NUM=$2
fi

if [ "$DEPLOY_USER" == "" ] || [ "$DEPLOY_HOST" == "" ]; then
    fail "No deploy user or host defined in .env.${APP_ENV}"
fi

DEPLOY_ROOT="${WWW}/${DOMAIN}"
DEPLOY_BUILDS="${DEPLOY_ROOT}/builds"
BUILD_FILE="build.${BUILD_NUM}.tgz"
DEPLOY_NEW_BUILD=${DEPLOY_BUILDS}/${BUILD_NUM}

site_up() {
    cat <<-EOF | ssh ${DEPLOY_USER}@${DEPLOY_HOST} 'bash -e -s -x'

    # Make sure site gets UP even if it was not replaced by new version or new artisan does not work:
    rm ${DEPLOY_ROOT}/storage/framework/down
    rm ${DEPLOY_ROOT}/storage/framework/maintenance.php

EOF
}

## Zip project file for upload
tar --exclude='./.git' --exclude='./node_modules' --no-same-owner --no-same-permissions -cvzf "../${BUILD_FILE}" . > /dev/null

## Upload the zip file to the server using scp
scp "../${BUILD_FILE}" ${DEPLOY_USER}@${DEPLOY_HOST}:${DEPLOY_BUILDS}

## ssh into the server
cat <<-EOF | ssh ${DEPLOY_USER}@${DEPLOY_HOST} 'bash -e -s -x'

    # Untar
    mkdir -p ${DEPLOY_NEW_BUILD} || true
    tar -zxf ${DEPLOY_BUILDS}/${BUILD_FILE} -C ${DEPLOY_NEW_BUILD} > /dev/null
    rm -rf ${DEPLOY_BUILDS}/${BUILD_FILE}

    ## Copy .env files
    rsync ${DEPLOY_ROOT}/current/.env ${DEPLOY_NEW_BUILD}

    cd ${DEPLOY_NEW_BUILD}
    sudo chmod 664 ${DEPLOY_NEW_BUILD}/.env.*

    mkdir -p ${DEPLOY_ROOT}/storage/framework/{cache,sessions,views} || true
    mkdir -p ${DEPLOY_ROOT}/storage/app || true
    mkdir -p ${DEPLOY_ROOT}/storage/exports || true
    mkdir -p ${DEPLOY_ROOT}/storage/imports || true

    # Take current site down
    ( php ${DEPLOY_ROOT}/current/artisan down || true )

    # Update frontend files only if it has changed.
    #CURRENT_INDEX_BUILD_FILE=${DEPLOY_ROOT}/current/resources/views/index.blade.php
    #NEW_INDEX_BUILD_FILE=${DEPLOY_NEW_BUILD}/resources/views/index.blade.php

    #STATUS="$(cmp -s $CURRENT_INDEX_BUILD_FILE $NEW_INDEX_BUILD_FILE; echo $?)"

    #if [ "$STATUS" == "1" ]; then
    #  echo "Frontend builds changed, we use new version."
    #else
    #  echo "Frontend builds are identical (or unavailable), we use current version."
    #  rsync ${DEPLOY_ROOT}/current/resources/views/index.blade.php ${DEPLOY_NEW_BUILD}/resources/views
    #  rsync -r ${DEPLOY_ROOT}/current/public/build/ ${DEPLOY_NEW_BUILD}/public/build
    #fi

    # Symlink Storage
    rm -rf ${DEPLOY_NEW_BUILD}/storage || true
    ln -s ${DEPLOY_ROOT}/storage ${DEPLOY_NEW_BUILD}/storage
    ln -snf ${DEPLOY_ROOT}/storage/app/public ${DEPLOY_NEW_BUILD}/public/storage

    echo -n ${BUILD_NUM} > ${DEPLOY_ROOT}/storage/app/build || true
    php -r "echo ' ' . date('Y-m-d-H:i:s');" >> ${DEPLOY_ROOT}/storage/app/build

    echo "Running mysql migrations"
    php ${DEPLOY_NEW_BUILD}/artisan migrate --force -vvv

#    if [ "$APP_ENV" == "production" ]; then
#        rm -rf ${DEPLOY_NEW_BUILD}/public/testcoverage
#    fi

    # php ${DEPLOY_NEW_BUILD}/artisan cache:clear --tags=nd_${APP_ENV}
    # sudo rm -rf ${DEPLOY_ROOT}/storage/framework/cache/*

    php ${DEPLOY_NEW_BUILD}/artisan horizon:terminate || true
    # TODO: Deprecate separate workers for packing slips
    php ${DEPLOY_NEW_BUILD}/artisan sku:packing-slip-handler:restart || true
    # sudo php ${DEPLOY_NEW_BUILD}/artisan websockets:restart || true

    # Add execute permission to jasper
    chmod a+x  vendor/geekcom/phpjasper/bin/jasperstarter/bin/jasperstarter
    # Add execute permission to Ghostscript
    chmod a+x app/Ghostscript/bin/gs*

    # php ${DEPLOY_NEW_BUILD}/artisan cache:clear

    cd ${DEPLOY_NEW_BUILD}
    sudo chown -R ${DEPLOY_USER}:${WEB_USER} storage
    sudo chmod 775 storage

    sudo find ${DEPLOY_NEW_BUILD}/storage -type d -exec chmod 755 {} +
    sudo find ${DEPLOY_NEW_BUILD}/storage -type f -exec chmod 644 {} +

    sudo setfacl -Rdm u:${WEB_USER}:rwx,u:${DEPLOY_USER}:rwx storage/ # Set default for newly created files and folders
    sudo setfacl -Rm  u:${WEB_USER}:rwx,u:${DEPLOY_USER}:rwx storage/ # Apply to current files and folders

    php ${DEPLOY_NEW_BUILD}/artisan vendor:publish --provider="romanzipp\QueueMonitor\Providers\QueueMonitorProvider" || true

    php ${DEPLOY_NEW_BUILD}/artisan telescope:publish || true
    php ${DEPLOY_NEW_BUILD}/artisan telescope:db:create || true

    php ${DEPLOY_NEW_BUILD}/artisan up

    # Symlink Build
    ln -snf ${DEPLOY_NEW_BUILD} ${DEPLOY_BUILDS}/current
    ln -snf ${DEPLOY_BUILDS}/current/public ${DEPLOY_ROOT}/release

    # Deploy devops utilities
    sudo mkdir -p /etc/sku
    sudo cp -R ${DEPLOY_NEW_BUILD}/utils/etc_sku/* /etc/sku
    sudo chown -R root:root /etc/sku
EOF

check_status

echo "Build Succeeded!"

exit 0
