<?php

namespace Database\Seeders\DummyData;

use App\Models\Incoterm;
use App\Models\PaymentTerm;
use App\Models\Product;
use App\Models\ProductBrand;
use App\Models\PurchaseOrder;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\InventoryManagement\InventoryUsageCollection;
use App\Services\InventoryManagement\InventoryUsageDto;
use Database\Factories\FactoryDataRecycler;
use Faker\Generator;
use Illuminate\Container\Container;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Seeder;

class DummyDataSeeder extends Seeder
{
    protected Generator $faker;

    /**
     * Create a new seeder instance.
     *
     * @return void
     *
     * @throws BindingResolutionException
     */
    public function __construct()
    {
        $this->faker = $this->withFaker();
    }

    /**
     * Get a new Faker instance.
     *
     * @throws BindingResolutionException
     */
    protected function withFaker(): Generator
    {
        return Container::getInstance()->make(Generator::class);
    }

    public function run(): void
    {
        // Create direct warehouses
        $warehouses = Warehouse::factory(3)->direct()->create();

        // Create brands
        $brands = ProductBrand::factory(20)->create();

        // Create suppliers
        $suppliers = Supplier::factory(5)->create();

        // Create products, all associated with suppliers
        $products = Product::factory(500)
            ->withProductPricing()
            ->withSupplierProduct(
                (new FactoryDataRecycler([
                    $suppliers,
                ]))
            )
            ->withImages()
            ->recycle($brands)
            ->factoryDataRecycler((new FactoryDataRecycler())->addRecycledData($brands, 'brand_id'))
            ->create();

        // Create purchase orders
        PurchaseOrder::factory(50)
            ->withLines(
                $this->faker->numberBetween(5, 20),
                (new FactoryDataRecycler())->addRecycledData($products, null, true)
            )
            /*
             * This is Laravel's built-in recycle method.  We need to explore the differences in functionality between
             * my FactoryDataRecycler class and Laravel's recycle method.
             */
            //->recycle($suppliers)
            ->factoryDataRecycler(
                (new FactoryDataRecycler())
                    ->addRecycledData($warehouses, 'destination_warehouse_id')
                    ->addRecycledData($suppliers)
                    ->addRecycledData(PaymentTerm::all())
                    ->addRecycledData(Incoterm::all())
            )
            ->receiveAll() // not working
            ->create();

        // Create sales orders
        SalesOrder::factory(250)->withLines(
            $this->faker->numberBetween(1, 10),
            (new FactoryDataRecycler())
                ->addRecycledData($products, null, true)
                ->addRecycledData($warehouses)
        )->noBackorders()->open()
            ->create([
                'sales_channel_id' => SalesChannel::query()->first()->id,
                'order_date' => $this->faker->dateTimeThisYear(),
            ]);

        $inventoryUsageCollection = new InventoryUsageCollection();

        SalesOrderLine::query()->each(function (SalesOrderLine $line) use ($inventoryUsageCollection) {
            $inventoryUsageCollection->add(InventoryUsageDto::from([
                'product' => $line->product,
                'warehouse' => $line->warehouse,
                'quantity' => $line->quantity,
                'event' => $line,
            ]));
        });

        $bulkInventoryManager = new BulkInventoryManager();

        $bulkInventoryManager->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());
    }
}
