<?php

use App\Models\Payment;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{

    private string $insertTriggerName = 'before_insert_payments';
    private string $updateTriggerName = 'before_update_payments';


    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Add hash field with new structure.
        $this->beforeMigration();

        Schema::table('payments', function (Blueprint $table) {
            $table->string('hash', 40)->after('accounting_integration_type')->nullable();
        });

        // handle records
        $this->hashPaymentsData();

        // Add unique index to hash field.
        Schema::table('payments', function(Blueprint $table){
            $table->unique('hash');
        });

        // Add data triggers to automatically add hash
        $this->registerDataTriggers();

    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Remove triggers
        DB::unprepared("DROP TRIGGER IF EXISTS $this->insertTriggerName");
        DB::unprepared("DROP TRIGGER IF EXISTS $this->updateTriggerName");

        // Drop hash column
        if(Schema::hasColumn('payments', 'hash')){
            Schema::dropColumns('payments', 'hash');
        }
    }


    private function beforeMigration(): void{
        if(Schema::hasColumn('payments', 'hash')){
            Schema::dropColumns('payments', 'hash');
        }
    }


    private function prepareFields(string $prefix = ''): string{
        return "COALESCE($prefix".implode(",''), COALESCE($prefix", Payment::getUniqueFields()).",'')";
    }

    private function registerDataTriggers(): void{
        // Creation triggers

        $fields = $this->prepareFields('NEW.');

        DB::unprepared("
            CREATE TRIGGER $this->insertTriggerName
            BEFORE INSERT ON payments
            FOR EACH ROW
            BEGIN
                SET NEW.hash = SHA1(CONCAT_WS('|', $fields));
            END;
        ");

        // Update triggers
        DB::unprepared("
            CREATE TRIGGER $this->updateTriggerName
            BEFORE UPDATE ON payments
            FOR EACH ROW
            BEGIN
                SET NEW.hash = SHA1(CONCAT_WS('|', $fields));
            END;
        ");
    }


    private function hashPaymentsData(): void{
        $fields = $this->prepareFields();
        DB::unprepared("
            UPDATE payments set `hash` = SHA1(CONCAT_WS('|', $fields)) WHERE id > 0 
        ");
    }

};
