<?php

use App\Models\Customer;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{

    private string $insertTriggerName = 'before_insert_customers';
    private string $updateTriggerName = 'before_update_customers';


    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Add hash field with new structure.
        $this->beforeMigration();

        Schema::table('customers', function (Blueprint $table) {
            $table->string('hash', 40)->after('warehouse_id')->nullable();
        });

        // handle records
        $this->hashCustomersData();
        $this->removeDuplicateCustomersByHashes();

        // Add unique index to hash field.
        Schema::table('customers', function(Blueprint $table){
            $table->unique('hash');
        });

        // Add data triggers to automatically add hash
        $this->registerDataTriggers();

    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Remove triggers
        DB::unprepared("DROP TRIGGER IF EXISTS $this->insertTriggerName");
        DB::unprepared("DROP TRIGGER IF EXISTS $this->updateTriggerName");

        // Drop hash column
        if(Schema::hasColumn('customers', 'hash')){
            Schema::dropColumns('customers', 'hash');
        }
    }


    private function beforeMigration(): void{
        if(Schema::hasColumn('customers', 'hash')){
            Schema::dropColumns('customers', 'hash');
        }
    }


    private function prepareFields(string $prefix = ''): string{
        return "COALESCE($prefix".implode(",''), COALESCE($prefix", Customer::getUniqueFields()).",'')";
    }

    private function registerDataTriggers(): void{
        // Creation triggers

        $fields = $this->prepareFields('NEW.');

        DB::unprepared("
            CREATE TRIGGER $this->insertTriggerName
            BEFORE INSERT ON customers
            FOR EACH ROW
            BEGIN
                SET NEW.hash = SHA1(CONCAT_WS('|', $fields));
            END;
        ");

        // Update triggers
        DB::unprepared("
            CREATE TRIGGER $this->updateTriggerName
            BEFORE UPDATE ON customers
            FOR EACH ROW
            BEGIN
                SET NEW.hash = SHA1(CONCAT_WS('|', $fields));
            END;
        ");
    }


    private function hashCustomersData(): void{
        $fields = $this->prepareFields();
        DB::unprepared("
            UPDATE customers set `hash` = SHA1(CONCAT_WS('|', $fields)) WHERE id > 0 
        ");
    }


    private function removeDuplicateCustomersByHashes(): void{
        Artisan::call('patch:delete-duplicate-customers');
    }

};
