<?php

use App\Models\Address;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{

    private string $insertTriggerName = 'before_insert_addresses';
    private string $updateTriggerName = 'before_update_addresses';


    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Add hash field with new structure.
        $this->beforeMigration();

        Schema::table('addresses', function (Blueprint $table) {
            $table->string('hash', 40)->after('label')->nullable();
        });

        // handle records
        $this->hashAddressesData();
        $this->removeDuplicateAddressesByHashes();

        // Add unique index to hash field.
        Schema::table('addresses', function(Blueprint $table){
            $table->unique('hash');
        });

        // Add data triggers to automatically add hash
        $this->registerDataTriggers();

    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Remove triggers
        DB::unprepared("DROP TRIGGER IF EXISTS $this->insertTriggerName");
        DB::unprepared("DROP TRIGGER IF EXISTS $this->updateTriggerName");

        // Drop hash column
        if(Schema::hasColumn('addresses', 'hash')){
            Schema::dropColumns('addresses', 'hash');
        }
    }


    private function beforeMigration(): void{
        if(Schema::hasColumn('addresses', 'hash')){
            Schema::dropColumns('addresses', 'hash');
        }
    }


    private function prepareFields(string $prefix = ''): string{
        return "COALESCE($prefix".implode(",''), COALESCE($prefix", Address::getUniqueFields()).",'')";
    }

    private function registerDataTriggers(): void{
        // Creation triggers

        $fields = $this->prepareFields('NEW.');

        DB::unprepared("
            CREATE TRIGGER $this->insertTriggerName
            BEFORE INSERT ON addresses
            FOR EACH ROW
            BEGIN
                SET NEW.hash = SHA1(CONCAT_WS('|', $fields));
            END;
        ");

        // Update triggers
        DB::unprepared("
            CREATE TRIGGER $this->updateTriggerName
            BEFORE UPDATE ON addresses
            FOR EACH ROW
            BEGIN
                SET NEW.hash = SHA1(CONCAT_WS('|', $fields));
            END;
        ");
    }


    private function hashAddressesData(): void{
        $fields = $this->prepareFields();
        DB::unprepared("
            UPDATE addresses set `hash` = SHA1(CONCAT_WS('|', $fields)) WHERE id > 0 
        ");
    }


    private function removeDuplicateAddressesByHashes(): void{
        Artisan::call('patch:delete-duplicate-addresses');
    }

};
