<?php

namespace Database\Factories;

use App\Exceptions\OversubscribedFifoLayerException;
use App\Models\Customer;
use App\Models\FifoLayer;
use App\Models\Integration;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\SalesChannel;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesCreditReturn;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\StockTakeItem;
use App\Models\Store;
use App\Services\StockTake\OpenStockTakeException;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Illuminate\Support\Facades\Cache;

class SalesOrderFactory extends Factory
{
    /**
     * Define the model's default state.
     */
    public function definition(): array
    {
        return [
            'sales_order_number' => 'SO-'.$this->faker->unique()->numerify('####'),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'order_date' => $this->faker->dateTimeBetween(Carbon::now()->subDays(30), Carbon::today())->format('Y-m-d'),
            'customer_id' => Customer::factory(),
            'store_id' => Store::factory(),
            'is_tax_included' => $this->faker->boolean(),
            'sales_channel_id' => SalesChannel::factory(),
        ];
    }

    /*
     * TODO: HasManyThrough relationship doesn't work for factories in this way
     */
    /*public function fullyCredited(int $countLines = 1)
    {
        return
            $this->hasSalesOrderLines($countLines)->has(
            SalesCredit::factory()
                ->has(
                   SalesCreditLine::factory($countLines)
                   ->state(new Sequence(
                       fn ($sequence) => [
                           'sales_order_line_id' => ($line = SalesOrderLine::query()->offset($sequence->index)->first()),
                           'quantity' => $line->quantity,
                       ]
                   ))
               )
           );
    }*/

    /**
     * Makes a sales order that's fully reserved
     *
     * @throws OpenStockTakeException|OversubscribedFifoLayerException
     */
    public function reserved(int $linesCount = 1): SalesOrder
    {
        /** @var SalesOrder $order */
        $order = $this->hasSalesOrderLines($linesCount)->create([
            'sales_channel_id' => Integration::with(['integrationInstances'])
                ->whereName(Integration::NAME_SKU_IO)
                ->first()
                ->integrationInstances
                ->first()
                ->salesChannel->id,
        ]);

        $order->load('salesOrderLines');
        foreach ($order->salesOrderLines->groupBy('warehouse_id') as $warehouseId => $warehouseLines) {
            /** @var FifoLayer $fifoLayer */
            $fifoLayer = null;

            /**
             * @var int $key
             * @var SalesOrderLine $salesOrderLine
             */
            foreach ($warehouseLines as $key => $salesOrderLine) {
                if ($key == 0) {
                    // Create positive fifo layer with movement
                    $fifoLayer = $this->makeFifoLayerForQuantity(
                        $salesOrderLine->processableQuantity,
                        $warehouseId,
                        $salesOrderLine->product_id
                    );
                }
                /** @var InventoryMovement $activeMovement */
                $activeMovement = InventoryMovement::factory()->create([
                    'quantity' => -$salesOrderLine->processableQuantity,
                    'link_id' => $salesOrderLine->id,
                    'link_type' => $salesOrderLine::class,
                    'type' => InventoryMovement::TYPE_SALE,
                    'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
                    'layer_id' => $fifoLayer->id,
                    'product_id' => $salesOrderLine->product_id,
                    'warehouse_id' => $warehouseId,
                ]);
                $reservationMovement = $activeMovement->replicate(['quantity', 'inventory_status']);
                $reservationMovement->quantity = -$activeMovement->quantity;
                $reservationMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_RESERVED;
                $reservationMovement->layer_id = $activeMovement->layer_id;
                $reservationMovement->save();
            }

            // Mark the fifo layer quantity as used up.
            $fifoLayer->fulfilled_quantity = $fifoLayer->original_quantity;
            $fifoLayer->save();
        }

        return $order->load('salesOrderLines', 'salesOrderLines.inventoryMovements');
    }

    private function makeFifoLayerForQuantity(int $quantity, int $warehouseId, int $productId): FifoLayer
    {
        $fifoLayer = FifoLayer::factory()->create([
            'warehouse_id' => $warehouseId,
            'original_quantity' => abs($quantity),
            'fulfilled_quantity' => 0,
            'product_id' => $productId,
            'link_id' => StockTakeItem::factory(),
            'link_type' => StockTakeItem::class,
        ]);

        InventoryMovement::factory()->create([
            'quantity' => $quantity,
            'link_id' => $fifoLayer->link_id,
            'link_type' => $fifoLayer->link_type,
            'type' => InventoryMovement::TYPE_STOCK_TAKE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'layer_id' => $fifoLayer->id,
            'product_id' => $productId,
            'warehouse_id' => $warehouseId,
        ]);

        return $fifoLayer;
    }

    public function fullyFulfilled(int $countLines = 1)
    {
        return
            $this->hasSalesOrderLines($countLines)->has(
                SalesOrderFulfillment::factory()
                    ->has(
                        SalesOrderFulfillmentLine::factory($countLines)
                            ->state(new Sequence(
                                fn ($sequence) => [
                                    'sales_order_line_id' => ($line = SalesOrderLine::query()->offset($sequence->index)->first())->id,
                                    'quantity' => $line->quantity,
                                ]
                            ))
                    )
            );
    }

    public function createWithSalesCredit(int $numLines = 1): SalesOrder
    {
        /** @var SalesOrder $salesOrder */
        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines($numLines)
            ->create();

        /** @var SalesCredit $salesCredit */
        $salesCredit = SalesCredit::factory()
            ->create([
                'customer_id' => $salesOrder->customer_id,
                'return_status' => SalesCredit::RETURN_STATUS_RETURNED,
                'sales_order_id' => $salesOrder->id,
            ]);

        SalesCreditLine::factory($numLines)
            ->state(new Sequence(
                fn ($sequence) => [
                    'sales_order_line_id' => ($line = $salesOrder->salesOrderLines()->offset($sequence->index)->first())->id,
                    'quantity' => $line->quantity,
                    'amount' => $line->amount,
                    'sales_credit_id' => $salesCredit->id,
                ]
            ))
            ->create();

        return $salesOrder;
    }

    public function createWithFullSalesCreditFullyReturned(int $numLines = 1, ?string $dateOfReturn = null): SalesOrder
    {
        /** @var SalesOrder $salesOrder */
        $salesOrder = SalesOrder::factory()
            ->hasSalesOrderLines($numLines)
            ->create();

        /** @var SalesCredit $salesCredit */
        $salesCredit = SalesCredit::factory()
            ->create([
                'customer_id' => $salesOrder->customer_id,
                'sales_order_id' => $salesOrder->id,
                'return_status' => SalesCredit::RETURN_STATUS_RETURNED,
                'credit_date' => $dateOfReturn,
            ]);

        SalesCreditLine::factory($numLines)
            ->state(new Sequence(
                fn ($sequence) => [
                    'sales_order_line_id' => ($line = $salesOrder->salesOrderLines()->offset($sequence->index)->first())->id,
                    'quantity' => $line->quantity,
                    'amount' => $line->amount,
                    'sales_credit_id' => $salesCredit->id,
                ]
            ))
            ->create();

        SalesCreditReturn::factory()
            ->returned($salesCredit, $numLines, $dateOfReturn);

        return $salesOrder;
    }

    public function withLines(int $numLines = 1, ?FactoryDataRecycler $factoryDataRecycler = null): self
    {
        /** @var SalesOrderLineFactory $salesOrderLineFactory */
        $salesOrderLineFactory = SalesOrderLine::factory($numLines);

        if ($factoryDataRecycler) {
            $salesOrderLineFactory = $salesOrderLineFactory->factoryDataRecycler($factoryDataRecycler);
        }

        return $this->has(
            $salesOrderLineFactory
        );
    }

    /**
     * Create inventory adjustments for each sales order lines so that there is sufficient stock.  If we don't do this
     * and there is no stock available, backorders will be created when using open method.
     */
    public function noBackorders(): self
    {
        return $this->afterCreating(function (SalesOrder $salesOrder) {
            $salesOrder->salesOrderLines->each(function (SalesOrderLine $line) {
                // Check if there is sufficient stock
                if ($line->quantity > $line->product->availableQuantity?->quantity) {
                    /** @var InventoryAdjustment $inventoryAdjustment */
                    InventoryAdjustment::factory()->create([
                        'product_id' => $line->product_id,
                        'quantity' => $line->quantity,
                        'warehouse_id' => $line->warehouse_id,
                        'adjustment_date' => $line->salesOrder->order_date,
                    ]);
                }
            });
        });
    }

    public function open(): self
    {
        return $this->afterCreating(function (SalesOrder $salesOrder) {
            Cache::increment('counter.sales_order_number');
            $salesOrder->order_status = SalesOrder::STATUS_OPEN;
            $salesOrder->save();
        });
    }
}
