<?php

namespace Database\Factories;

use App\Models\Incoterm;
use App\Models\InventoryMovement;
use App\Models\PaymentTerm;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseInvoiceLine;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipment;
use App\Models\PurchaseOrderShipmentLine;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Services\PurchaseOrder\ShipmentManager;
use Carbon\Carbon;
use Database\Factories\Concerns\HasFactoryDataRecycler;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\Sequence;

class PurchaseOrderFactory extends Factory
{
    use HasFactoryDataRecycler;

    /**
     * Define the model's default state.
     */
    public function definition(): array
    {
        return [
            'purchase_order_number' => $this->faker->unique()->numberBetween(10, 10000),
            'purchase_order_date' => $this->faker->date(),
            'currency_code' => 'USD',
            'order_status' => PurchaseOrder::STATUS_OPEN,
            'supplier_id' => Supplier::factory()->withWarehouse(),
            'destination_warehouse_id' => Warehouse::factory(),
            'other_date' => $this->faker->date(),
            'payment_term_id' => PaymentTerm::factory(),
            'incoterm_id' => Incoterm::factory(),
            'supplier_warehouse_id' => null,
            //            'purchase_order_lines'     => PurchaseOrderLine::factory()->raw()
        ];
    }

    public function configure(): self
    {
        return $this->afterCreating(function (PurchaseOrder $purchaseOrder) {
            $purchaseOrder->supplier_warehouse_id = $purchaseOrder->supplier->defaultWarehouse->id;
            $purchaseOrder->save();
        });
    }

    public function receiveAll(): self
    {
        return $this->afterCreating(function (PurchaseOrder $purchaseOrder) {
            (new ShipmentManager())->receiveShipment([
                'received_at' => Carbon::parse($purchaseOrder->purchase_order_date)->addDays($this->faker->numberBetween(5, 30))->toIso8601ZuluString(),
                'receipt_lines' => $purchaseOrder->purchaseOrderLines->map(function (PurchaseOrderLine $line) {
                    return [
                        'quantity' => $line->quantity,
                        'purchase_order_line_id' => $line->id,
                    ];
                })->toArray(),
            ]);
        });
    }

    public function receiveQuantity(int $quantity): self
    {
        return $this->afterCreating(function (PurchaseOrder $purchaseOrder) use ($quantity) {
            (new ShipmentManager())->receiveShipment([
                'received_at' => Carbon::parse($purchaseOrder->purchase_order_date)->addDays($this->faker->numberBetween(5, 30))->toIso8601ZuluString(),
                'receipt_lines' => $purchaseOrder->purchaseOrderLines->map(function (PurchaseOrderLine $line) use ($quantity) {
                    return [
                        'quantity' => $quantity,
                        'purchase_order_line_id' => $line->id,
                    ];
                })->toArray(),
            ]);
        });
    }

    public function fullyInvoiced(int $countLines = 1, $products = null)
    {
        return
            $this->hasPurchaseOrderLines($countLines)->has(
                PurchaseInvoice::factory()
                    ->has(
                        PurchaseInvoiceLine::factory($countLines)
                            ->state(new Sequence(
                                fn ($sequence) => [
                                    'purchase_order_line_id' => ($line = PurchaseOrderLine::query()->offset($sequence->index)->first())->id,
                                    'quantity_invoiced' => $line->quantity,
                                ]
                            ))
                    )
            );
    }

    /*public function fullyReceived(int $countLines = 1)
    {
        return
            $this->hasPurchaseOrderLines($countLines)->has(
                PurchaseOrderShipment::factory()
                    ->has(
                        PurchaseOrderShipmentLine::factory($countLines)
                        ->state(new Sequence(
                            fn ($sequence) => [
                                'purchase_order_line_id' => ($line = PurchaseOrderLine::query()->offset($sequence->index)->first())->id,
                                'quantity' => $line->quantity
                            ]
                        ))
                    )
                    ->has(
                        PurchaseOrderShipmentReceipt::factory()
                            ->has(
                                PurchaseOrderShipmentReceiptLine::factory($countLines)
                                ->state(new Sequence(
                                    fn ($sequence) => [
                                        'purchase_order_shipment_line_id' => ($line = PurchaseOrderShipmentLine::query()->offset($sequence->index)->first())->id,
                                        'quantity' => $line->quantity
                                    ]
                                ))->has(
                                        InventoryMovement::factory()->hasFifoLayers()
                                    )->hasFifoLayers()
                            )
                    )
            );
    }*/

    public function received(int $numLines = 1)
    {
        return
            $this
                ->has(PurchaseOrderLine::factory($numLines)->state(new Sequence(
                    fn ($sequence) => [
                        'quantity' => $randomQuantity = $this->faker->numberBetween(1, 10),
                        'received_quantity' => $randomQuantity,
                    ]
                )))
                ->has(
                    PurchaseOrderShipment::factory()->shipped($numLines)->received($numLines)
                );
    }

    public function withLines(int $numLines = 1, ?FactoryDataRecycler $factoryDataRecycler = null, $quantity = null): self
    {
        /** @var PurchaseOrderLineFactory $purchaseOrderLineFactory */
        $purchaseOrderLineFactory = PurchaseOrderLine::factory($numLines, ['quantity' => $quantity ?? $this->faker->numberBetween(1, 10)]);

        if ($factoryDataRecycler) {
            $purchaseOrderLineFactory = $purchaseOrderLineFactory->factoryDataRecycler($factoryDataRecycler);
        }

        return $this->has(
            $purchaseOrderLineFactory
        );
    }

    /*public function fullyReceivedEasy()
    {
        $purchaseOrder = PurchaseOrder::factory()->create();

        (new ShipmentManager())->receiveShipment();

        return $purchaseOrder;
    }*/
}
