<?php

namespace App;

use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\SalesOrder;
use Validator;

class Validators
{
    /**
     * Validation product that get from CSV file.
     */
    public static function productFromCSV($data): \Illuminate\Contracts\Validation\Validator
    {
        return Validator::make($data, [
            'sku' => 'required|max:255',
            'name' => 'required|max:255',
            'barcode' => 'present|nullable|max:255',
            'brand' => 'present|nullable|max:255',
            'type' => 'required|in:'.implode(',', Product::TYPES),
            'weight' => 'required|numeric',
            'weight-unit' => 'required|in:'.implode(',', Product::WEIGHT_UNITS),
            'length' => 'present|nullable|numeric',
            'width' => 'present|nullable|numeric',
            'height' => 'present|nullable|numeric',
            'dimension-unit' => 'present|nullable|in:'.implode(',', Product::DIMENSION_UNITS),
        ]);
    }

    /**
     * Validation Sales order lines that get from CSV file.
     */
    public static function salesOrderLineFromCSV($data): \Illuminate\Contracts\Validation\Validator
    {
        return Validator::make($data, [
            'sales_order_id' => 'nullable|exists:sales_orders,id',
            'sales_order_number' => 'required|max:255',
            'store_name' => 'nullable|exists:stores,name',
            'shipping_method_id' => 'nullable|exists:shipping_methods,id',
            'shipping_method_name' => 'nullable|exists:shipping_methods,name',
            'order_status' => 'required|in:'.implode(',', SalesOrder::STATUSES),
            'payment_status' => 'nullable',
            'currency_code' => 'required|exists:currencies,code',
            'order_date' => 'required|date',
            'payment_date' => 'nullable|date',
            'ship_by_date' => 'nullable|date',
            'deliver_by_date' => 'nullable|date',
            'fulfilled_at' => 'date|required_if:order_status,'.SalesOrder::STATUS_CLOSED,
            'tracking_number' => 'nullable|max:255',

            // lines

            'order_line_id' => 'nullable|exists:sales_order_lines,id|max:255',
            'order_line_description' => 'required|max:255',
            'order_line_sku' => 'nullable',
            'order_line_quantity' => 'required|numeric|min:1',
            'order_line_amount' => 'required|numeric',
            'order_line_tax' => 'nullable|numeric',
            'order_line_discount' => 'nullable|numeric',
            'order_line_nominal_code' => 'nullable|exists:nominal_codes,code',
            'order_line_warehouse_id' => 'required_with_without:order_line_sku,order_line_warehouse_name|exists:warehouses,id',
            'order_line_warehouse_name' => 'required_with_without:order_line_sku,order_line_warehouse_id|exists:warehouses,name',

            // customer

            'customer_company' => 'nullable|max:255',
            'customer_name' => 'required|max:255',
            'customer_email' => 'nullable|max:255|email',
            'customer_phone' => 'nullable|max:255',
            'customer_fax' => 'nullable|max:255',
            'customer_address1' => 'nullable|max:255',
            'customer_address2' => 'nullable|max:255',
            'customer_address3' => 'nullable|max:255',
            'customer_city' => 'nullable|max:255',
            'customer_province' => 'nullable|max:255',
            'customer_province_code' => 'nullable|max:255',
            'customer_zip' => 'nullable|max:255',
            'customer_country' => 'nullable|max:255',
            'customer_country_code' => 'nullable|exists:constants_countries,code',
            'customer_label' => 'nullable|max:255',

            // shipping address

            'shipping_company' => 'nullable|max:255',
            'shipping_name' => 'nullable|max:255',
            'shipping_email' => 'nullable|max:255|email',
            'shipping_phone' => 'nullable|max:255',
            'shipping_fax' => 'nullable|max:255',
            'shipping_address1' => 'nullable|max:255',
            'shipping_address2' => 'nullable|max:255',
            'shipping_address3' => 'nullable|max:255',
            'shipping_city' => 'nullable|max:255',
            'shipping_province' => 'nullable|max:255',
            'shipping_province_code' => 'nullable|max:255',
            'shipping_zip' => 'nullable|max:255',
            'shipping_county' => 'nullable|max:255',
            'shipping_country_code' => 'nullable|exists:constants_countries,code',
            'shipping_label' => 'nullable|max:255',

            // billing Address

            'billing_company' => 'nullable|max:255',
            'billing_name' => 'nullable|max:255',
            'billing_email' => 'nullable|max:255|email',
            'billing_phone' => 'nullable|max:255',
            'billing_fax' => 'nullable|max:255',
            'billing_address1' => 'nullable|max:255',
            'billing_address2' => 'nullable|max:255',
            'billing_address3' => 'nullable|max:255',
            'billing_city' => 'nullable|max:255',
            'billing_province' => 'nullable|max:255',
            'billing_province_code' => 'nullable|max:255',
            'billing_zip' => 'nullable|max:255',
            'billing_county' => 'nullable|max:255',
            'billing_country_code' => 'nullable|exists:constants_countries,code',
            'billing_label' => 'nullable|max:255',
        ]);
    }

    public static function purchaseOrderLineFromCSV($data)
    {
        return Validator::make($data, [
            'purchase_order_id' => 'nullable|exists:purchase_orders,id',
            'purchase_order_date' => 'required|date',
            'other_date' => 'nullable|date',
            'purchase_order_number' => 'nullable|max:255',
            'submission_format' => 'nullable|in:'.implode(',', PurchaseOrder::SUBMISSION_FORMATS),
            'order_status' => 'nullable|in:'.implode(',', [
                PurchaseOrder::STATUS_DRAFT,
                PurchaseOrder::STATUS_OPEN,
            ]),
            'payment_term_name' => 'nullable|exists:payment_terms,name',
            'incoterm_code' => 'nullable|exists:incoterms,code',
            'shipping_method_id' => 'nullable|exists:shipping_methods,id',
            'shipping_method_name' => 'nullable|exists:shipping_methods,name',
            'supplier_name' => 'required|exists:suppliers,name',
            'supplier_warehouse_id' => 'nullable|exists:warehouses,id',
            'destination_warehouse_id' => 'nullable|exists:warehouses,id',
            'destination_warehouse_name' => 'nullable|exists:warehouses,name',
            'currency_code' => 'required|exists:currencies,code',
            'estimated_delivery_date' => 'nullable|date',
            'supplier_notes' => 'nullable|max:255',

            // lines

            'order_line_id' => 'nullable|exists:purchase_order_lines,id',
            'order_line_description' => 'required|max:255',
            'order_line_sku' => 'required|exists:products,sku',
            'order_line_quantity' => 'required|numeric:min:1',
            'order_line_amount' => 'required|numeric|gt:0',
            'order_line_tax' => 'nullable|numeric',
            'order_line_discount' => 'nullable|numeric',
            'order_line_estimated_delivery_date' => 'nullable|date',
            'order_line_nominal_code' => 'nullable|exists:nominal_codes,code',
        ]);
    }
}
