<?php

namespace App\Uploader;

use App\Helpers;
use Illuminate\Http\UploadedFile;

/**
 * Class Uploader.
 */
class Uploader
{
    /**
     * @throws \Exception
     */
    public function upload(UploadedFile $file, ?string $destination): array
    {
        $destination = $destination ?? 'default';

        $default_destination = [
            'target' => 'app/uploads/',
            'allowed_mime_types' => [],
            'allowed_extensions' => [],
            'max_size' => 7000000,
            'public_access' => false,
        ];

        $destination_info = array_merge(
            $default_destination,
            config('uploader.'.$destination, $default_destination),
        );

        $destination_info['max_size'] = max($default_destination['max_size'], Helpers::returnBytes(
            ini_get('upload_max_filesize')
        ));

        if (! $destination_info['public_access'] && auth()->guest()) {
            abort(403);
        }

        $folder = rtrim(storage_path(rtrim($destination_info['target'], '/')), '/');

        if (! file_exists($folder)) {
            mkdir($folder);
        }
        if (! file_exists($folder)) {
            throw new \Exception("Couldn't create destination folder.");
        }

        if (is_null($file)) {
            throw new \InvalidArgumentException('No file was uploaded');
        }

        try {
            $file->getMimeType();
        } catch (\Exception $e) {
            throw new \Exception('Error processing received file');
        }

        $original_name = $file->getClientOriginalName();
        $original_extension = $file->getClientOriginalExtension();
        $temporary_path = $file->getPathname();

        $original_name_without_extension = pathinfo($original_name, PATHINFO_FILENAME);

        if (! empty($destination_info['allowed_mime_types']) &&
         (
             ! in_array($file->getMimeType(), $destination_info['allowed_mime_types']) ||
             ! in_array($file->getClientMimeType(), $destination_info['allowed_mime_types'])
         )
        ) {
            throw new \Exception("Uploaded file's MIME type is not allowed: {$file->getMimeType()}, {$file->getClientMimeType()}.");
        }

        if (! empty($destination_info['max_size']) &&
         intval($destination_info['max_size']) < $file->getSize()
        ) {
            throw new \Exception('Uploaded file exceeds maximum allowed size of '.intval($destination_info['max_size']).' bytes.');
        }

        $stored = $file->move($folder, date('YmdHis').rand(0, 9999999).'_'.$original_name);

        $stored_name = $stored->getBasename();

        if (! file_exists($stored->getPathname())) {
            throw new \Exception("Couldn't store file");
        }

        return ['original_name' => $original_name, 'stored_name' => $stored_name];
    }

    public function getFilePath(string $storedName, ?string $destination)
    {
        $destination = $destination ?? 'default';

        $default_destination = [
            'target' => 'app/uploads/',
            'allowed_mime_types' => [],
            'allowed_extensions' => [],
            'max_size' => 7000000,
            'public_access' => false,
        ];

        $destination_info = array_merge($default_destination, config('uploader.'.$destination, $default_destination));

        return storage_path("{$destination_info['target']}{$storedName}");
    }
}
