<?php

namespace App\Services\Shopify;

use App\Integrations\Shopify;
use App\Jobs\Shopify\ShopifyGetProductVariantsJob;
use App\Jobs\Shopify\ShopifyDeleteProductListingsRemovedFromShopifyJob;
use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyBulkOperation;
use App\Models\Shopify\ShopifyProduct;
use App\Repositories\Shopify\ShopifyProductRepository;
use Illuminate\Support\Facades\Log;
use PHPShopify\Exception\ApiException;
use PHPShopify\Exception\CurlException;
use Throwable;

class ShopifyProductManager
{
    private ShopifyProductRepository $shopifyProductRepository;

    public function __construct(protected IntegrationInstance $integrationInstance)
    {
        $this->shopifyProductRepository = app(ShopifyProductRepository::class);
    }

    /**
     * @throws ApiException
     * @throws CurlException
     */
    public function getBulkProductVariants(): array
    {
        return (new Shopify($this->integrationInstance))->getBulkProductVariants();
    }

    /**
     * @throws ApiException
     * @throws CurlException
     */
    public function getProductVariants(): array
    {
        $shopify = new Shopify($this->integrationInstance);

        $productVariantGid = 'gid://shopify/ProductVariant/';
        $productVariants = [];
        $cursor = null;
        $currentlyAvailable = 50;
        $retryAttempts = 0;
        $maxRetryAttempts = 5;
        $log = Log::build([
            'driver' => 'single',
            'path' => storage_path('logs/shopify-product-variants.log'),
        ]);

        while (true) {
            try {
                $data = $shopify->getProductVariants($currentlyAvailable, $cursor);

                $edges = $data['data']['productVariants']['edges'];
                $hasNextPage = $data['data']['productVariants']['pageInfo']['hasNextPage'];
                $cursor = $data['data']['productVariants']['pageInfo']['endCursor'];
                $queryCost = $data['extensions']['cost'];
                foreach ($edges as $edge) {
                    $productVariants[] = trim($edge['node']['id'], $productVariantGid);
                    $log->info(trim($edge['node']['id'], $productVariantGid));
                }
                $currentlyAvailable = $queryCost['throttleStatus']['currentlyAvailable'] ?? $currentlyAvailable;
                $delay = $shopify::getGraphqlTimeToRetry($queryCost);
                if ($delay > 0) {
                    sleep($delay + 2);
                }

                if (! $hasNextPage) {
                    break;
                }
            } catch (\Exception $e) {
                if (strpos($e->getMessage(), 'Throttled') !== false && $retryAttempts < $maxRetryAttempts) {
                    $retryAttempts++;
                    sleep(3);
                } else {
                    throw $e;
                }
            }
        }

        return $productVariants;
    }

    /**
     * @throws ApiException
     * @throws CurlException
     */
    public function deleteMissingProductVariants(): void
    {
        set_time_limit(0);
        dispatch(new ShopifyGetProductVariantsJob($this->integrationInstance));
        /*
         * I don't think users want to remove mappings just because a product is removed from shopify.  What if they need
         * to import historical records?  It is better to force user to manually delete listings.
         */
        //dispatch(new ShopifyDeleteProductListingsRemovedFromShopifyJob($this->integrationInstance));
    }

    /**
     * @throws Throwable
     */
    public function getProductVariationIdsFromJSONL(ShopifyBulkOperation $bulkOperation): array
    {
        return $this->shopifyProductRepository->getProductVariationIdsFromJSONL($bulkOperation);
    }

    public function markProductVariantsActiveFromShopify(array $shopifyVariantIds): void
    {
        $this->shopifyProductRepository->markProductVariantsActiveFromShopify($this->integrationInstance, $shopifyVariantIds);
    }

    public function markProductVariantsRemovedThatExistInSkuButMissingFromShopify(array $shopifyVariantIds): void
    {
        $this->shopifyProductRepository->markProductVariantsRemovedThatExistInSkuButMissingFromShopify($this->integrationInstance, $shopifyVariantIds);
    }

    public function deleteProductListingsRemovedFromShopify(): void
    {
        //Log::info('Deleting product listings removed from Shopify for ' . $this->integrationInstance->name);
        $this->shopifyProductRepository->deleteProductListingsRemovedFromShopify($this->integrationInstance);
    }
}
