<?php

namespace App\Services\Shopify\Orders\Actions;

use App\Exceptions\IntegrationInstance\Shopify\LegacyRestockRefundsException;
use App\Exceptions\IntegrationInstance\Shopify\UnsupportedFinancialStatusException;
use App\Exceptions\UnableToFulfillUnopenedOrderException;
use App\Helpers;
use App\Integrations\Shopify;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\Shopify\ShopifyOrder;
use Carbon\Carbon;
use Facades\App\Services\InventoryManagement\Actions\ReleaseBackorderQueues;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Throwable;

class ShopifyDownloadOrder
{
    protected function getOrderId(ShopifyOrder $order)
    {
        return $order->json_object['id'];
    }

    public function refresh(ShopifyOrder $order): ShopifyOrder
    {
        /*
         * TODO: Should we get order transactions for refresh as well?
         */
        $shopify = new Shopify($order->integrationInstance);
        $orderId = $this->getOrderId($order);
        static::saveShopifyOrderToSkuDB(
            $order->integrationInstance->id,
            $shopify->getSalesOrderById($orderId),
            'User',
            $order
        );

        // translate Shopify orders to sku sales orders
        return $this->updateOrder($order);
    }

    /**
     * @throws Throwable
     * @throws UnableToFulfillUnopenedOrderException
     * @throws UnsupportedFinancialStatusException
     * @throws LegacyRestockRefundsException
     */
    protected function updateOrder(ShopifyOrder $order)
    {
        try {
            $order->refresh();
            $order->updateSKUOrder();
            /**
             * We attempt to run release script and update inventory
             * cache for the products in the lines.
             */
            $this->afterUpdate($order);
        } catch (LegacyRestockRefundsException|UnsupportedFinancialStatusException|UnableToFulfillUnopenedOrderException $e) {
            $order->errors = [
                'message' => $e->getMessage(),
                'code' => $e->getCode(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
            ];
            $order->save();
            throw $e;
        }

        return $order;
    }

    protected function afterUpdate(ShopifyOrder $order)
    {
        $productIds = $order->salesOrder->salesOrderLines()->where('is_product', true)
            ->whereNotNull('product_id')
            ->pluck('product_id')
            ->toArray();

        if (empty($productIds)) {
            return;
        }

        // Attempt to process any unprocessed backorder queue releases.
        ReleaseBackorderQueues::release($productIds);

        // Refresh cache
        dispatch(new UpdateProductsInventoryAndAvgCost($productIds))->onQueue('syncInventory');
    }

    public static function saveShopifyOrderToSkuDB(int $integrationInstanceId, array $order, string $downloadedBy, ?ShopifyOrder $shopifyOrder = null)
    {
        // lock saving the order to prevent another process to add the same order (lock to 60 seconds)
        $lock = Cache::driver('redis')->lock("saveShopifyOrderToSkuDB:{$integrationInstanceId}:{$order['order_number']}", 60);
        if (! $lock->get()) {
            return;
        }

        if (! $shopifyOrder) {
            $shopifyOrder = \App\Models\Shopify\ShopifyOrder::with([])->firstOrNew([
                'order_number' => $order['order_number'],
                'integration_instance_id' => $integrationInstanceId,
            ]);
        }

        if ($shopifyOrder->exists) {
            $shopifyOrder->updated_by = $downloadedBy;
        } else {
            $shopifyOrder->downloaded_by = $downloadedBy;
        }

        $shopifyOrder->fill([
            'order_number' => $order['order_number'],
            'integration_instance_id' => $integrationInstanceId,
            'updatedAtUtc' => Carbon::parse($order['updated_at'])->setTimezone('UTC')->toDateTimeString(),
            'createdAtUtc' => Carbon::parse($order['created_at'])->setTimezone('UTC')->toDateTimeString(),
            'purchased_at_utc' => $order['processed_at'] ? Carbon::parse($order['processed_at'])->setTimezone('UTC')->toDateTimeString() : null,
            'json_object' => $order,
        ])->save();

        // release the lock
        $lock->release();

        return $shopifyOrder->id;
    }

    public static function bulkSaveShopifyOrderToSkuDB(int $integrationInstanceId, array $orders, string $downloadedBy)
    {
        /*
         * temp table with data
         */

        $dataFileResource = Helpers::array2csvFile(
            $orders,
            function ($record) {
                return [
                    Carbon::parse($record['updated_at'])->setTimezone('UTC')->toDateTimeString(),
                    Carbon::parse($record['created_at'])->setTimezone('UTC')->toDateTimeString(),
                    Carbon::parse($record['processed_at'])->setTimezone('UTC')->toDateTimeString(),
                    str_replace('\"', '', json_encode($record)),
                ];
            }
        );

        $collate = config('database.connections.mysql.collation');

        $tempShopifyOrdersTable = 'tempShopifyOrders'.Str::random(10);

        $createTemporaryTableQuery = <<<SQL
            CREATE TEMPORARY TABLE IF NOT EXISTS $tempShopifyOrdersTable (
                `shopify_order_id` varchar(200) AS (JSON_UNQUOTE(json_extract(json_object,'$.id'))) STORED,
                `order_number` varchar(200) AS (JSON_UNQUOTE(json_extract(json_object,'$.order_number'))) STORED,
                `updatedAtUtc` datetime,
                `createdAtUtc` datetime,
                `processedAtUtc` datetime,
                `json_object` json,
                UNIQUE INDEX (`shopify_order_id`)
            ) ENGINE=INNODB DEFAULT COLLATE=$collate;
        SQL;

        DB::statement($createTemporaryTableQuery);

        $insertTemporaryTableQuery = "
            LOAD DATA LOCAL INFILE '".stream_get_meta_data($dataFileResource)['uri']."'
                INTO TABLE ".$tempShopifyOrdersTable."
                FIELDS TERMINATED BY ','
                ENCLOSED BY '\"'
                ESCAPED BY ''
                LINES TERMINATED BY '\n'
                (`updatedAtUtc`, `createdAtUtc`, `processedAtUtc`, `json_object`)
        ";

        DB::statement($insertTemporaryTableQuery);

        $jsonOrdersQuery = '
            SELECT so.shopify_order_id, tso.json_object AS updated_order_json,
            so.json_object AS existing_order_json
            FROM shopify_orders AS so
            INNER JOIN '.$tempShopifyOrdersTable.' AS tso
                ON tso.shopify_order_id = so.shopify_order_id
            WHERE so.integration_instance_id = '.$integrationInstanceId.'
        ';

        $jsonOrders = DB::select($jsonOrdersQuery);

        $updated_orders = [];
        foreach ($jsonOrders as $jsonOrder) {
            $updatedOrder = json_decode($jsonOrder->updated_order_json, 1);
            $updated_orders[] = $updatedOrder;
        }
        $dataFileResource = Helpers::array2csvFile(
            $updated_orders,
            function ($record) {
                return [
                    Carbon::parse($record['updated_at'])->setTimezone('UTC')->toDateTimeString(),
                    Carbon::parse($record['created_at'])->setTimezone('UTC')->toDateTimeString(),
                    Carbon::parse($record['processed_at'])->setTimezone('UTC')->toDateTimeString(),
                    json_encode($record),
                ];
            }
        );

        $collate = config('database.connections.mysql.collation');

        $tempShopifyOrdersUpdateTable = 'tempShopifyOrdersUpdate'.Str::random(10);

        $createTemporaryTableQuery = <<<SQL
            CREATE TEMPORARY TABLE IF NOT EXISTS $tempShopifyOrdersUpdateTable (
                `shopify_order_id` varchar(200) AS (JSON_UNQUOTE(json_extract(json_object,'$.id'))) STORED,
                `updatedAtUtc` datetime,
                `createdAtUtc` datetime,
                `processedAtUtc` datetime,
                `json_object` json,
                UNIQUE INDEX (`shopify_order_id`)
            ) ENGINE=INNODB DEFAULT COLLATE=$collate;
        SQL;

        DB::statement($createTemporaryTableQuery);

        $insertTemporaryTableQuery = "
            LOAD DATA LOCAL INFILE '".stream_get_meta_data($dataFileResource)['uri']."'
                INTO TABLE ".$tempShopifyOrdersUpdateTable."
                FIELDS TERMINATED BY ','
                ENCLOSED BY '\"'
                ESCAPED BY ''
                LINES TERMINATED BY '\n'
                (`updatedAtUtc`, `createdAtUtc`, `processedAtUtc`, `json_object`)
        ";

        DB::statement($insertTemporaryTableQuery);

        $updateShopifyQuery = <<<SQL
            UPDATE shopify_orders AS so
            INNER JOIN $tempShopifyOrdersUpdateTable AS tso
            ON tso.shopify_order_id = so.shopify_order_id
            SET so.json_object = tso.json_object,
                so.updatedAtUtc = tso.updatedAtUtc,
                so.createdAtUtc = tso.createdAtUtc,
                so.processedAtUtc = tso.processedAtUtc,
                so.updated_by = '$downloadedBy',
                so.sku_updated_at = NOW()
            WHERE so.integration_instance_id = {$integrationInstanceId}
        SQL;

        DB::statement($updateShopifyQuery);

        $insertShopifyOrdersQuery = <<<SQL
            INSERT INTO shopify_orders (`id`, `integration_instance_id`, `order_number`, `json_object`, `sku_created_at`, `sku_updated_at`, `downloaded_by`, `updatedAtUtc`, `createdAtUtc`, `processedAtUtc`)
            SELECT NULL, $integrationInstanceId, tso.`order_number`, tso.`json_object`, NOW(), NOW(), '$downloadedBy', tso.updatedAtUtc, tso.createdAtUtc, tso.processedAtUtc FROM $tempShopifyOrdersTable AS tso
            LEFT JOIN shopify_orders AS so
            ON so.shopify_order_id = tso.shopify_order_id
            WHERE so.shopify_order_id IS NULL
        SQL;

        DB::statement($insertShopifyOrdersQuery);

        /*
         * TODO: we Should use bulk way here by using Bulk Import trait
         */
        ShopifyOrder::where('integration_instance_id', $integrationInstanceId)
            ->whereIn('shopify_order_id', collect($orders)->pluck('id')->toArray())
            ->each(function (ShopifyOrder $order) {
                $order->createLineItems();
            });
    }
}
