<?php

namespace App\Services\ShippingProvider;

use App;
use App\Integrations\ShipStation;
use App\Models\IntegrationInstance;
use App\Models\ShipStation\ShipstationWarehouse;
use App\Models\SalesOrderFulfillment;
use App\Models\ShippingMethodMappingsSkuToShippingProviderMethod;
use App\Models\ShipStation\ShipstationOrder;
use App\Models\ShipStation\Webhook;
use App\SDKs\ShipStation\Model\ShipStationWebhook;
use App\SDKs\ShipStation\ShipStationException;
use App\Services\SalesOrder\FulfillSalesOrderService;
use Exception;
use Generator;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Str;
use Throwable;

class ShipStationManager
{
    /**
     * @throws Throwable
     * @throws ShipStationException
     * @throws BindingResolutionException
     */
    public function createWebhooks(
        IntegrationInstance $integrationInstance,
        bool $syncFromShipStation = false
    ): void {
        if (! $integrationInstance->isShipStation()) {
            return;
        }

        $route = route('webhooks.shipstation');

        $disallowedAddresses = [
            '127.0.0.1',
            'localhost',
        ];

        if (! Str::contains($route, $disallowedAddresses)) {
            $this->createWebhook($integrationInstance, ShipstationWebhook::TYPE_SHIP_NOTIFY, 'webhooks.shipstation');
            $this->createWebhook($integrationInstance, ShipstationWebhook::TYPE_ORDER_NOTIFY, 'webhooks.shipstation');

            if ($syncFromShipStation) {
                $this->syncWebhooksFromShipStation($integrationInstance);
            }
        }
    }

    /**
     * @throws BindingResolutionException
     * @throws Throwable
     * @throws ShipStationException
     */
    public function createWebhook(
        IntegrationInstance $integrationInstance,
        string $event,
        string $routeName,
    ): void {
        $shipStation = app()->makeWith(ShipStation::class, ['integrationInstance' => $integrationInstance]);
        try {
            // Create the webhook on ShipStation
            $shipstationWebhook = new ShipstationWebhook();
            $shipstationWebhook->event = $event;
            $shipstationWebhook->target_url = route($routeName);
            $shipstationWebhook = $shipStation->createWebhook($shipstationWebhook);

            // Save it to the database
            Webhook::query()->upsert([
                'integration_instance_id' => $integrationInstance->id,
                'json_data' => json_encode($shipstationWebhook->toArray()),
            ], []);
        } catch (Throwable $exception) {
            if (App::environment() == 'testing') {
                throw $exception;
            }
            // Handle the exception
        }
    }

    /**
     * @throws BindingResolutionException
     * @throws ShipStationException
     */
    private function syncWebhooksFromShipStation(IntegrationInstance $integrationInstance): void
    {
        // Remove webhook records of the integration instance
        Webhook::query()->where('integration_instance_id', $integrationInstance->id)->delete();

        // Get webhooks from ShipStation and add them to the shipstation_webhooks table
        $shipStation = app()->make(ShipStation::class, [$integrationInstance]);

        foreach ($shipStation->getWebhooks() as $webhook) {
            $webhook = array_merge($webhook, [
                'id' => $webhook['WebHookID'],
                'target_url' => $webhook['Url'],
                'event' => $webhook['HookType'],
            ]);

            Webhook::query()->upsert([
                'integration_instance_id' => $integrationInstance->id,
                'json_data' => json_encode($webhook),
            ], []);
        }
    }

    /**
     * @throws BindingResolutionException
     * @throws ShipStationException
     */
    public function deleteWebhooks(IntegrationInstance $integrationInstance): void
    {
        if (! $integrationInstance->isShipStation()) {
            return;
        }

        $shipStation = app()->makeWith(ShipStation::class, ['integrationInstance' => $integrationInstance]);

        foreach (Webhook::getWebhooks($integrationInstance->id) as $webhook) {
            if (Str::startsWith($webhook->target_url, config('app.url'))) {
                // Delete ShipStation webhook by ID
                $shipStation->deleteWebhook($webhook->shipstation_webhook_id);
                $webhook->delete();
            }
        }
    }

    /**
     * @throws BindingResolutionException
     * @throws ShipStationException
     * @throws Exception
     */
    public function handleShipStationWebhook(IntegrationInstance $integrationInstance, ShipStationWebhook $webhook): void
    {
        $shipStation = app()->makeWith(ShipStation::class, ['integrationInstance' => $integrationInstance]);

        try {
            $results = $shipStation->getWebhooksResult($webhook);
        } catch (ShipStationException $shipStationException) {
            // 401 Unauthorized
            if ($shipStationException->getCode() == 401) {
                $integrationInstance->unauthorizedConnection();
            }

            throw $shipStationException;
        }

        switch ($webhook->resource_type) {
            case ShipStationWebhook::TYPE_SHIP_NOTIFY:
                $this->handleShipStationShipments($integrationInstance, $results);
                break;
            case ShipStationWebhook::TYPE_ORDER_NOTIFY:
                // Right now we are not taking any action on this webhook
                break;
            default:
                throw new Exception('ShipStation webhook type '.$webhook->resource_type.' handler does not exist.');
        }
    }

    public function handleShipStationShipments(IntegrationInstance $integrationInstance, Generator|array $shipments): array
    {
        $summary = ['success' => 0, 'fails' => 0, 'errors' => []];

        foreach ($shipments as $shipmentResponse) {
            foreach ($shipmentResponse->shipments as $shipment) {
                try {
                    $order = ShipstationOrder::where('orderId', $shipment->orderId)->first();

                    if (! $order) {
                        continue;
                    }

                    $order->shipment = $shipment->toArray() + ['isWebhook' => false];
                    $order->save();

                    $skuFulfilmentId = $order->sku_fulfillment_id;
                    if (! $skuFulfilmentId) {
                        continue;
                    }
                    $salesOrderFulfillment = SalesOrderFulfillment::with([])->find($skuFulfilmentId);
                    if (! $salesOrderFulfillment) {
                        continue;
                    }

                    $shippingProviderMethodMap = ShippingMethodMappingsSkuToShippingProviderMethod::query()->firstOrCreate([
                        'shipping_provider_id' => $integrationInstance->id,
                        'shipping_provider_carrier' => $shipment->carrierCode,
                        'shipping_provider_method' => $shipment->serviceCode, ]);

                    $salesOrderFulfillment->tracking_number = $shipment->trackingNumber;
                    $salesOrderFulfillment->cost = $shipment->shipmentCost + $shipment->insuranceCost;
                    $salesOrderFulfillment->fulfilled_shipping_method_id = $shippingProviderMethodMap->shipping_method_id;
                    $salesOrderFulfillment->fulfilled_shipping_method = $shippingProviderMethodMap->full_name;
                    $salesOrderFulfillment->status = SalesOrderFulfillment::STATUS_FULFILLED;
                    $salesOrderFulfillment->fulfilled_at = $shipment->shipDate ?: $shipment->createDate;
                    $salesOrderFulfillment->save();

                    FulfillSalesOrderService::make($salesOrderFulfillment->salesOrder)->updateFulfillmentStatus($salesOrderFulfillment->fulfilled_at);
                    $summary['success']++;
                } catch (Exception $exception) {
                    $summary['fails']++;
                    $summary['errors'][] = [
                        'shipmentId' => $shipment->shipmentId,
                        'file' => $exception->getFile(),
                        'line' => $exception->getFile(),
                        'message' => $exception->getMessage(),
                    ];
                }
            }
        }

        return $summary;
    }

    public function downloadWarehouses(IntegrationInstance $integrationInstance)
    {
        $shipStation = new ShipStation($integrationInstance);
        $warehouses = $shipStation->getWarehouses();

        ShipstationWarehouse::query()->upsert(
            array_map(function($current){
                return [
                    'shipstation_id' => $current['warehouseId'],
                    'name' => $current['warehouseName'],
                    'is_default' => $current['isDefault'],
                    'json_object' => json_encode($current)
                ];
            }, $warehouses),
            'shipstation_id'
        );
    }
}
