<?php

namespace App\Services\ShippingProvider;

use App\Exceptions\NotIntegratedException;
use App\Integrations\Starshipit;
use App\Jobs\Starshipit\GetShippedOrdersJob;
use App\Jobs\Starshipit\GetTrackingJob;
use App\Models\Amazon\SubmittedFulfillment;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\SalesOrderFulfillment;
use App\Models\ShipStation\ShipstationOrder;
use App\Models\Starshipit\StarshipitOrder;
use App\Models\Warehouse;
use App\SDKs\ShipStation\ShipStationException;
use App\SDKs\Starshipit\StarshipitException;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Log;

class RequestTrackingUpdatesService
{
    const MAX_SYNC_COUNT = 10;

    private ?array $oldestUpdatedDates = null;

    public bool $queued = false;

    public function __construct(protected Request $request)
    {
    }

    /**
     * Request tracking updates from the shipping providers
     *
     * @throws StarshipitException|ShipStationException|NotIntegratedException
     */
    public function execute(): static
    {
        $this->handleStarshipit();
        $this->handleShipstation();
        $this->handleFBA();

        return $this;
    }

    /**
     * Get the oldest updated_at dates of the selected fulfillments
     */
    private function getOldestUpdateDates(): ?array
    {
        if (! isset($this->oldestUpdatedDates)) {
            $this->getBuilder()
                ->groupBy('fulfillment_type')
                ->selectRaw('`fulfillment_type`, GROUP_CONCAT(`id`) as ids')
                ->orderBy('fulfillment_type') // to remove the default orderBy column
                ->each(function ($groupFulfillment) {
                    $ids = array_map('intval', explode(',', $groupFulfillment->ids));

                    $builder = match ($groupFulfillment->fulfillment_type) {
                        SalesOrderFulfillment::TYPE_SHIPSTATION => ShipstationOrder::query(),
                        SalesOrderFulfillment::TYPE_STARSHIPIT => StarshipitOrder::query(),
                        SalesOrderFulfillment::TYPE_FBA => SubmittedFulfillment::query(),
                        default => throw new \InvalidArgumentException("The {$groupFulfillment->fulfillment_type} is not implemented yet."),
                    };

                    $oldestDate = $builder->whereIn('sku_fulfillment_id', $ids)->oldest('updated_at')->value('updated_at');
                    $this->oldestUpdatedDates[$groupFulfillment->fulfillment_type] = $oldestDate?->toISOString();
                });
        }

        return $this->oldestUpdatedDates;
    }

    /**
     * Get the fulfillments query builder from the request
     */
    private function getBuilder(): Builder
    {
        $builder = SalesOrderFulfillment::with([])
            ->whereIn('fulfillment_type', $this->getSupportedTypes())
            ->where('status', '!=', SalesOrderFulfillment::STATUS_FULFILLED);

        if ($this->request->has('ids')) {
            $ids = $this->request->input('ids', []);
            $ids = is_string($ids) ? explode(',', $ids) : $ids;

            $builder->whereIn('id', array_unique($ids));
        } else {
            $builder->filter($this->request)->archived($this->request->input('archived', 0));
        }

        return $builder;
    }

    /**
     * Get supported fulfillment types
     */
    private function getSupportedTypes(): array
    {
        return [SalesOrderFulfillment::TYPE_STARSHIPIT, SalesOrderFulfillment::TYPE_SHIPSTATION, SalesOrderFulfillment::TYPE_FBA];
    }

    /**
     * Get tracking updates from Starshipit based on the oldest date
     *
     * @throws StarshipitException|NotIntegratedException
     */
    private function handleStarshipit(): void
    {
        $oldestDate = $this->getOldestUpdateDates()[SalesOrderFulfillment::TYPE_STARSHIPIT] ?? null;
        if (! $oldestDate) {
            // means the request does not have any Starshipit fulfillment
            return;
        }

        $starshipitIntegrationInstance = IntegrationInstance::starshipit()->first();
        if (! $starshipitIntegrationInstance) {
            throw new NotIntegratedException(Integration::NAME_STARSHIPIT);
        }

        /** @see SKU-4867 */
        if ($this->maxSyncCountExceeded(SalesOrderFulfillment::TYPE_STARSHIPIT)) {
            dispatch(new GetShippedOrdersJob($starshipitIntegrationInstance, ['since_last_updated' => $oldestDate, 'limit' => 250]));
        } else {
            // update them one by one
            $this->getBuilder()
                ->with('starshipitOrder')
                ->where('fulfillment_type', SalesOrderFulfillment::TYPE_STARSHIPIT)
                ->each(function (SalesOrderFulfillment $fulfillment) use ($starshipitIntegrationInstance) {
                    $starshipit = new Starshipit($starshipitIntegrationInstance);
                    $response = $starshipit->getOrder($fulfillment->starshipitOrder->order_id);

                    if (! $response->body['success']) {
                        throw new StarshipitException(json_encode($response->body), $response->statusCode);
                    }

                    $order = $response->body['order'];
                    // update local starshipit order
                    $fulfillment->starshipitOrder->fill($order);
                    $fulfillment->starshipitOrder->save();
                    // check is it manifested
                    if (! $order['manifested']) {
                        return; // continue
                    }

                    $trackingInfo = [
                        'status' => 'Manifested', // at least the status is "Manifested" because we are checked the manifested attribute
                        'carrier_name' => $order['carrier_name'],
                        'carrier_service' => $order['carrier_service_code'], // "code" because the webhook and tracking API return the "code" only
                        'tracking_number' => $order['packages'][0]['tracking_number'] ?? null,
                        'shipment_date' => $order['packages'][0]['label_created_date'],
                        'tracking_status' => 'Delivered',
                        'received_by' => 'RequestTrackingUpdatesService',
                        'price_breakdown' => $order['price_breakdown'] ?? null,
                    ];

                    // update sales order fulfillment
                    GetTrackingJob::addTrackingToSalesOrderFulfillment($fulfillment, $trackingInfo);
                    // add tracking data to the Starshipit order
                    GetTrackingJob::updateStarshipitOrder($fulfillment->starshipitOrder, $trackingInfo, false);
                });
        }
    }

    /**
     * Get tracking updates from Shipstation based on the oldest date
     *
     * @throws ShipStationException|NotIntegratedException
     */
    private function handleShipstation(): void
    {
        $oldestDate = $this->getOldestUpdateDates()[SalesOrderFulfillment::TYPE_SHIPSTATION] ?? null;
        if (! $oldestDate) {
            // means the request does not have any Shipstation fulfillment
            return;
        }

        $shipstationIntegrationInstance = IntegrationInstance::shipstation()->first();
        if (! $shipstationIntegrationInstance) {
            throw new NotIntegratedException(Integration::NAME_SHIPSTATION);
        }

        /** @see SKU-4867 */
        if ($this->maxSyncCountExceeded(SalesOrderFulfillment::TYPE_SHIPSTATION)) {
            $getShipstationShipments = new \App\Jobs\ShipStation\GetShipments($shipstationIntegrationInstance);
            $getShipstationShipments->setRequestOptions(['createDateStart' => $oldestDate, 'pageSize' => 250]);
            dispatch($getShipstationShipments);
        } else {
            // update them one by one
            $this->getBuilder()
                ->with('shipstationOrder')
                ->where('fulfillment_type', SalesOrderFulfillment::TYPE_SHIPSTATION)
                ->each(function (SalesOrderFulfillment $fulfillment) use ($shipstationIntegrationInstance) {
                    $getShipstationShipments = new \App\Jobs\ShipStation\GetShipments($shipstationIntegrationInstance);
                    $getShipstationShipments->setRequestOptions(['orderId' => $fulfillment->shipstationOrder->orderId]);
                    $getShipstationShipments->handle();
                });
        }
    }

    /**
     * Request fulfilled shipments report from Amazon based on the oldest date
     */
    private function handleFBA(): void
    {
        // Request reports by the warehouse
        $this->getBuilder()
            ->where('fulfillment_type', SalesOrderFulfillment::TYPE_FBA)
            ->groupBy('warehouse_id')
            ->selectRaw('`warehouse_id`, GROUP_CONCAT(`id`) as ids')
            ->orderBy('warehouse_id') // to remove the default orderBy column
            ->each(function ($groupFulfillment) {
                /** @var Warehouse $warehouse */
                $warehouse = Warehouse::with(['integrationInstance'])->findOrFail($groupFulfillment->warehouse_id);
                $ids = array_map('intval', explode(',', $groupFulfillment->ids));

                $oldestDate = SubmittedFulfillment::query()->whereIn('sku_fulfillment_id', $ids)->oldest('updated_at')->value('updated_at');
                if ($oldestDate) {
                    Artisan::call('sku:amazon:get-fulfilled-shipments', ['integrationInstance' => $warehouse->integrationInstance, '--start-date' => $oldestDate?->toISOString()]);
                } else {
                    Log::debug('the system has FBA fulfillments without "Submitted Fulfillment" models');
                }
            });
    }

    /**
     * Determined if the fulfillments number exceed the Max Sync Count
     */
    private function maxSyncCountExceeded(string $fulfillmentType): bool
    {
        $maxExceeded = $this->getBuilder()->where('fulfillment_type', $fulfillmentType)->limit(self::MAX_SYNC_COUNT + 1)->count() > self::MAX_SYNC_COUNT;
        if ($maxExceeded) {
            $this->queued = true;
        }

        return $maxExceeded;
    }
}
