<?php

namespace App\Services\SalesOrderFulfillment;

use App\Models\SalesOrderFulfillment;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Jobs\EbayFulfillmentJob;
use Modules\WooCommerce\Entities\WooCommerceIntegrationInstance;
use Modules\WooCommerce\Jobs\WooCommerceFulfillmentJob;

abstract class SubmitTrackingInfo
{
    /**
     * @var SalesOrderFulfillment
     */
    protected $salesOrderFulfillment;

    public function __construct(SalesOrderFulfillment $salesOrderFulfillment)
    {
        $this->salesOrderFulfillment = $salesOrderFulfillment->load('salesOrderFulfillmentLines.salesOrderLine.bundle');
    }

    /**
     * @return static|null
     */
    public static function factory(SalesOrderFulfillment $salesOrderFulfillment)
    {
        $integrationInstance = $salesOrderFulfillment->salesOrder->salesChannel->integrationInstance;

        if (! $integrationInstance->is_automatic_sync_enabled) {
            return null;
        }

        if ($integrationInstance->isShopify()) {
            return new \App\Services\Shopify\ShopifySubmitTrackingInfo($salesOrderFulfillment);
        }

        // TODO: Jatin
        if ($integrationInstance->isAmazonInstance()) {
            customlog('amazon-submit-tracking-info-mfn', 'Running submit tracking info for Amazon');
            return new \App\Services\Amazon\SubmitTrackingInfo($salesOrderFulfillment);
        }

        if ($integrationInstance->isMagento()) {
            return new \App\Services\Magento\SubmitTrackingInfo($salesOrderFulfillment);
        }

        if ($integrationInstance->isEbay()) {
            $ebayIntegrationInstance = EbayIntegrationInstance::find($integrationInstance->id);
            dispatch(new EbayFulfillmentJob($ebayIntegrationInstance, $salesOrderFulfillment));
        }

        if ($integrationInstance->isWooCommerce()) {
            $wooCommerceIntegrationInstance = WooCommerceIntegrationInstance::find($integrationInstance->id);
            dispatch(new WooCommerceFulfillmentJob($wooCommerceIntegrationInstance, $salesOrderFulfillment));
        }

        return null;
    }

    /**
     * Upload tracking info to the sales channel.
     */
    abstract public function submit(): array;
}
