<?php

namespace App\Services\SalesOrder;

use App\Enums\FinancialLineProrationStrategyEnum;
use App\Models\Customer;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Validator;
use Illuminate\Validation\Rule;
use InvalidArgumentException;

class SalesOrderValidator
{
    const SALES_ORDER_ACTION_CREATE = 'create';

    const SALES_ORDER_ACTION_UPDATE = 'update';

    const SALES_ORDER_ACTIONS = [
        self::SALES_ORDER_ACTION_UPDATE,
        self::SALES_ORDER_ACTION_CREATE,
    ];

    private $action = self::SALES_ORDER_ACTION_CREATE;

    /**
     * @var array
     */
    private $data;

    /**
     * @var SalesOrder
     */
    private $salesOrder;

    private function creating(): bool
    {
        return $this->action === self::SALES_ORDER_ACTION_CREATE;
    }

    private function updating(): bool
    {
        return $this->action === self::SALES_ORDER_ACTION_UPDATE;
    }

    public function withAction(string $action, ?SalesOrder $salesOrder = null): self
    {
        if (! in_array($action, self::SALES_ORDER_ACTIONS)) {
            throw new InvalidArgumentException('Unknown action: '.$action);
        }
        if ($action === self::SALES_ORDER_ACTION_UPDATE && ! $salesOrder) {
            throw new InvalidArgumentException('Provide sales order to be updated.');
        }
        $this->action = $action;
        $this->salesOrder = $salesOrder;

        return $this;
    }

    public function rules(): array
    {
        $rules = [
            'sales_order_number' => [
                'nullable',
                'max:255',
                Rule::unique('sales_orders')
                    ->where(function ($query) {
                        $query->where('sales_order_number', request()->input('sales_order_number'));

                        if (request()->input('sales_channel_id')) {
                            $query->where('sales_channel_id', request()->input('sales_channel_id'));
                        }

                        if (request()->route('sales_order')?->id) {
                            $query->where('id', '!=', request()->route('sales_order')?->id);
                        }

                        return $query;
                    }),
            ],
            'store_id' => 'nullable|exists:stores,id',
            'sales_channel_id' => 'nullable|exists:sales_channels,id',
            'store_name' => 'nullable|exists:stores,name',
            'shipping_method_id' => 'nullable|exists:shipping_methods,id',
            'requested_shipping_method' => 'nullable|max:255',
            'order_status' => 'required|in:'.implode(',', SalesOrder::STATUSES),
            'customer_id' => 'sometimes|nullable|exists:customers,id',
            'shipping_address_id' => 'nullable|exists:addresses,id',
            'billing_address_id' => 'nullable|exists:addresses,id',
            'payment_status' => 'nullable',
            'currency_id' => 'required_without:currency_code|exists:currencies,id',
            'currency_code' => 'required_without:currency_id|exists:currencies,code',
            'order_date' => 'required|date',
            'payment_date' => 'nullable|date',
            'ship_by_date' => 'nullable|date',
            'deliver_by_date' => 'nullable|date',
            'packing_slip_printed_at' => 'nullable|date',
            'discount' => 'nullable|numeric|lt:1000000',
            'is_tax_included' => 'nullable|boolean',
            'tax_rate_id' => 'nullable|numeric',
            'tags' => 'nullable|array',

            // Sales order lines

            'sales_order_lines' => 'nullable|array|required_unless:order_status,'.SalesOrder::STATUS_DRAFT,
            'sales_order_lines.*.description' => 'required|max:255',
            'sales_order_lines.*.product_id' => 'nullable|exists:products,id',
            'sales_order_lines.*.sku' => 'nullable|exists:products,sku',
            'sales_order_lines.*.amount' => 'required|numeric|lt:100000000',
            'sales_order_lines.*.quantity' => 'required|numeric|gte:0|lt:100000',
            'sales_order_lines.*.externally_fulfilled_quantity' => 'nullable|numeric|gte:0|lt:100000',
            'sales_order_lines.*.canceled_quantity' => 'required_if:sales_order_lines.*.quantity,0|numeric|lt:100000',
            'sales_order_lines.*.tax_rate_id' => 'nullable|numeric',
            'sales_order_lines.*.sales_channel_line_id' => 'nullable',
            'sales_order_lines.*.nominal_code_id' => 'nullable|exists:nominal_codes,id',
            'sales_order_lines.*.nominal_code_name' => 'nullable|exists:nominal_codes,name',
            'sales_order_lines.*.nominal_code' => 'nullable|exists:nominal_codes,code',
            'sales_order_lines.*.is_product' => 'nullable|boolean',
            'sales_order_lines.*.warehouse_id' => 'nullable|exists:warehouses,id|required_if:sales_order_lines.*.warehouse_routing_method,'.WarehouseRoutingMethod::WAREHOUSE->value,
            'sales_order_lines.*.warehouse_routing_method' => 'nullable|in:'.implode(',', array_map(fn (WarehouseRoutingMethod $case) => $case->value, WarehouseRoutingMethod::cases())),
            'sales_order_lines.*.bundle_id' => 'nullable|exists:products,id',

            // Financial lines

            'financial_lines' => 'nullable|array',
            'financial_lines.*.financial_line_type_id' => 'nullable|exists:financial_line_types,id',
            'financial_lines.*.nominal_code_id' => 'nullable|exists:nominal_codes,id',
            'financial_lines.*.description' => 'nullable|string',
            'financial_lines.*.quantity' => 'nullable|numeric|gte:0|lt:100000',
            'financial_lines.*.amount' => 'nullable|numeric|lt:100000000',
            'financial_lines.*.tax_allocation' => 'nullable|numeric|lt:100000000',
            'financial_lines.*.allocate_to_products' => 'nullable|boolean',
            'financial_lines.*.proration_strategy' => 'nullable|in:'.implode(',', array_map(fn (FinancialLineProrationStrategyEnum $case) => $case->value, FinancialLineProrationStrategyEnum::cases())),
            'financial_lines.*.allocate_to_id' => 'nullable|exists:sales_order_lines,id',
            'financial_lines.*.allocate_to_type' => 'nullable|in:'.implode(',', [SalesOrderLine::class]),

            // Custom field values
            'custom_field_values' => 'nullable|array',

            // Discount Lines
            'discount_lines' => 'nullable|array',
            'discount_lines.*.name' => 'required|max:255',
            'discount_lines.*.rate' => 'nullable|numeric',
            'discount_lines.*.amount' => 'nullable|numeric',

            // Tax Lines
            'tax_lines' => 'nullable|array',
            'tax_lines.*.name' => 'required|max:255',
            'tax_lines.*.rate' => 'nullable|numeric',
        ];

        // I think updating the Sales Order should require specific ids for both addresses and customers.
        if ($this->creating() || $this->updating()) {
            $rules = array_merge($rules, [
                // customer

                'customer' => 'sometimes|nullable|array',
                'customer.company' => 'nullable|max:255',
                'customer.name' => 'nullable|max:255',
                'customer.email' => 'nullable|max:255|email',
                'customer.phone' => 'nullable|max:255',
                'customer.fax' => 'nullable|max:255',
                'customer.address1' => 'nullable|max:255',
                'customer.address2' => 'nullable|max:255',
                'customer.address3' => 'nullable|max:255',
                'customer.city' => 'nullable|max:255',
                'customer.province' => 'nullable|max:255',
                'customer.province_code' => 'nullable|max:255',
                'customer.zip' => 'nullable|max:255',
                'customer.country' => 'nullable|max:255',
                'customer.country_code' => 'nullable|exists:constants_countries,code',
                'customer.label' => 'nullable|max:255',

                // shipping address

                'shipping_address' => 'nullable|array',
                'shipping_address.company' => 'nullable|max:255',
                'shipping_address.name' => 'nullable|max:255',
                'shipping_address.email' => 'nullable|max:255|email',
                'shipping_address.phone' => 'nullable|max:255',
                'shipping_address.fax' => 'nullable|max:255',
                'shipping_address.address1' => 'nullable|max:255',
                'shipping_address.address2' => 'nullable|max:255',
                'shipping_address.address3' => 'nullable|max:255',
                'shipping_address.city' => 'nullable|max:255',
                'shipping_address.province' => 'nullable|max:255',
                'shipping_address.province_code' => 'nullable|max:255',
                'shipping_address.zip' => 'nullable|max:255',
                'shipping_address.county' => 'nullable|max:255',
                'shipping_address.country_code' => 'nullable|exists:constants_countries,code',
                'shipping_address.label' => 'nullable|max:255',

                // billing Address

                'billing_address' => 'nullable|array',
                'billing_address.company' => 'nullable|max:255',
                'billing_address.name' => 'nullable|max:255',
                'billing_address.email' => 'nullable|max:255|email',
                'billing_address.phone' => 'nullable|max:255',
                'billing_address.fax' => 'nullable|max:255',
                'billing_address.address1' => 'nullable|max:255',
                'billing_address.address2' => 'nullable|max:255',
                'billing_address.address3' => 'nullable|max:255',
                'billing_address.city' => 'nullable|max:255',
                'billing_address.province' => 'nullable|max:255',
                'billing_address.province_code' => 'nullable|max:255',
                'billing_address.zip' => 'nullable|max:255',
                'billing_address.county' => 'nullable|max:255',
                'billing_address.country_code' => 'nullable|exists:constants_countries,code',
                'billing_address.label' => 'nullable|max:255',
            ]);
        }

        if ($this->updating()) {
            $rules['order_status'] = 'sometimes|'.$rules['order_status'];
            $rules['currency_id'] = 'sometimes|'.$rules['currency_id'];
            $rules['currency_code'] = 'sometimes|'.$rules['currency_code'];
            $rules['order_date'] = 'sometimes|'.$rules['order_date'];
            $rules['customer_id'] = 'nullable|exists:customers,id';
            $rules['sales_order_lines'] = 'sometimes|'.$rules['sales_order_lines'];
            $rules['sales_order_lines.*.id'] = 'nullable|exists:sales_order_lines,id';
            //            unset($rules['sales_order_lines.*.description']);
            //            unset($rules['sales_order_lines.*.amount']);
            //            unset($rules['sales_order_lines.*.quantity']);
        }

        // add to rules only if "order_status" is closed
        // I added it by this way to prevent return them in validated data if "order_status" is not closed
        if (isset($this->data['order_status']) && $this->data['order_status'] == SalesOrder::STATUS_CLOSED) {
            $rules['fulfilled_at'] = 'date|required_if:order_status,'.SalesOrder::STATUS_CLOSED;
            $rules['tracking_number'] = 'nullable|max:255';
        }

        return $rules;
    }

    public function withValidator(Validator $validator)
    {
        $attributes = $validator->attributes();

        // If an address_id is provided, it must belong to the customer.  If it doesn't, it should fail.
        if (! empty($attributes['shipping_address_id']) || ! empty($attributes['billing_address_id'])) {
            $customer = $this->getCustomer($attributes);

            if ($customer) {
                // check shipping address id is belong to customer
                if (! empty($attributes['shipping_address_id']) && ! $customer->isAddressBelongsToMe($attributes['shipping_address_id'])) {
                    $validator->addFailure('shipping_address_id', 'MustBeBelongToCustomer');
                }

                // check billing address id is belong to customer
                if (! empty($attributes['billing_address_id']) && ! $customer->isAddressBelongsToMe($attributes['billing_address_id'])) {
                    $validator->addFailure('billing_address_id', 'MustBeBelongToCustomer');
                }
            }
        }

        // check if sales order line fulfill
        if ($this->updating()) {
            if (array_key_exists('sales_order_lines', $attributes)) {
                $salesOrder = $this->salesOrder;
                $salesOrder->load('salesOrderLines.salesOrderFulfillmentLines');

                // Sales orders lines can be removed even if fulfilled, the manager takes care of deleting the fulfillment (if it can) or throwing an exception if not
                //                $removedLines = $salesOrder->salesOrderLines->whereNotIn('id', array_column($attributes['sales_order_lines'] ?: [], 'id'));
                //
                //                foreach ($removedLines as $salesOrderLine) {
                //                    if ($salesOrderLine->is_product && $salesOrderLine->fulfilled_quantity) {
                //                        $validator->addFailure("sales_order_lines.{$salesOrderLine->id}.id", 'SalesOrderLineIsFulfilled');
                //                    }
                //                }

                // If sales order line id is provided for an update, it must belong to the given sales order
                if ($this->updating()) {
                    foreach ($attributes['sales_order_lines'] as $salesOrderLine) {
                        if (! empty($salesOrderLine['id']) && ! $this->salesOrder
                            ->salesOrderLines()->where('id', $salesOrderLine['id'])->first()) {
                            $validator->addFailure("sales_order_lines.{$salesOrderLine['id']}.id", 'SalesOrderLineNotOnSalesOrder');
                        }
                    }
                }
            }
        }

        return $validator;
    }

    public function messages()
    {
        return [
            'must_be_belong_to_customer' => 'The :attribute must belong to the customer',
            'sales_order_line_is_fulfilled' => __('messages.sales_order.line_is_fulfilled'),
            'sales_order_line_not_on_sales_order' => __('messages.sales_order.line_not_exists_in_order'),
        ];
    }

    private function getCustomer(array $attributes)
    {
        if (! empty($attributes['customer_id'])) {
            return Customer::with([])->findOrFail($attributes['customer_id']);
        } elseif ($this->creating()) {
            return Customer::exists($attributes['customer'] ?? []);
        } else {
            return $this->salesOrder->customer;
        }
    }
}
