<?php

namespace App\Services\SalesOrder;

use App\Exceptions\InsufficientStockException;
use App\Models\SalesOrderLine;
use App\Models\Warehouse;
use App\Repositories\BackorderQueueRepository;
use App\Repositories\SalesOrder\SalesOrderRepository;
use App\Services\InventoryManagement\InventoryManager;
use Exception;
use InvalidArgumentException;
use Throwable;

class SalesOrderLineInventoryManager
{
    public function __construct(
        private readonly SalesOrderRepository $orders,
        private readonly BackorderQueueRepository $backorderQueues
    ) {
    }

    /**
     * This method synchronises the inventory reservation for the
     * given sales order line. It assumes that any cancellation
     * quantity is already present on the sales order line.
     *
     * @throws InsufficientStockException
     * @throws Throwable
     */
    public function refreshLineReservation(SalesOrderLine $orderLine, ?array $originalLine = null): SalesOrderLine
    {
        customlog('SKU-6240', $orderLine->salesOrder->sales_order_number.' refreshLineReservation for '.$orderLine->id.' SKU '.$orderLine->product?->sku);
        if (! $orderLine->isWarehousedProduct() || $orderLine->is_dropship) {
            return $orderLine;
        }

        // If the warehouse is changing, we first reverse the inventory at the old warehouse.
        if ($this->isChangingLineWarehouse($orderLine, $originalLine)) {

            if ($this->isNewWarehouseIsFba($orderLine)) {
                $orderLine->resetMovements();
                return $orderLine;
            }

            $this->reverseInventoryAtWarehouse($orderLine, $originalLine['warehouse_id']);
        }

        // We reserve inventory for any excess unreserved quantity on the line.
        $totalExistingReservations = $this->orders->getTotalExistingReservationsForLine($orderLine);
        $disparity = $orderLine->processableQuantity - $totalExistingReservations;

        $manager = InventoryManager::with(
            warehouseId: $orderLine->warehouse_id,
            product: $orderLine->product
        );

        // Normally we would accept backorder queues as
        // sales order layers. The only exception is for
        // blemished products. @see SKU-4440
        $fifoOnly = $orderLine->product->isBlemished();

        if ($totalExistingReservations == 0 && $disparity > 0) {
            // We reserve inventory for the first time for the line.
            $manager->takeFromStock(
                quantity: $disparity,
                event: $orderLine,
                fifoOnly: $fifoOnly,
                backorderLink: $orderLine
            );
        } elseif ($disparity > 0) {
            // There is extra quantity to reserve.
            $manager->increaseNegativeEventQty(
                quantity: $disparity,
                event: $orderLine,
                fifoOnly: $fifoOnly,
                backorderLink: $orderLine
            );
        } elseif ($disparity < 0) {
            // The quantity has reduced, we reduce the inventory reservation.
            $manager->decreaseNegativeEventQty(
                quantity: abs($disparity),
                event: $orderLine
            );

            // We decrease backorder queue coverage.
            if ($orderLine->active_backordered_quantity > 0) {
                $this->backorderQueues->reduceCoverageQtyBy($orderLine, abs($disparity));
            }
        }

        return $orderLine;
    }

    /**
     * @throws InsufficientStockException
     * @throws Throwable
     */
    public function reserveInventoryForLine(SalesOrderLine $orderLine, int $quantity): SalesOrderLine
    {
        $totalExistingReservations = $this->orders->getTotalExistingReservationsForLine($orderLine);

        $fifoOnly = $orderLine->product->isBlemished();

        throw_if(
            $orderLine->processableQuantity < $quantity,
            new InvalidArgumentException(
                "Quantity to reserve must not exceed processable quantity. SKU: ({$orderLine->product->sku})"
            )
        );

        $applicableQty = abs(min($orderLine->processableQuantity, abs($quantity)));

        $manager = InventoryManager::with(
            warehouseId: $orderLine->warehouse_id,
            product: $orderLine->product
        );

        if ($totalExistingReservations == 0) {
            $manager->takeFromStock(
                quantity: $applicableQty,
                event: $orderLine,
                fifoOnly: $fifoOnly,
                backorderLink: $orderLine
            );
        } else {
            $manager->increaseNegativeEventQty(
                quantity: $applicableQty,
                event: $orderLine,
                fifoOnly: $fifoOnly,
                backorderLink: $orderLine
            );
        }

        return $orderLine;
    }

    /**
     * @throws Throwable
     */
    public function reverseLineReservation(SalesOrderLine $warehousedLine, int $quantity): SalesOrderLine
    {
        // Quantity to reverse must not exceed total reservations on the line.
        throw_if(
            $this->orders->getTotalExistingReservationsForLine($warehousedLine) < $quantity,
            new InvalidArgumentException(
                "Quantity to reverse must not exceed total reserved quantity. SKU: ({$warehousedLine->product->sku})"
            )
        );

        InventoryManager::with(
            warehouseId: $warehousedLine->warehouse_id,
            product: $warehousedLine->product
        )->decreaseNegativeEventQty(
            quantity: abs($quantity),
            event: $warehousedLine,
        );

        return $warehousedLine;
    }

    private function isChangingLineWarehouse(SalesOrderLine $orderLine, ?array $originalLine): bool
    {
        return $originalLine
            && ! empty($originalLine['warehouse_id'])
            && ! empty($orderLine->warehouse_id)
            && $originalLine['warehouse_id'] != $orderLine->warehouse_id;
    }

    private function isNewWarehouseIsFba(SalesOrderLine $orderLine): bool
    {
        return $orderLine->warehouse->type == Warehouse::TYPE_AMAZON_FBA;
    }

    /**
     * @throws Exception
     */
    private function reverseInventoryAtWarehouse(SalesOrderLine $orderLine, ?int $warehouseId): void
    {
        InventoryManager::with(warehouseId: $warehouseId, product: $orderLine->product)
            ->reverseNegativeEvent($orderLine);
    }
}
