<?php

namespace App\Services\SalesOrder\Fulfillments;

use App\Exceptions\InsufficientStockException;
use App\Exceptions\SalesOrder\SalesOrderFulfillmentDispatchException;
use App\Exceptions\SalesOrder\SalesOrderFulfillmentException;
use App\Exceptions\SalesOrderFulfillmentReservationException;
use App\Models\InventoryAdjustment;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Warehouse;
use App\Notifications\SalesOrderFulfilledNotification;
use App\Repositories\SettingRepository;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\SalesOrder\Fulfillments\Dispatchers\ShipmentDispatcherFactory;
use App\Services\SalesOrder\SalesOrderManager;
use App\Services\SalesOrderFulfillment\SubmitTrackingInfo;
use Carbon\Carbon;
use Carbon\CarbonImmutable;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Managers\AmazonOutboundFulfillmentManager;
use Throwable;

class FulfillmentManager extends SalesOrderManager
{
    /**
     * @param  SalesOrder  $order
     * @param  array  $payload
     * @param  bool  $submitToShippingProvider
     * @param  bool  $submitToSalesChannel
     * @return SalesOrderFulfillment|null
     * @throws Throwable
     * @throws SalesOrderFulfillmentDispatchException
     * @throws BindingResolutionException
     */
    public function fulfill(
        SalesOrder $order,
        array $payload,
        bool $submitToShippingProvider = true,
        bool $submitToSalesChannel = true
    ): ?SalesOrderFulfillment {
        // All products must be mapped & warehoused
        throw_if(
            ! $order->allProductsWarehoused(),
            new SalesOrderFulfillmentException(
                salesOrder: $order,
                message: 'All products must be mapped and have a warehouse.'
            )
        );

        $fulfillment = DB::transaction(
            function () use ($order, $payload, &$submitToShippingProvider, $submitToSalesChannel) {
            // Remove non positive quantity fulfillment lines
            $payload['fulfillment_lines'] = array_filter($payload['fulfillment_lines'], function ($line) {
                return $line['quantity'] > 0;
            });

            if (! isset($payload['fulfilled_at'])) {
                $payload['fulfilled_at'] = Carbon::now();
            }

            // Handle pre inventory start date fulfillment.
            if ($payload['fulfilled_at'] < SettingRepository::getInventoryStartDate()) {
                /*
                 * Reduce quantity of reservation for amount externally fulfilled
                 * Split line as needed (split externally fulfilled into new line)
                 */
                foreach ($payload['fulfillment_lines'] as $fulfillment_line) {
                    /** @var SalesOrderLine $salesOrderLine */
                    $salesOrderLine = SalesOrderLine::query()->findOrFail($fulfillment_line['sales_order_line_id']);
                    $inventoryManager = (new InventoryManager($salesOrderLine->warehouse_id, $salesOrderLine->product));
                    $inventoryManager->decreaseNegativeEventQty($fulfillment_line['quantity'], $salesOrderLine);
                    $salesOrderLine->externally_fulfilled_quantity = $fulfillment_line['quantity'];
                    $salesOrderLine->save();
                    if ($salesOrderLine->quantity == $fulfillment_line['quantity']) {
                        $salesOrderLine->warehouse_id = null;
                        $salesOrderLine->save();
                        $order->updateFulfillmentStatus();
                    } else {
                        $splitLine = $salesOrderLine->splitExternallyFulfilled();
                    }
                }

                return null;
            }

            // Create the fulfillment.
            $fulfillment = $this->orders->createFulfillment($order, $payload);

            // We create a shipment for dropship warehouses
            $warehouse = $this->warehouses->findById($payload['warehouse_id']);

            if ($warehouse->is_dropship) {
                $this->shipDropshipPurchaseOrders($fulfillment, $warehouse, $payload);
                return $fulfillment;
            }

            // Negate the inventory reservations for the fulfillment.
            if ($fulfillment->warehouse->type != Warehouse::TYPE_AMAZON_FBA) {
                $this->negateReservationMovements($fulfillment);
            }

            // Handle edge case of if the Amazon Multi-channel fulfillment already exists
            if ($fulfillment->warehouse->type == Warehouse::TYPE_AMAZON_FBA) {
                $integrationInstance = $fulfillment->warehouse->integrationInstance;
                $amazonIntegrationInstance = AmazonIntegrationInstance::findOrFail($integrationInstance->id);
                if ((new AmazonOutboundFulfillmentManager($amazonIntegrationInstance))->linkSalesOrderFulfillmentToExistingOutboundFulfillment($fulfillment)) {
                    $submitToShippingProvider = false;
                    $fulfillment->refresh();
                }
            }

            return $fulfillment;
        }, 1);

        if(!$fulfillment){
            return null;
        }

        try{
            // We dispatch the fulfillment to the shipping provider
            // for non-manual fulfillments.
            if ($submitToShippingProvider && ! $fulfillment->is_manual) {
                ShipmentDispatcherFactory::make($fulfillment->fulfillment_type)
                    ->dispatchFulfillmentToProvider($fulfillment);
            }
        }catch (Exception $e){
            $fulfillment->delete();
            throw $e;
        }

        // Submit tracking info for manual fulfillments
        if ($submitToSalesChannel && $fulfillment->is_manual && $fulfillment->fulfilled()) {
            SubmitTrackingInfo::factory($fulfillment)?->submit();
        }

        // We update the fulfillment status of the sales order.
        $order->updateFulfillmentStatus();

        if ($fulfillment->salesOrder->salesChannel->emailsCustomers() && $fulfillment->fulfilled()) {
            Notification::send(
                notifiables: [$order->customer],
                notification: new SalesOrderFulfilledNotification($order)
            );
        }


        return $fulfillment;

    }

    /**
     * @throws Throwable
     */
    public function fullyFulfill(
        SalesOrder $order,
        ?CarbonImmutable $fulfillmentDate = null,
        bool $submitToShippingProvider = false,
        bool $submitToSalesChannel = false
    ): void {
        $this->concurrency->allOrNothing(function () use ($order, $fulfillmentDate, $submitToShippingProvider, $submitToSalesChannel) {
            $payloads = $this->buildPayloadsForFullFulfillments($order, $fulfillmentDate);

            foreach ($payloads as $payload) {
                $this->fulfill($order, $payload, $submitToShippingProvider, $submitToSalesChannel);
            }
        });
    }

    /**
     * @throws Throwable
     */
    public function updateFulfillment(SalesOrderFulfillment $fulfillment, $payload): SalesOrderFulfillment
    {
        return $this->concurrency->allOrNothing(function () use ($fulfillment, $payload) {
            // Update the fulfillment with the payload
            $fulfillment->update($payload);

            // We delete the fulfillment lines if requested,
            // or update them if available.
            if ($this->shouldDeleteLines($payload)) {
                $fulfillment = $this->orders->deleteFulfillmentLines($fulfillment);
            } elseif ($this->payloadContainsKey($payload, 'fulfillment_lines')) {
                $fulfillment = $this->updateFulfillmentLines($fulfillment, $payload['fulfillment_lines']);
            }

            // Update the inventory movement dates if fulfillment date changes.
            if ($this->payloadContainsKey($payload, 'fulfilled_at')) {
                $this->orders->updateFulfillmentMovementsDate(
                    fulfillment: $fulfillment,
                    date: CarbonImmutable::parse($fulfillment->fulfilled_at)
                );

                //submit tracking info for manual fulfillments
                if ($fulfillment->is_manual && $fulfillment->fulfilled()) {
                    SubmitTrackingInfo::factory($fulfillment)?->submit();
                }
            }

            // Update fulfillment status of the sales order
            $fulfillment->salesOrder->updateFulfillmentStatus($fulfillment->fulfilled_at);

            return $fulfillment;
        });
    }

    /**
     * @throws InsufficientStockException
     * @throws SalesOrderFulfillmentException|Throwable
     */
    private function updateFulfillmentLines(SalesOrderFulfillment $fulfillment, array $lines): SalesOrderFulfillment
    {
        $retainedFulfillmentLineIds = [];
        collect($lines)
            ->each(/**
             * @throws InsufficientStockException
             * @throws SalesOrderFulfillmentException|Throwable
             */ function (array $line) use ($fulfillment, &$retainedFulfillmentLineIds) {
                $fulfillmentLine = $this->orders->findOrCreateFulfillmentLine($fulfillment, $line);
                // We first clear existing movements for the fulfillment line,
                // so we can re-create the reversal. This is easier than trying
                // to update the movement since we use the same method on the
                // fulfillment line for both create and update.
                $fulfillmentLine->inventoryMovements()->delete();
                $this->negateReservationMovementForFulfillmentLine(
                    fulfillmentLine: $fulfillmentLine,
                    ignoreExistingQty: true // For updates
                );

                $retainedFulfillmentLineIds[] = $fulfillmentLine->id;
            });

        // Trip the fulfillment lines for those not retained.
        $fulfillment->removeLinesNotIn($retainedFulfillmentLineIds);

        // Refresh product inventory
        $this->refreshProductInventoryCache($fulfillment->salesOrder);

        return $fulfillment;
    }

    private function shouldDeleteLines(array $payload): bool
    {
        return $this->payloadContainsKey($payload, 'fulfillment_lines') &&
            empty($payload['fulfillment_lines']);
    }

    /**
     * @throws InsufficientStockException|SalesOrderFulfillmentException|Throwable
     */
    private function negateReservationMovements(SalesOrderFulfillment $fulfillment): void
    {
        $fulfillment->nonDropshipFulfillmentLines->each(/**
         * @throws InsufficientStockException
         * @throws Exception|Throwable
         */ function (SalesOrderFulfillmentLine $fulfillmentLine) {
            $this->negateReservationMovementForFulfillmentLine($fulfillmentLine);
        });

        $this->refreshProductInventoryCache($fulfillment->salesOrder);
    }

    /**
     * @throws InsufficientStockException
     * @throws Exception
     * @throws Throwable
     */
    private function negateReservationMovementForFulfillmentLine(
        SalesOrderFulfillmentLine $fulfillmentLine,
        bool $ignoreExistingQty = false
    ): void {
        if (! $fulfillmentLine->salesOrderLine->is_product || $fulfillmentLine->salesOrderLine->is_dropship) {
            return;
        }

        // If the quantity is greater than the fulfillable quantity, we will
        // create a reservation for the extra quantity. This is to prevent
        // over-fulfillments.
        if($fulfillmentLine->salesOrderLine->fulfillable_quantity > 0){
            $overFulfillment = max(0, $fulfillmentLine->quantity - $fulfillmentLine->salesOrderLine->fulfillable_quantity);
            if($overFulfillment > 0){
                $this->handleExtraFulfillmentQuantity(
                    fulfillmentLine: $fulfillmentLine,
                    quantity: $overFulfillment
                );
            }

        }

        if ($fulfillmentLine->negateReservationMovements(
            quantity: $fulfillmentLine->quantity,
            ignoreExistingQty: $ignoreExistingQty
           )
        ) {
            $fulfillmentLine->salesOrderLine->incrementFulfilledQuantity($fulfillmentLine->quantity);
        } else {
            // We could not negate the reservation movements for a line.
            throw new SalesOrderFulfillmentReservationException(
                salesOrder: $fulfillmentLine->salesOrderFulfillment->salesOrder,
                salesOrderLine: $fulfillmentLine->salesOrderLine,
                fulfillment: $fulfillmentLine->salesOrderFulfillment,
                message: __('messages.sales_order.no_reservation_movements'),
            );
        }
    }

    /**
     * @param  SalesOrderFulfillmentLine  $fulfillmentLine
     * @param  int  $quantity
     * @return void
     * @throws Throwable
     */
    protected function handleExtraFulfillmentQuantity(SalesOrderFulfillmentLine $fulfillmentLine, int $quantity): void
    {
        $salesOrderLine = $fulfillmentLine->salesOrderLine;

        $note = "$quantity of over-fulfillment on fulfillment: {$fulfillmentLine->salesOrderFulfillment->fulfillment_sequence}.";

        /** @var InventoryAdjustment $adjustment */
        $adjustment = InventoryAdjustment::query()->create([
            'adjustment_date' => now(),
            'warehouse_id' => $salesOrderLine->warehouse_id,
            'product_id' => $salesOrderLine->product_id,
            'quantity' => -$quantity,
            'notes' => $note,
            'link_type' => SalesOrderLine::class,
            'link_id' => $salesOrderLine->id,
            'reason' => InventoryAdjustment::ADJUSTMENT_REASON_OVER_FULFILLMENT
        ]);

        $inventoryManager = new InventoryManager($salesOrderLine->warehouse_id, $salesOrderLine->product);
        $inventoryManager->takeFromStock(
            quantity: $quantity,
            event: $adjustment,
            excludedSources: [$salesOrderLine->id]
        );

        $salesOrderLine->salesOrder->notes()->create([
            'note' => $note
        ]);

        // Reduce the sales order fulfillment line quantity.
        $fulfillmentLine->update([
            'quantity' => max(0, $fulfillmentLine->quantity - $quantity)
        ]);
    }

    protected function buildPayloadsForFullFulfillments(
        SalesOrder $order,
        ?CarbonImmutable $fulfillmentDate = null,
        bool $withDropshipLines = false
    ): array
    {
        $payloads = [];

        $order->salesOrderLines->groupBy('warehouse_id')->each(function (Collection $lines) use (&$payloads, &$payload, $fulfillmentDate, $withDropshipLines, $order) {

           $fulfillmentType = $lines->first()->warehouse?->order_fulfillment ?? SalesOrderFulfillment::TYPE_MANUAL;

            if($fulfillmentType === SalesOrderFulfillment::TYPE_MANUAL){
                $shippingMethodIdField = 'fulfilled_shipping_method_id';
                $shippingMethodField = 'fulfilled_shipping_method';
            } else {
                $shippingMethodIdField = 'requested_shipping_method_id';
                $shippingMethodField = 'requested_shipping_method';
            }

            $payload = [
                'fulfilled_at' => $fulfillmentDate ?? CarbonImmutable::now(),
                'fulfillment_type' => $fulfillmentType,
                'fulfillment_lines' => [],
                'warehouse_id' => $lines->first()->warehouse_id,
                'status' => SalesOrderFulfillment::getDefaultStatusByFulfillmentType($fulfillmentType),
                'sales_order_id' => $order->id,
                $shippingMethodIdField => $order->shipping_method_id,
                $shippingMethodField => $order->requested_shipping_method,
            ];
            $lines->each(function (SalesOrderLine $line) use (&$payload, $withDropshipLines, $order, $fulfillmentType) {
                if ($line->is_product && ($withDropshipLines || ! $line->is_dropship)) {

                    $payload['fulfillment_lines'][] = [
                        'sales_order_line_id' => $line->id,
                        'quantity' => $line->quantity,
                        'is_dropship' => $line->is_dropship,
                        'fulfillable_quantity' => max(0, $line->unfulfilled_quantity - $line->active_backordered_quantity),
                    ];
                }
            });
            $payloads[] = $payload;
        });

        return $payloads;
    }
}
