<?php

namespace App\Services\SalesOrder\Fulfillments\Dispatchers;

use App\Exceptions\SalesOrder\SalesOrderFulfillmentException;
use App\Jobs\Starshipit\SubmitOrders;
use App\Models\SalesOrderFulfillment;
use App\Services\SalesOrder\Fulfillments\ShipmentDispatcher;
use Throwable;

class StarshipitDispatcher implements ShipmentDispatcher
{
    /**
     * @throws SalesOrderFulfillmentException
     * @throws Throwable
     */
    public function dispatchFulfillmentToProvider(SalesOrderFulfillment $fulfillment)
    {
        $validationErrors = $fulfillment->toStarshipitOrder()->validate();
        if (!empty($validationErrors))
        {
            customlog('starshipit', 'dispatchFulfillmentToProvider:validationErrors', [
                'validationErrors' => $validationErrors
            ]);
        }
        throw_if(! empty($validationErrors), new SalesOrderFulfillmentException(
            salesOrder: $fulfillment->salesOrder,
            fulfillment: $fulfillment,
            message: __('messages.integration_instance.can_not_submit_order', ['resource' => 'Starshipit'])
        ));

        $submitResponse = SubmitOrders::submitOrderFulfillmentToStarshipit($fulfillment);

        if (! $submitResponse['success']) {

            customlog('starshipit', 'dispatchFulfillmentToProvider:submitResponse', [
                'submitResponse' => $submitResponse
            ]);

            if ($submitResponse['error'] === 'archived') {
                // We attempt to resubmit with next fulfillment sequence.
                $fulfillment->update(['fulfillment_sequence' => $fulfillment->fulfillment_sequence + 1]);
                $this->dispatchFulfillmentToProvider($fulfillment);

                return;
            }

            $message = $submitResponse['error'] === 'unfulfillable' ?
                __('messages.integration_instance.unfulfillable_order', ['resource' => 'Starshipit']) :
                __('messages.integration_instance.can_not_submit_order', ['resource' => 'Starshipit']);

            throw new SalesOrderFulfillmentException(
                salesOrder: $fulfillment->salesOrder,
                message: $message
            );
        }

        if (isset($submitResponse['linked']) && $submitResponse['linked']) {
            $fulfillment->ssi_imported = true;
        }
    }
}
