<?php

namespace App\Services\SalesOrder\Fulfillments\Dispatchers;

use App\Exceptions\SalesOrder\SalesOrderFulfillmentException;
use App\Integrations\ShipStation;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\ShipStation\ShipstationOrder;
use App\SDKs\ShipStation\ShipStationException;
use App\Services\SalesOrder\Fulfillments\ShipmentDispatcher;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Facades\DB;
use Throwable;

class ShipstationDispatcher implements ShipmentDispatcher
{
    /**
     * @throws Throwable
     */
    public function dispatchFulfillmentToProvider(SalesOrderFulfillment $fulfillment)
    {
        $validationErrors = $fulfillment->toShipStationOrder()->validate();

        if (!empty($validationErrors)) {
            DB::rollBack();
            throw new SalesOrderFulfillmentException(
                salesOrder: $fulfillment->salesOrder,
                fulfillment: $fulfillment,
                message: 'Fulfillment is not valid for submission',
                errors: $validationErrors
            );
        }

        try {
            $this->submitToShipstation($fulfillment);
        } catch (ShipStationException $shipStationException) {
            DB::rollBack();

            // 401 Unauthorized
            if ($shipStationException->getCode() == 401) {
                IntegrationInstance::shipstation()->first()->unauthorizedConnection();
            }

            if ($shipStationException->getCode() == 404) {
                $salesOrder = $fulfillment->salesOrder;
                $salesOrder->syncTags(['shipstation_error']);
                $salesOrder->notes()->create([
                    'note' => 'Submission to Shipstation failed due to a problem with Shipstation.  Payload sent: '.json_encode($fulfillment->toShipStationOrder()->toArray(), JSON_PRETTY_PRINT),
                ]);
                $salesOrder->save();
            }

            throw (new SalesOrderFulfillmentException(
                salesOrder : $fulfillment->salesOrder,
                fulfillment: $fulfillment,
                message    : __('messages.integration_instance.can_not_submit_order',
                    ['resource' => 'ShipStation']).' - '.$shipStationException->getMessage(),
                code       : $shipStationException->getCode()
            ));
        }
    }

    /**
     * @throws ShipStationException
     * @throws BindingResolutionException
     */
    public function submitToShipstation(SalesOrderFulfillment $fulfillment)
    {
        /** @var IntegrationInstance $integrationInstance */
        $integrationInstance = IntegrationInstance::with([])->whereHas('integration', function ($query) {
            $query->where('name', Integration::NAME_SHIPSTATION);
        })->first();

        $shipStation = app()->makeWith(ShipStation::class, ['integrationInstance' => $integrationInstance]);

        customlog('SKU-6191', 'Submitting fulfillment '.$fulfillment->fulfillment_number.' for '.$fulfillment->salesOrder->sales_order_number.' to ShipStation', $fulfillment->toArray(), 7);
        $response = $shipStation->submitOrder($fulfillment->toShipStationOrder());

        $result = $response->getResultAttribute();
        $result->sku_fulfillment_id = $fulfillment->id;

        $fulfillment->salesOrder()->update(['fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_AWAITING_TRACKING]);
        ShipstationOrder::query()->where('orderId', $result->orderId)->firstOr(function () use ($result, $fulfillment) {
            ShipstationOrder::query()->create([
                'sku_fulfillment_id' => $fulfillment->id,
                'json_data' => $result->toArray(),
            ]);
        });
    }
}
