<?php

namespace App\Services\SalesOrder\Fulfillments;

use App\Data\BulkActionData;
use App\Data\BulkActionResultData;
use App\Data\SalesOrderFulfillmentData;
use App\Jobs\SubmitFulfillmentToThirdPartiesJob;
use App\Models\InventoryMovement;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Services\SalesOrder\Fulfillments\Actions\BuildFulfillmentDataFromSalesOrders;
use Illuminate\Support\Collection;
use Throwable;

class BulkFulfillmentManager extends FulfillmentManager
{

    /**
     * @var array
     */
    private array $errors = [];
    /**
     * @var int
     */
    private int $errorCount = 0;
    /**
     * @var array
     */
    private array $linesWithoutEnoughReservations = [];

    public function fulfillBulkOrders(BulkActionData $payload): BulkActionResultData
    {

        $total = 0;

        $this->orders->chunkBy($payload, function(Collection $salesOrders) use (&$total) {
            $total += $salesOrders->count();
            $this->fulfillOrders($salesOrders);
        });

        return BulkActionResultData::from([
            'errors' => $this->errors,
            'total' => $total,
            'success' => $total - $this->errorCount,
        ]);
    }


    /**
     * @param  Collection  $salesOrders
     * @return void
     * @throws Throwable
     */
    public function fulfillOrders(Collection $salesOrders): void
    {
        $resultData = app(BuildFulfillmentDataFromSalesOrders::class)->handle($salesOrders);

        if (!empty($resultData->errors)) {
            $this->errors = array_merge($this->errors, [
                'message' => 'Some errors occurred while preparing the fulfillments.',
                'errors' => $resultData->errors
            ]);
        }

        $salesOrderFulfillmentCollection = $resultData->salesOrderFulfillments;

        // Create the fulfillments and their lines.
        $fulfillments = $this->fulfillments->createMany($salesOrderFulfillmentCollection);

        // Negate inventory movements for reservations.
        [$nonDropshipLines, $dropshipLines] = $this->splitLinesByDropship(
            $salesOrderFulfillmentCollection->pluck('fulfillment_lines')->flatten(1)
        );

        if(!empty($nonDropshipLines)) {
            $this->negateReservationMovementsForMany($nonDropshipLines, $fulfillments);
        }

        if(!empty($dropshipLines)) {
            $this->shipDropshipPurchaseOrdersForMany($fulfillments);
        }

        // Next, we attempt to submit the fulfillments to third parties.
        // This includes shipping providers and sales channels for manual orders.
        $this->submitNonDropshipFulfillmentsToThirdParties($fulfillments);


    }

    /**
     * @param  Collection  $fulfillmentLines
     * @return array
     */
    private function splitLinesByDropship(Collection $fulfillmentLines): array
    {
        return [
            $fulfillmentLines->where('is_dropship', false),
            $fulfillmentLines->where('is_dropship', true)
        ];
    }

    /**
     * @param  Collection  $nonDropshipLines
     * @param  Collection  $fulfillments
     * @return void
     */
    private function negateReservationMovementsForMany(Collection $nonDropshipLines, Collection $fulfillments): void
    {
        $reservations = $this->fulfillments
            ->getExistingReservationsForLines($nonDropshipLines->pluck('sales_order_line_id')->toArray())
            ->groupBy('link_id');

        $fulfillmentLines = $fulfillments->pluck('salesOrderFulfillmentLines')->flatten(1);

        $movements = [];
        $fulfilledQuantityUpdates = [];

        foreach($reservations as $salesOrderLineId => $lineMovements){

            $matchingLine = $nonDropshipLines->firstWhere('sales_order_line_id', $salesOrderLineId);
            if(!$matchingLine) continue;

            $quantity = $matchingLine['fulfillable_quantity'];
            $fulfillmentLine = $fulfillmentLines->where('sales_order_line_id', $salesOrderLineId)->firstOrFail();

            /** @var SalesOrderFulfillment $fulfillment */
            $fulfillment = $fulfillments
                ->where('id', $fulfillmentLine->sales_order_fulfillment_id)
                ->firstOrFail();

            $movementDate = ($fulfillment->fulfilled_at ?? $fulfillment->created_at)->format('Y-m-d H:i:s');

            // We initialize to the existing fulfilled quantity for the line.
            $fulfilledQuantityForLine = $lineMovements->first()?->fulfilled_quantity ?? 0;

            /** @var InventoryMovement $lineMovement */
            foreach($lineMovements as $lineMovement){
                $applicableQuantity = $lineMovement->quantity;
                $fulfilledQuantityForLine += $applicableQuantity;
                unset($lineMovement->fulfilled_quantity);
                $now = now();
                if($quantity >= $lineMovement->quantity){
                    $quantity -= $applicableQuantity;
                    $negativeMovement = $lineMovement->replicate();
                    $negativeMovement->quantity = -$lineMovement->quantity;
                    $negativeMovement->link_id = $fulfillmentLine->id;
                    $negativeMovement->link_type = SalesOrderFulfillmentLine::class;
                    $movement = $negativeMovement->toArray();
                    $movement['inventory_movement_date'] = $movementDate;
                    $movement['reference'] = $fulfillment->getReference();
                    $movement['created_at'] = $now;
                    $movement['updated_at'] = $now;
                    $movements[] = $movement;
                } else {
                    $negativeMovement = $lineMovement->replicate();
                    $negativeMovement->quantity = -$applicableQuantity;
                    $negativeMovement->link_id = $fulfillmentLine->id;
                    $negativeMovement->link_type = SalesOrderFulfillmentLine::class;
                    $negativeMovement->inventory_movement_date = $movementDate;

                    $movement = $negativeMovement->toArray();
                    $movement['reference'] = $fulfillment->getReference();
                    $movement['inventory_movement_date'] = $movementDate;
                    $movement['created_at'] = $now;
                    $movement['updated_at'] = $now;
                    $movements[] = $movement;

                    break;
                }

                if($quantity <= 0) break;
            }

            $fulfilledQuantityUpdates[] = [
                'id' => $salesOrderLineId,
                'fulfilled_quantity' => $fulfilledQuantityForLine
            ];
        }


        $this->fulfillments->createReservationMovements($movements);
        // Update the fulfilled quantity for the sales order lines.
        if(!empty($fulfilledQuantityUpdates)){
            $this->orders->updateFulfilledQuantityCacheForLines($fulfilledQuantityUpdates);
        }
    }

    /**
     * @param  Collection  $fulfillments
     * @return void
     */
    private function shipDropshipPurchaseOrdersForMany(Collection $fulfillments): void
    {
        // For now, we don't handle dropship in bulk.
        $dropshipOrders = $fulfillments
            ->whereNotNull('warehouse.supplier_id')
            ->whereNotNull('warehouse.dropship_enabled')
            ->where('warehouse.dropship_enabled', true);

        if($dropshipOrders->isEmpty()) return;

        $dropshipOrders->each(/**
         */ function(SalesOrderFulfillment $fulfillment) {
            try{
                $this->shipDropshipPurchaseOrders(
                    $fulfillment,
                    $fulfillment->warehouse
                );
            }catch (Throwable $e){
                // We delete the fulfillment so that it can be retried later.
                $this->errors[] = [
                    'id' => $fulfillment->sales_order_id,
                    'message' => $e->getMessage()
                ];
                $fulfillment->delete();
                $this->errorCount++;
            }
        });
    }

    /**
     * @param  Collection  $fulfillments
     * @return void
     */
    private function submitNonDropshipFulfillmentsToThirdParties(Collection $fulfillments): void
    {
        $fulfillments
            ->where('warehouse.dropship_enabled', false)
            ->whereNull('warehouse.supplier_id')
            ->each(function(SalesOrderFulfillment $fulfillment){
                dispatch(new SubmitFulfillmentToThirdPartiesJob($fulfillment));
            });
    }

}