<?php

namespace App\Services\SalesOrder;

use App\Jobs\ShipStation\AutoFulfillmentOrder;
use App\Models\SalesOrder;
use App\Response;
use Illuminate\Http\Request;

class BulkApproveSalesOrderService
{
    protected $request;

    private $response = [];

    private $errorsCount = 0;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    /**
     * @return array array of errors
     *
     * @throws \Throwable
     */
    public function approve(): array
    {
        set_time_limit(0);

        $this->getBuilder()->each(function (SalesOrder $salesOrder) {
            if (! $salesOrder->isDraft() && ! $salesOrder->isReserved()) {
                $this->response[$salesOrder->id][] = Response::getError(__('messages.sales_order.is_not_draft', ['id' => $salesOrder->sales_order_number]), 'SalesOrder'.Response::CODE_IS_NOT_DRAFT, 'id');
                $this->errorsCount++;

                return; // continue
            }

            $salesOrder->approve(ApproveSalesOrderService::APPROVE_DROPSHIP_UPDATE_ONLY);
            if ($salesOrder->isOpen()) {
                ( new AutoFulfillmentOrder($salesOrder) )->fulfillOpenOrder();
            }
            $this->response[$salesOrder->id] = __('messages.sales_order.approve_success', ['id' => $salesOrder->sales_order_number]);
        });

        return $this->response;
    }

    public function getErrorsCount(): int
    {
        return $this->errorsCount;
    }

    private function getBuilder()
    {
        $builder = SalesOrder::with([
            'salesOrderLines.warehouse',
            'salesOrderFulfillments',
            'purchaseOrders',
        ]);
        if ($this->request->has('ids')) {
            $builder->whereIn('id', array_unique($this->request->input('ids', [])));
        } else {
            /*
             * Not sure what this is for?
             */
            $builder->filter($this->request)->archived($this->request->input('archived', 0));
        }

        return $builder;
    }
}
