<?php

namespace App\Services\SalesOrder\Actions;

use App\Data\SalesOrderLineData;
use App\Data\UpdateSalesOrderData;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use Closure;
use Illuminate\Support\Collection;
use Spatie\LaravelData\DataCollection;
use Spatie\LaravelData\Optional;
use Throwable;

// TODO: replace bundle with kit if in stock (See SalesOrder::splitBundleComponents)
class TransformBundlesSalesOrderLinesIntoComponentsForUpdate
{
    public function handle(UpdateSalesOrderData $data, Closure $next)
    {
        if ($data->payload->sales_order_lines instanceof Optional) {
            return $next($data);
        }

        $salesOrder = &$data->salesOrder;
        $payloadSalesOrderLines = &$data->payload->sales_order_lines;

        $bundleLines = $this->getBundleLines($payloadSalesOrderLines);

        if ($bundleLines->count() == 0) {
            return $next($data);
        }

        $this->transformBundleLinesIntoComponents($data, $bundleLines, $salesOrder, $payloadSalesOrderLines);

        return $next($data);
    }

    /**
     * @throws Throwable
     */
    private function getBundleQuantityCache(Collection $componentLines): int
    {
        throw_if(!$componentLines->every(
            fn (SalesOrderLine $line) => $line->bundle_quantity_cache === $componentLines->first()->bundle_quantity_cache
        ), 'Bundle quantity cache is not uniform for ' . $componentLines->first()->salesOrder->sales_order_number);

        $bundleQuantityCache = $componentLines->first()->bundle_quantity_cache;

        throw_if(!$bundleQuantityCache, 'Bundle quantity cache is not set for ' . $componentLines->first()->salesOrder->sales_order_number);

        return $bundleQuantityCache;
    }

    private function transformBundleLinesIntoComponents(
        UpdateSalesOrderData &$data,
        DataCollection $bundleLines,
        SalesOrder $salesOrder,
        DataCollection &$payloadSalesOrderLines
    ): void
    {
        $bundleLines->each(function (SalesOrderLineData $payloadBundleLine) use (
            $salesOrder,
            &$payloadSalesOrderLines,
            &$data,
        ) {
            $componentLines = $salesOrder->salesOrderLines
                ->whereNotNull('bundle_id')
                ->where('sales_channel_line_id', $payloadBundleLine->sales_channel_line_id);
            if ($componentLines->isNotEmpty()) {
                $payloadSalesOrderLines = $this->handleBundleQuantityChange(
                    $data,
                    $componentLines,
                    $payloadBundleLine,
                    $payloadSalesOrderLines
                );
            } else {
                $payloadSalesOrderLines = $this->splitBundleLineIntoComponents($payloadBundleLine, $payloadSalesOrderLines);
            }
        });

        $payloadSalesOrderLines = $payloadSalesOrderLines->reject(function (SalesOrderLineData $line) {
            return (!$line->product instanceof Optional) && $line->product->type === Product::TYPE_BUNDLE;
        })->values();
    }

    private function getBundleLines(DataCollection $payloadSalesOrderLines): DataCollection
    {
        return $payloadSalesOrderLines->filter(function (SalesOrderLineData $line) {
            if ($line->product instanceof Optional) {
                return false;
            }
            return $line->product?->type === Product::TYPE_BUNDLE;
        });
    }

    /**
     * @throws Throwable
     */
    private function handleBundleQuantityChange(
        UpdateSalesOrderData &$data,
        Collection $componentLines,
        SalesOrderLineData $payloadBundleLine,
        DataCollection $payloadSalesOrderLines
    ): DataCollection
    {
        $bundleQuantityCache = $this->getBundleQuantityCache($componentLines);

        $multiplier = 1;
        if ($bundleQuantityCache != $payloadBundleLine->quantity) {
            $data->payload->shouldDeleteFulfillments = true;
            $data->payload->deleteFulfillmentsReason = 'Bundle quantity changed';
            $multiplier                              = $payloadBundleLine->quantity / $bundleQuantityCache;
        }
        if (!$payloadBundleLine->quantity_to_cancel instanceof Optional && $payloadBundleLine->quantity_to_cancel > 0) {
            $data->payload->shouldDeleteFulfillments = true;
            $data->payload->deleteFulfillmentsReason = 'Bundle quantity was cancelled';
            $multiplier = $payloadBundleLine->quantity / $bundleQuantityCache;
        }

        $componentLines->each(function (SalesOrderLine $componentLine) use (
            $multiplier,
            $payloadBundleLine,
            &$payloadSalesOrderLines
        ) {
            $componentLine->quantity              = $multiplier * $componentLine->quantity;
            $componentLine->bundle_quantity_cache = $payloadBundleLine->quantity;
            $componentLine = SalesOrderLineData::from($componentLine);
            if (!$payloadBundleLine->quantity_to_cancel instanceof Optional) {
                $componentLine->quantity_to_cancel = ($multiplier * $payloadBundleLine->quantity_to_cancel
                    * $componentLine->bundle_component_quantity_cache) - $componentLine->canceled_quantity;
            }
            $payloadSalesOrderLines[] = $componentLine;
        });
        return $payloadSalesOrderLines;
    }

    /**
     * @param  SalesOrderLineData  $payloadBundleLine
     * @param  DataCollection  $payloadSalesOrderLines
     * @return DataCollection
     * @throws \Exception
     */
    private function splitBundleLineIntoComponents(
        SalesOrderLineData $payloadBundleLine,
        DataCollection $payloadSalesOrderLines
    ): DataCollection {
        // Add component lines, once split into components
        $payloadBundleLine->product->components->each(function (Product $componentProduct) use (
            $payloadBundleLine,
            &$payloadSalesOrderLines
        ) {
            $quantityToCancel = $payloadBundleLine->quantity_to_cancel instanceof Optional
                ? 0
                : $payloadBundleLine->quantity_to_cancel;
            $externallyFulfilledQuantity = $payloadBundleLine->externally_fulfilled_quantity instanceof Optional
                ? 0
                : $payloadBundleLine->externally_fulfilled_quantity;
            $newComponentLine              = clone $payloadBundleLine;
            $newComponentLine->id          = null;
            $componentQuantity = $componentProduct->pivot->quantity;
            $newComponentLine->product_id  = $componentProduct->id;
            $newComponentLine->product     = $componentProduct;
            $newComponentLine->bundle_id   = $payloadBundleLine->product_id;
            $newComponentLine->description = $componentProduct->name ?? $componentProduct->sku;
            $newComponentLine->quantity                      = $payloadBundleLine->quantity * $componentQuantity;
            $newComponentLine->externally_fulfilled_quantity = $externallyFulfilledQuantity * $componentQuantity;
            $newComponentLine->amount                        = $payloadBundleLine->amount * $componentProduct->getBundlePriceProration($payloadBundleLine->product);
            $newComponentLine->quantity_to_cancel            = $quantityToCancel * $componentQuantity;
            $newComponentLine->bundle_quantity_cache         = $payloadBundleLine->quantity;
            $payloadSalesOrderLines[]                        = $newComponentLine;
        });

        // Existing bundle line needs to be deleted
        if (!$payloadBundleLine->id instanceof Optional) {
            SalesOrderLine::where('id', $payloadBundleLine->id)->delete();
        }

        return $payloadSalesOrderLines;
    }
}