<?php

namespace App\Services\SalesOrder\Actions;

use App\Data\SalesOrderLineData;
use App\Data\UpdateSalesOrderData;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Repositories\InventoryMovementRepository;
use Closure;
use Spatie\LaravelData\Optional;

class SyncInventoryMovementReferences
{

    public function __construct(
        private readonly InventoryMovementRepository $movements
    ) {}


    /**
     * Update the references of the inventory movements associated with the sales order
     * to the new sales order number.
     *
     * @param  UpdateSalesOrderData  $data
     * @param  Closure  $next
     * @return mixed
     */
    public function handle(UpdateSalesOrderData $data, Closure $next): mixed
    {
        if(
            $data->payload->sales_order_number instanceof Optional ||
            $data->salesOrder->sales_order_number === $data->payload->sales_order_number
        ) {
            return $next($data);
        }

        // Update the references of the inventory movements associated with the sales order
        // to the new sales order number.
        $salesOrderLineIds = $data->salesOrder->salesOrderLines->pluck('id')->toArray();

        $this->movements->updateReferences(
            ids: $salesOrderLineIds,
            type: SalesOrderLine::class,
            reference: $data->payload->sales_order_number,
        );

        // Load sales order fulfillments lines.
        $data->salesOrder->loadMissing('salesOrderFulfillments.salesOrderFulfillmentLines');
        $data->salesOrder->salesOrderFulfillments->each(function(SalesOrderFulfillment $fulfillment) use ($data) {
            $salesOrderFulfillmentLineIds = $fulfillment->salesOrderFulfillmentLines->pluck('id')->toArray();
            $this->movements->updateReferences(
                ids: $salesOrderFulfillmentLineIds,
                type: SalesOrderFulfillmentLine::class,
                reference: "{$data->payload->sales_order_number}.$fulfillment->fulfillment_sequence",
            );
        });


        return $next($data);
    }

}