<?php

namespace App\Services\SalesOrder\Actions;

use App\Data\SalesOrderLineData;
use App\Data\UpdateSalesOrderData;
use App\Exceptions\CantUpdateNonExistingSalesOrderLineException;
use Closure;
use Illuminate\Support\ItemNotFoundException;
use Spatie\LaravelData\Optional;

class LoadExistingSalesOrderLines
{
    public function handle(UpdateSalesOrderData $data, Closure $next)
    {
        if ($data->payload->sales_order_lines instanceof Optional) {
            return $next($data);
        }

        $data->payload->sales_order_lines = $data->payload->sales_order_lines->map(function (SalesOrderLineData $line) use ($data)
        {
            if ($line->id && !$line->id instanceof Optional) {
                try {
                    $line->salesOrderLine = $data->salesOrder->salesOrderLines->where('id', $line->id)->firstOrFail();
                } catch (ItemNotFoundException) {
                    throw new CantUpdateNonExistingSalesOrderLineException("Can't update non-existing sales order line with id: {$line->id}. for sales order with id: {$data->salesOrder->sales_order_number}");
                }
                return $line;
            }

            /*
             * TODO: There there is a potential edge case when a sales channel, which only knows about the original sales
             *  order line, tries to update the original sales order line, there is currently no logic to manage the re-merging
             *  or re-splitting of the sales order line. This is a rare edge case, but it should be handled.  It has never
             *  been handled though so shouldn't need to be handled right away.
             */
            $matchingLine = $data->salesOrder
                ->salesOrderLines
                ->where('sales_channel_line_id', $line->sales_channel_line_id instanceof Optional ? null : $line->sales_channel_line_id)
                ->where('product_listing_id', $line->product_listing_id instanceof Optional ? null : $line->product_listing_id)
                ->where('split_from_line_id', $line->split_from_line_id instanceof Optional ? null : $line->split_from_line_id)
                ->where('product_id', $line->product_id instanceof Optional ? null : $line->product_id)
                ->first();

            if ($matchingLine) {
                $line->salesOrderLine = $matchingLine;
            }

            return $line;
        });

        return $next($data);
    }
}