<?php

namespace App\Services\SalesOrder\Actions;

use App\Data\SalesOrderLineData;
use App\Data\UpdateSalesOrderData;
use App\Models\SalesOrder;
use App\Models\TaxRate;
use App\Repositories\TaxRateRepository;
use Closure;
use Spatie\LaravelData\Optional;

class HandleTaxesForSalesOrderUpdate
{
    public function __construct(
        private readonly TaxRateRepository $taxRates
    ) {}

    public function handle(UpdateSalesOrderData $data, Closure $next)
    {
        if ($data->payload->sales_order_lines instanceof Optional) {
            return $next($data);
        }

        $taxRateIds =  $data
            ->payload
            ->sales_order_lines
            ->reject(fn($line) => !$line->tax_rate_id || $line->tax_rate_id instanceof Optional)
            ->map(fn(SalesOrderLineData $line) => $line->tax_rate_id)
            ->toCollection()
            ->unique()
            ->toArray();

        $taxRateCollection = $this->taxRates->getForValues(
            $taxRateIds,
            'id',
            TaxRate::class
        );

        $data->payload->sales_order_lines->each(function (SalesOrderLineData $line) use ($data, $taxRateCollection)
        {
            if ($line->tax_rate_id instanceof Optional) {
                return $line;
            }

            if ($taxRate = $taxRateCollection->firstWhere('id', $line->tax_rate_id))
            {
                $line->tax_allocation = $this->getTaxAmount($data->salesOrder, $line, $taxRate);
                $line->tax_rate = $taxRate->rate;
            }

            return $line;
        });

        return $next($data);
    }

    private function getTaxAmount(SalesOrder $salesOrder, SalesOrderLineData $line, TaxRate $taxRate): float
    {
        $amount = $line->amount instanceof Optional ? $line->salesOrderLine->amount : $line->amount;
        $total = $line->quantity * $amount;
        $rate = $taxRate->rate / 100;
        return $salesOrder->is_tax_included ? $total - ($total / (1 + $rate)) : $total * $rate;
    }
}