<?php

namespace App\Services\SalesOrder\Actions;

use App\Data\UpdateSalesOrderData;
use App\Models\Integration;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use Closure;
use Exception;
use Throwable;

class DeleteSubmittedFulfillmentsDueToSalesOrderChange
{
    /**
     * @throws Exception
     */
    public function handle(UpdateSalesOrderData $data, Closure $next)
    {
        if ($this->shouldSkipDeletion($data)) {
            return $next($data);
        }

        $deletableSalesOrderFulfillments = $this->getDeletableFulfillments($data);

        if ($deletableSalesOrderFulfillments->isEmpty()) {
            return $next($data);
        }

        try {
            $this->deleteFulfillments($deletableSalesOrderFulfillments);
        } catch (Exception $e) {
            $this->handleDeletionError($data, $e);
        }

        $this->addNoteAndMarkOutOfSync($data);

        return $next($data);
    }

    private function shouldSkipDeletion(UpdateSalesOrderData $data): bool
    {
        return !$data->payload->shouldDeleteFulfillments ||
            (
                $data->salesOrder->salesChannel->integrationInstance->integration->name == Integration::NAME_SKU_IO &&
                $data->payload->shippingAddressChanged == false
            ) ||
            $data->salesOrder->fulfillment_status == SalesOrder::FULFILLMENT_STATUS_FULFILLED ||
            $data->salesOrder->order_status == SalesOrder::STATUS_CLOSED;
    }

    private function getDeletableFulfillments(UpdateSalesOrderData $data)
    {
        return $data->salesOrder
            ->salesOrderFulfillments
            ->where('status', SalesOrderFulfillment::STATUS_SUBMITTED)
            ->whereNull('packing_slip_printed_at');
    }

    private function deleteFulfillments($fulfillments): void
    {
        $fulfillments->each(/**
         * @throws Throwable
         */ function (SalesOrderFulfillment $fulfillment) {
            $fulfillment->delete();
        });
    }

    /**
     * @throws Exception
     */
    private function handleDeletionError(UpdateSalesOrderData $data, Exception $e): void
    {
        $note = 'We tried to delete the fulfillment from the shipping provider due to a change in the sales order lines, which failed with the following error message from the shipping provider: '.$e->getMessage();
        customlog('out-of-sync', $data->salesOrder->sales_order_number.' had fulfillments that were submitted (with no packing slip printed) deleted, so marking out of sync');
        app(AddSalesOrderNote::class)->handle($data, $note);
    }

    /**
     * @throws Exception
     */
    private function addNoteAndMarkOutOfSync(UpdateSalesOrderData $data): void
    {
        $note = 'There were submitted fulfillments that were removed from shipping provider due to the following reason: ' . $data->payload->deleteFulfillmentsReason . '.  So we marked the fulfillment status as Out of Sync.';
        customlog('out-of-sync', $data->salesOrder->sales_order_number.' had fulfillments that were submitted (with no packing slip printed) deleted, so marking out of sync');
        app(AddSalesOrderNote::class)->handle($data, $note);
        $data->salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC;
    }
}