<?php

/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 9/25/20
 * Time: 1:44 PM.
 */

namespace App\Services\PurchaseOrder;

use App\Data\AdjustmentForReceiptData;
use App\DTO\FifoLayerDto;
use App\DTO\InventoryMovementDto;
use App\DTO\PurchaseOrderLineDto;
use App\DTO\PurchaseOrderShipmentReceiptDto;
use App\DTO\PurchaseOrderShipmentReceiptLineDto;
use App\Events\SKUProgressEvent;
use App\Exceptions\InsufficientStockException;
use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\PurchaseOrder\ReceivePurchaseOrderLineException;
use App\Helpers;
use App\Jobs\AutomatedSalesOrderFulfillmentJob;
use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Jobs\ReleaseBackorderQueuesJob;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\FifoLayer;
use App\Models\IntegrationInstance;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductInventory;
use App\Models\ProductListing;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipment;
use App\Models\PurchaseOrderShipmentLine;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesOrder;
use App\Models\Warehouse;
use App\Notifications\MonitoringMessage;
use App\Repositories\FifoLayerRepository;
use App\Repositories\InventorySnapshotRepository;
use App\Repositories\PurchaseOrderLineRepository;
use App\Repositories\PurchaseOrderShipmentLineRepository;
use App\Repositories\PurchaseOrderShipmentReceiptLineRepository;
use App\Repositories\PurchaseOrderShipmentReceiptRepository;
use App\Repositories\PurchaseOrderShipmentRepository;
use App\Services\InventoryManagement\BackorderManager;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\Product\ReleaseBackorderQueues;
use App\Services\Receipts\CreatesAdjustmentForReceipt;
use Carbon\Carbon;
use Config;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use InvalidArgumentException;
use Str;
use Throwable;

const UNEXPECTED_RECEIPT_ACTION_ADD_TO_PO = 'add_to_po';
const UNEXPECTED_RECEIPT_ACTION_CREATE_ADJUSTMENT = 'create_adjustment';

/**
 * Class ShipmentManager.
 */
class ShipmentManager
{

    use CreatesAdjustmentForReceipt;

    private InventorySnapshotRepository $inventorySnapshotRepository;

    private PurchaseOrderLineRepository $purchaseOrderLineRepository;

    private PurchaseOrderShipmentRepository $purchaseOrderShipmentRepository;

    private PurchaseOrderShipmentReceiptRepository $purchaseOrderShipmentReceiptRepository;

    private PurchaseOrderShipmentLineRepository $purchaseOrderShipmentLineRepository;

    private PurchaseOrderShipmentReceiptLineRepository $purchaseOrderShipmentReceiptLineRepository;

    private FifoLayerRepository $fifoLayerRepository;

    public function __construct()
    {
        $this->inventorySnapshotRepository = app(InventorySnapshotRepository::class);
        $this->purchaseOrderLineRepository = app(PurchaseOrderLineRepository::class);
        $this->purchaseOrderShipmentRepository = app(PurchaseOrderShipmentRepository::class);
        $this->purchaseOrderShipmentReceiptRepository = app(PurchaseOrderShipmentReceiptRepository::class);
        $this->purchaseOrderShipmentLineRepository = app(PurchaseOrderShipmentLineRepository::class);
        $this->purchaseOrderShipmentReceiptLineRepository = app(PurchaseOrderShipmentReceiptLineRepository::class);
        $this->fifoLayerRepository = app(FifoLayerRepository::class);
    }

    /**
     * @return mixed
     */
    public function createShipment(array $payload)
    {
        return DB::transaction(function () use ($payload) {
            /** @var PurchaseOrder $purchaseOrder */
            $purchaseOrder = PurchaseOrder::with([])->findOrFail($payload['purchase_order_id']);
            // The purchase order must be open
            $this->checkPurchaseOrderStatus($purchaseOrder);

            // create Purchase Order Shipment
            // add Purchase Order Shipment lines
            if (isset($payload['shipment_lines']) && ! empty($payload['shipment_lines'])) {
                /** @var PurchaseOrderShipment $purchaseOrderShipment */
                $purchaseOrderShipment = $purchaseOrder->purchaseOrderShipments()->create($payload);

                // First, bind in shipment lines ids for lines with line reference
                $payload['shipment_lines'] = PurchaseOrderShipment::aggregateShipmentLines($payload['shipment_lines']);
                $purchaseOrderShipment->purchaseOrderShipmentLines()->createMany($payload['shipment_lines']);
            } else {
                // No lines are provided, we ensure that all the lines are shipped
                $purchaseOrderShipment = $purchaseOrder->shipAllLines($payload);
            }

            // mark Purchase Order as shipped
            $purchaseOrder->shipped($purchaseOrderShipment->shipment_date);

            return $purchaseOrderShipment;
        });
    }

    /**
     * @throws NotOpenPurchaseOrderException
     */
    private function checkPurchaseOrderStatus(PurchaseOrder $purchaseOrder)
    {
        if ($purchaseOrder->isDraft()) {
            throw new NotOpenPurchaseOrderException($purchaseOrder);
        }
    }

    /**
     * @throws NotOpenPurchaseOrderException
     */
    private function parseUnexpectedItemsInReceipt(array $payload): array
    {
        $receiptLines = $payload['receipt_lines'] ?? [];
        foreach ($payload['unexpected_items'] as $key => $unexpectedItem) {
            $product = $this->getProductForUnexpectedItem($unexpectedItem);
            // Attempt to find a PO line for the product
            /** @var PurchaseOrderLine $matchingLine */
            $matchingLine = PurchaseOrderLine::with([])
                ->where('purchase_order_id', $payload['purchase_order_id'])
                ->where('product_id', $product->id)
                ->first();

            if ($matchingLine) {
                // An unexpected item is actually on the purchase order.
                // We receive the min of quantity remaining and quantity in receipt
                // and we register any excess quantity as unexpected item (creating positive adjustment).
                $unreceivedQuantity = $matchingLine->quantity - $matchingLine->received_quantity;
                $minQuantity = min($unreceivedQuantity, $unexpectedItem['quantity']);
                if ($minQuantity > 0) {
                    $receiptLines[] = array_merge($unexpectedItem, [
                        'quantity' => $minQuantity,
                        'purchase_order_line_id' => $matchingLine->id,
                    ]);
                }

                $excessQuantity = $matchingLine->received_quantity + $unexpectedItem['quantity'] - $matchingLine->quantity;
                if ($excessQuantity > 0) {
                    $payload['unexpected_items'][$key]['quantity'] = $excessQuantity;
                }
            }
        }

        if (! empty($receiptLines)) {
            $payload['receipt_lines'] = $receiptLines;
        }

        $this->handleUnexpectedItemsInReceipt($payload['unexpected_items'], $payload);

        return $payload;
    }

    /**
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     * @throws Throwable
     */
    public function receiveShipment(
        array $payload,
        bool $addToInventory = true
    ): PurchaseOrderShipmentReceipt|PurchaseOrder|Model {
        /** @var PurchaseOrderShipmentReceipt|PurchaseOrder $result */
        $result = null;

        // Get lines with negative quantities and create adjustments for those.
        if ($this->hasNegativeReceiptLines($payload)) {
            $payload = $this->handleNegativeReceiptLines($payload);
            if(empty($payload['receipt_lines'])) {
                return PurchaseOrder::query()->findOrFail($payload['purchase_order_id']);
            }
        }

        // When an unexpected item is on the purchase order,
        // we add it as a receipt line
        if ($this->hasUnexpectedItems($payload)) {
            $payload = $this->parseUnexpectedItemsInReceipt($payload);
        }

        if ($this->hasReceiptLines($payload)) {
            /**
             * We receive small lines immediately
             * while deferring larger lines to the queue.
             */
            $result = $this->newReceiveLines($payload['receipt_lines'], $payload);
        } else {
            // No shipment lines provided, we either completely
            // receive for the shipment provided (if any) or
            // completely receive the purchase order.
            if ($this->receivingForShipment($payload)) {
                $result = $this->completelyReceiveForShipment($payload['purchase_order_shipment_id'], $payload, $addToInventory);
            } elseif (! empty($payload['purchase_order_id']) && empty($payload['unexpected_items'])) {
                $result = $this->completelyReceiveForOrderWithDataFeed($payload['purchase_order_id'], $payload);
            //$purchaseOrder->refreshReceiptStatus($payload['received_at'] ? Carbon::parse($payload['received_at']) : null);
            } else {
                Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage('receiveShipment with unexpected payload: '.json_encode($payload)));
            }
        }

        return $result;
    }

    private function hasNegativeReceiptLines(array $payload): bool
    {
        return !empty($payload['receipt_lines']) &&
            collect($payload['receipt_lines'])->contains(fn ($line) => $line['quantity'] < 0);
    }

    /**
     * @throws Throwable
     * @throws ReceivePurchaseOrderLineException
     * @throws InsufficientStockException
     * @throws NotOpenPurchaseOrderException
     */
    private function handleNegativeReceiptLines(array $payload): array
    {
        $negativeReceiptLines = collect($payload['receipt_lines'])->filter(fn ($line) => $line['quantity'] < 0);
        $positiveReceiptLines = collect($payload['receipt_lines'])->filter(fn ($line) => $line['quantity'] > 0);

        // We create adjustments for the negative lines
        if($negativeReceiptLines->isNotEmpty()) {
            $purchaseOrder = $this->validateOrderFromReceiptLines($payload['receipt_lines']);
            $negativeReceiptPOLineIds = $negativeReceiptLines->pluck('purchase_order_line_id')->values()->all();
            $purchaseOrderLines = $purchaseOrder->purchaseOrderLines
                ->whereIn('id', $negativeReceiptPOLineIds)
                ->keyBy('id')
                ->toArray();

            $existingReceiptFifoLayers = PurchaseOrderShipmentReceiptLine::with(['fifoLayers'])
                ->whereHas('purchaseOrderLine', function (Builder $query) use ($negativeReceiptPOLineIds) {
                    $query->whereIn('id', $negativeReceiptPOLineIds);
                })
                ->get()
                ->keyBy('purchase_order_line_id')
                ->toArray();

            if(empty($existingReceiptFifoLayers)){
                throw new ReceivePurchaseOrderLineException(
                    'No existing receipt fifo layers found for negative receipt lines.'
                );
            }

            $lineReceivedQuantityUpdates = [];

            $negativeReceiptLines->each(/**
             * @throws InsufficientStockException
             * @throws Throwable
             */ function ($line) use (
                $purchaseOrder,
                $purchaseOrderLines,
                $payload,
                $existingReceiptFifoLayers,
                &$lineReceivedQuantityUpdates
            ) {
                $purchaseOrderLine = $purchaseOrderLines[$line['purchase_order_line_id']];
                $existingReceiptFifoIds = collect(
                    $existingReceiptFifoLayers[$line['purchase_order_line_id']]['fifo_layers'] ?? []
                )->pluck('id')->toArray();

                if(empty($existingReceiptFifoIds)){
                    throw new ReceivePurchaseOrderLineException(
                        'No existing receipt fifo layers found for negative receipt lines.'
                    );
                }

                $this->createAdjustment(
                    AdjustmentForReceiptData::from([
                        'quantity' => $line['quantity'],
                        'warehouse_id' => $purchaseOrder->destination_warehouse_id,
                        'product_id' => $purchaseOrderLine['product_id'],
                        'note' => 'Negative purchase receipt.',
                        'received_at' => $payload['received_at'],
                        'link_type' => PurchaseOrderLine::class,
                        'link_id' => $purchaseOrderLine['id'],
                        'applicable_fifo_layer_ids' => $existingReceiptFifoIds,
                    ])
                );

                $lineReceivedQuantityUpdates[] = [
                    'id' => $purchaseOrderLine['id'],
                    'received_quantity' => $purchaseOrderLine['received_quantity'] + $line['quantity'],
                ];
            });
            $payload['purchase_order_id'] = $purchaseOrder->id;

            // Update received quantity for negative lines
            batch()->update(new PurchaseOrderLine, $lineReceivedQuantityUpdates, 'id');
        }

        // Return remaining payload for positive receipt quantities
        $payload['receipt_lines'] = $positiveReceiptLines->toArray();
        return $payload;
    }

    /**
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     * @throws Throwable
     */
    public function completelyReceiveForOrderWithDataFeed(int $purchaseOrderId, array $payload): PurchaseOrderShipmentReceipt
    {
        /** @var PurchaseOrder $purchaseOrder */
        $purchaseOrder = PurchaseOrder::query()->findOrFail($purchaseOrderId);

        $receiptLines = [];

        $purchaseOrder->purchaseOrderLines->each(function (PurchaseOrderLine $purchaseOrderLine) use (&$receiptLines) {
            $receiptLines[] = [
                'purchase_order_line_id' => $purchaseOrderLine->id,
                'quantity' => $purchaseOrderLine->quantity - $purchaseOrderLine->received_quantity,
            ];
        });

        return $this->newReceiveLines($receiptLines, $payload);
    }

    private function getProductForUnexpectedItem(array $unexpectedItem): Product|Model
    {
        return isset($unexpectedItem['product_id']) ?
            Product::with([])->findOrFail($unexpectedItem['product_id']) :
            Product::with([])->where('sku', $unexpectedItem['product_sku'])->firstOrFail();
    }

    /**
     * @throws NotOpenPurchaseOrderException
     */
    public function receiveLines(
        array $receiptLines,
        array $payload,
        ?bool $addToInventory = true,
        ?PurchaseOrderShipment $shipment = null,
        ?int $userId = null,
    ): PurchaseOrderShipmentReceipt {
        // Aggregate the purchase order receipt lines
        $receiptLines = PurchaseOrderShipmentReceipt::aggregateReceiptLines($receiptLines);

        // We get the purchase order from the purchase order lines
        $purchaseOrder = $this->validateOrderFromReceiptLines($receiptLines);

        // Next, we get the purchase order shipment. If none is provided,
        // we create a new shipment for the receipt lines
        if (! $shipment) {
            $shipment = $this->getShipmentForReceipts($purchaseOrder, $payload);
        }

        // Create Purchase Order Shipment Receipt
        /** @var PurchaseOrderShipmentReceipt $receipt */
        $receipt = $shipment->purchaseOrderShipmentReceipts()->create(array_merge($payload, [
            'user_id' => $userId ?? auth()->user()?->id,
        ]));

        /**
         * We create a shipment and receipt line for each
         * of the receipt lines and receive it into inventory.
         */
        $progress = SKUProgressEvent::sendOn(
            $purchaseOrder->getReceiptBroadcastChannel(),
            count($receiptLines)
        );
        foreach ($receiptLines as $receiptLine) {
            $this->receiveLine($receiptLine, $shipment, $receipt, $addToInventory);
            // Update the progress
            $progress->advance();
        }

        /**
         * In case there are no successful shipment lines
         * or receipt lines, we remove the shipment and
         * receipt respectively.
         */
        if ($shipment->purchaseOrderShipmentLines()->count() == 0) {
            $shipment->delete();
        } else {
            $purchaseOrder->shipped($shipment->shipment_date);
        }

        if ($receipt->purchaseOrderShipmentReceiptLines()->count() == 0) {
            $receipt->delete();
        } elseif ($shipment->exists) {
            $shipment->received($receipt->received_at);
        }

        return $receipt;
    }

    protected function receiveLine(
        array $receiptLine,
        PurchaseOrderShipment $shipment,
        PurchaseOrderShipmentReceipt $receipt,
        ?bool $addToInventory = true,
        ?SKUProgressEvent $progress = null
    ): void {
        DB::beginTransaction();

        try {
            // Create shipment and receipt lines.
            /** @var PurchaseOrderShipmentLine $shipmentLine */
            if (! empty($receiptLine['purchase_order_shipment_line_id'])) {
                $shipmentLine = PurchaseOrderShipmentLine::with([])->findOrFail($receiptLine['purchase_order_shipment_line_id']);
                $shipmentLine->update(['quantity' => $receiptLine['quantity']]);
            } else {
                $shipmentLine = $shipment->purchaseOrderShipmentLines()->create([
                    'purchase_order_line_id' => $receiptLine['purchase_order_line_id'],
                    'quantity' => $receiptLine['quantity'],
                ]);
            }

            /** @var PurchaseOrderShipmentReceiptLine $shipmentReceiptLine */
            $shipmentReceiptLine = $receipt->purchaseOrderShipmentReceiptLines()->create([
                'quantity' => $receiptLine['quantity'],
                'purchase_order_line_id' => $shipmentLine->purchase_order_line_id,
                'purchase_order_shipment_line_id' => $shipmentLine->id,
            ]);
            $shipmentLine->purchaseOrderLine->received_quantity = $shipmentLine->purchaseOrderLine->received_quantity + $receiptLine['quantity'];
            $shipmentLine->purchaseOrderLine->save();

            // We add the receipt to inventory if requested
            if ($addToInventory) {
                // First, we get warehouse id from the payload or fall back to the
                // one on the purchase order.
                $warehouseId = $payload['warehouse_id'] ?? $shipment->purchaseOrder->destination_warehouse_id;

                if (! empty($warehouseId)) {
                    if (empty($shipment->purchaseOrder->destination_warehouse_id)) {
                        $shipment->purchaseOrder->destination_warehouse_id = $warehouseId;
                        $shipment->purchaseOrder->save();
                    }

                    $receipt->addLineToInventory($shipmentReceiptLine, $warehouseId);
                }
            }

            DB::commit();
        } catch (Throwable $exception) {
            /**
             * We rollback the transaction and
             * log the error.
             */
            DB::rollBack();
            if ($progress) {
                /** @var PurchaseOrderLine $orderLine */
                $orderLine = PurchaseOrderLine::with([])->find($receiptLine['purchase_order_line_id']);
                if (! $orderLine || ! $orderLine->product_id) {
                    $progress->addError($receiptLine['purchase_order_line_id'], "PO line id: {$receiptLine['purchase_order_line_id']} not found.");

                    return;
                }

                $progress->addError($orderLine->product->sku, "Unable to receive sku: {$orderLine->product->sku}", [
                    'exception' => [
                        'message' => $exception->getMessage(),
                        'file' => $exception->getFile(),
                        'line' => $exception->getLine(),
                    ],
                ]);
            }
        }
    }

    /*
     * MODEL_BYPASS: App\Models\PurchaseOrderShipmentLine
     */
    private function createPurchaseShipmentLines($shipment, $receiptLines)
    {
        return PurchaseOrderShipmentLine::dataFeedBulkImport([
            'data_to_import' => [
                'type' => 'json',
                'data' => json_encode($receiptLines),
            ],
            'insert' => true,
            'update' => false,
            'mappings' => [
                [
                    'expected_column_name' => 'quantity',
                    'data_column_name' => 'quantity',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'purchase_order_line_id',
                    'data_column_name' => 'purchase_order_line_id',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'uid',
                    'data_column_name' => 'uid',
                    'expected_column_type' => 'string',
                ],
            ],
            'default_columns' => [
                [
                    'expected_column_name' => 'purchase_order_shipment_id',
                    'default_value' => $shipment->id,
                    'expected_column_type' => 'integer',
                ],
            ],
            'unique_by_columns' => [
                'purchase_order_line_id',
                'purchase_order_shipment_id',
            ],
        ]);
    }

    private function revertPurchaseShipmentLines($receiptLines, PurchaseOrderShipment $shipment)
    {
        PurchaseOrderShipmentLine::whereIn('uid', $receiptLines->pluck('uid')->values()->all())->delete();
        // We delete the shipment as well.
        $shipment->delete();
    }

    /*
     * MODEL_BYPASS: App\Models\PurchaseOrderShipmentReceiptLine
     */
    private function createPurchaseShipmentReceiptLines($receipt, $receiptLines)
    {
        return PurchaseOrderShipmentReceiptLine::dataFeedBulkImport([
            'data_to_import' => [
                'type' => 'json',
                'data' => json_encode($receiptLines),
            ],
            'insert' => true,
            'update' => false,
            'mappings' => [
                [
                    'expected_column_name' => 'quantity',
                    'data_column_name' => 'quantity',
                    'expected_column_type' => 'integer',
                ],
                [
                    'joins' => [
                        [
                            'table' => (new PurchaseOrderShipmentLine())->getTable(),
                            'type' => 'leftJoin',
                            'on' => [
                                [
                                    'stage_column_to_join' => 'uid',
                                    'database_column' => 'uid',
                                ],
                            ],
                            'column_type' => 'integer',
                            'add_select' => [
                                [
                                    'expected_column_name' => 'purchase_order_shipment_line_id',
                                    'data_column_name' => 'id',
                                    'expected_column_type' => 'integer',
                                    'is_importable' => true,
                                ],

                            ],
                        ],
                    ],
                ],
                [
                    'expected_column_name' => 'uid',
                    'data_column_name' => 'uid',
                    'expected_column_type' => 'string',
                ],
            ],
            'default_columns' => [
                [
                    'expected_column_name' => 'purchase_order_shipment_receipt_id',
                    'default_value' => $receipt->id,
                    'expected_column_type' => 'integer',
                ],
            ],
            'unique_by_columns' => [
                'purchase_order_shipment_line_id',
                'purchase_order_shipment_receipt_id',
            ],
        ]);
    }

    private function revertPurchaseShipmentReceiptLines($receiptLines)
    {
        PurchaseOrderShipmentReceiptLine::whereIn('uid', $receiptLines->pluck('uid')->values()->all())->delete();
    }

    /*
     * MODEL_BYPASS: App\Models\FifoLayer
     */
    private function createFifoLayerLines($purchaseOrder, $receipt, $receiptLines)
    {
        $purhaseOrderLines = PurchaseOrderLine::whereIn('id', collect($receiptLines)->pluck('purchase_order_line_id')->values()->all())
            ->select((new PurchaseOrderLine())->getTable().'.*')
            ->where('purchase_order_lines.purchase_order_id', $purchaseOrder->id)
            ->withCostValues($purchaseOrder->id)
            ->get();

        //Create Fifo layers data
        $fifoLayerLines = collect($receiptLines)->map(function ($line) use ($purhaseOrderLines) {
            $purchaeOrderLine = $purhaseOrderLines->where('id', $line['purchase_order_line_id'])->first();

            return [
                'quantity' => $line['quantity'],
                'product_id' => $purchaeOrderLine->product_id,
                'total_cost' => ($purchaeOrderLine->total_cost / $purchaeOrderLine->quantity) * $line['quantity'],
                'uid' => $line['uid'],
            ];
        })
            ->toArray();

        $tempTable = FifoLayer::dataFeedBulkImport([
            'debug' => false,
            'data_to_import' => [
                'type' => 'json',
                'data' => json_encode($fifoLayerLines),
            ],
            'insert' => true,
            'update' => false,
            'mappings' => [
                [
                    'expected_column_name' => 'original_quantity',
                    'data_column_name' => 'quantity',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'product_id',
                    'data_column_name' => 'product_id',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'total_cost',
                    'data_column_name' => 'total_cost',
                    'expected_column_type' => 'decimal',
                ],
                [
                    'expected_column_name' => 'uid',
                    'data_column_name' => 'uid',
                    'expected_column_type' => 'string',
                ],
                [
                    'joins' => [
                        [
                            'table' => (new PurchaseOrderShipmentReceiptLine())->getTable(),
                            'type' => 'leftJoinSub',
                            'query' => PurchaseOrderShipmentReceiptLine::select('id', 'uid')
                                ->selectRaw('\'App\\\Models\\\PurchaseOrderShipmentReceiptLine\' as link_type'),
                            'on' => [
                                [
                                    'stage_column_to_join' => 'uid',
                                    'database_column' => 'uid',
                                ],
                                [
                                    'stage_column_to_join' => 'link_type',
                                    'database_column' => 'link_type',
                                ],
                            ],
                            'column_type' => 'integer',
                            'add_select' => [
                                [
                                    'expected_column_name' => 'link_id',
                                    'data_column_name' => 'id',
                                    'expected_column_type' => 'integer',
                                    'is_importable' => true,
                                ],

                            ],
                        ],
                    ],
                ],
            ],
            'default_columns' => [
                [
                    'expected_column_name' => 'fulfilled_quantity',
                    'default_value' => 0,
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'warehouse_id',
                    'default_value' => $purchaseOrder->destination_warehouse_id,
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'fifo_layer_date',
                    'default_value' => $receipt->received_at->toDateTimeString(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'link_type',
                    'default_value' => \App\Models\PurchaseOrderShipmentReceiptLine::class,
                    'expected_column_type' => 'string',
                ],
                [
                    'expected_column_name' => 'created_at',
                    'default_value' => now()->toDateTimeString(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'updated_at',
                    'default_value' => now()->toDateTimeString(),
                    'expected_column_type' => 'datetime',
                ],
            ],
            'unique_by_columns' => [
                'link_id',
                'link_type',
            ],
        ]);

        return [
            'data_feed_lines' => $fifoLayerLines,
            'data_feed_table' => $tempTable,
        ];
    }

    private function revertFifoLayerLines($receiptLines)
    {
        FifoLayer::whereIn('uid', $receiptLines->pluck('uid')->values()->all())->delete();
    }

    /*
     * MODEL_BYPASS: App\Models\InventoryMovement
     */
    private function createInventoryMovements($purchaseOrder, $receipt, $fifoLayerLines): void
    {
        InventoryMovement::dataFeedBulkImport([
            'debug' => false,
            'data_to_import' => [
                'type' => 'json',
                'data' => json_encode($fifoLayerLines),
            ],
            'insert' => true,
            'update' => false,
            'mappings' => [
                [
                    'expected_column_name' => 'quantity',
                    'data_column_name' => 'quantity',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'product_id',
                    'data_column_name' => 'product_id',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'uid',
                    'data_column_name' => 'uid',
                    'expected_column_type' => 'string',
                    'is_importable' => false,
                ],
                [
                    'joins' => [
                        [
                            'table' => (new FifoLayer())->getTable(),
                            'type' => 'leftJoinSub',
                            'query' => FifoLayer::select('id', 'uid')
                                ->selectRaw('\'App\\\Models\\\FifoLayer\' as layer_type'),
                            'on' => [
                                [
                                    'stage_column_to_join' => 'uid',
                                    'database_column' => 'uid',
                                ],
                                [
                                    'stage_column_to_join' => 'layer_type',
                                    'database_column' => 'layer_type',
                                ],
                            ],
                            'add_select' => [
                                [
                                    'expected_column_name' => 'layer_id',
                                    'data_column_name' => 'id',
                                    'expected_column_type' => 'integer',
                                    'is_importable' => true,
                                ],

                            ],
                        ],
                    ],
                ],
                [
                    'joins' => [
                        [
                            'table' => (new PurchaseOrderShipmentReceiptLine())->getTable(),
                            'type' => 'leftJoinSub',
                            'query' => PurchaseOrderShipmentReceiptLine::select('id', 'uid')
                                ->selectRaw('\'App\\\Models\\\PurchaseOrderShipmentReceiptLine\' as link_type'),
                            'on' => [
                                [
                                    'stage_column_to_join' => 'uid',
                                    'database_column' => 'uid',
                                ],
                                [
                                    'stage_column_to_join' => 'link_type',
                                    'database_column' => 'link_type',
                                ],
                            ],
                            'add_select' => [
                                [
                                    'expected_column_name' => 'link_id',
                                    'data_column_name' => 'id',
                                    'expected_column_type' => 'integer',
                                    'is_importable' => true,
                                ],

                            ],
                        ],
                    ],
                ],
            ],
            'default_columns' => [
                [
                    'expected_column_name' => 'type',
                    'default_value' => InventoryMovement::TYPE_PURCHASE_RECEIPT,
                    'expected_column_type' => 'string',
                ],
                [
                    'expected_column_name' => 'inventory_status',
                    'default_value' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
                    'expected_column_type' => 'string',
                ],
                [
                    'expected_column_name' => 'warehouse_id',
                    'default_value' => $purchaseOrder->destination_warehouse_id,
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'inventory_movement_date',
                    'default_value' => $receipt->received_at->toDateTimeString(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'warehouse_location_id',
                    'default_value' => Warehouse::findOrFail($purchaseOrder->destination_warehouse_id)->defaultLocation->id ?? null,
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'layer_type',
                    'default_value' => \App\Models\FifoLayer::class,
                    'expected_column_type' => 'string',
                ],
                [
                    'expected_column_name' => 'link_type',
                    'default_value' => \App\Models\PurchaseOrderShipmentReceiptLine::class,
                    'expected_column_type' => 'string',
                ],
                [
                    'expected_column_name' => 'reference',
                    'default_value' => $purchaseOrder->purchase_order_number,
                    'expected_column_type' => 'string',
                ],
                [
                    'expected_column_name' => 'created_at',
                    'default_value' => now()->toDateTimeString(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'updated_at',
                    'default_value' => now()->toDateTimeString(),
                    'expected_column_type' => 'datetime',
                ],
            ],
            'unique_by_columns' => [
                'layer_id',
                'layer_type',
                'link_id',
                'link_type',
            ],
        ]);
    }

    /*
     * MODEL_BYPASS: App\Models\ProductInventory
     */
    private function createProductsInventoryForWarehouse($purchaseOrder, $fifoLayersTable)
    {
        $stockValueQuery = FifoLayer::select('product_id', 'warehouse_id')
            ->selectRaw('SUM(available_quantity * (CASE WHEN original_quantity != 0 THEN (total_cost / original_quantity) ELSE 0 END)) as actual_cost')
            ->where('warehouse_id', $purchaseOrder->destination_warehouse_id)
            ->whereIn('product_id', function ($query) use ($fifoLayersTable) {
                $query->select('product_id')->from($fifoLayersTable);
            })
            ->groupBy(['product_id', 'warehouse_id']);

        $unreceivedQuantity = PurchaseOrderLine::select('product_id', 'purchase_orders.destination_warehouse_id as warehouse_id')
            ->selectRaw('SUM(purchase_order_lines.quantity - received_quantity_subquery.received_quantity) as unreceived_quantity')
            ->joinRelationship('purchaseOrder', function ($query) {
                $query->where('order_status', PurchaseOrder::STATUS_OPEN);
            })
            ->withReceivedQuantity($fifoLayersTable)
            ->groupBy(['product_id', 'purchase_orders.destination_warehouse_id']);

        //Update products inventory for specific warehouse
        $productInventoryQuery = InventoryMovement::withInventoryDetails()
            ->where('warehouse_id', $purchaseOrder->destination_warehouse_id)
            ->groupInventory(['product_id', 'warehouse_id']);

        $productInventoryRecords = DB::table('product_inventory_records')
            ->select('product_inventory_records.*', 'unreceived_quantity_records.*', 'stock_value_records.actual_cost')
            ->withExpression('product_inventory_records', $productInventoryQuery)
            ->withExpression('unreceived_quantity_records', $unreceivedQuantity)
            ->withExpression('stock_value_records', $stockValueQuery)
            ->leftJoin('unreceived_quantity_records', function ($join) {
                $join->on('product_inventory_records.product_id', '=', 'unreceived_quantity_records.product_id')
                    ->on('product_inventory_records.warehouse_id', '=', 'unreceived_quantity_records.warehouse_id');
            })
            ->leftJoin('stock_value_records', function ($join) {
                $join->on('product_inventory_records.product_id', '=', 'stock_value_records.product_id')
                    ->on('product_inventory_records.warehouse_id', '=', 'stock_value_records.warehouse_id');
            })
            ->get()
            ->map(function ($record) {
                $record = (array) $record;
                $record['inventory_available'] = $record['in_warehouse_quantity'] - $record['in_warehouse_reserved_quantity'];
                $record['inventory_incoming'] = max(0, $record['unreceived_quantity']);
                $record['inventory_stock_value'] = $record['actual_cost'];

                return $record;
            });

        //Update product inventory
        return ProductInventory::dataFeedBulkImport([
            'debug' => false,
            'data_to_import' => [
                'type' => 'json',
                'data' => json_encode($productInventoryRecords),
            ],
            'insert' => false,
            'update' => true,
            'mappings' => [
                [
                    'expected_column_name' => 'product_id',
                    'data_column_name' => 'product_id',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'warehouse_id',
                    'data_column_name' => 'warehouse_id',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'inventory_total',
                    'data_column_name' => 'in_warehouse_quantity',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'inventory_reserved',
                    'data_column_name' => 'in_warehouse_reserved_quantity',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'inventory_available',
                    'data_column_name' => 'inventory_available',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'inventory_in_transit',
                    'data_column_name' => 'in_warehouse_transit_quantity',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'inventory_stock_value',
                    'data_column_name' => 'inventory_stock_value',
                    'expected_column_type' => 'decimal',
                ],
            ],
            'default_columns' => [
                [
                    'expected_column_name' => 'created_at',
                    'default_value' => now()->toDateTimeString(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'updated_at',
                    'default_value' => now()->toDateTimeString(),
                    'expected_column_type' => 'datetime',
                ],
            ],
            'unique_by_columns' => [
                'product_id',
                'warehouse_id',
            ],
        ]);
    }

    /*
     * MODEL_BYPASS: App\Models\ProductInventory
     */
    private function createProductsInventoryForAllWarehouses($fifoLayersTable)
    {
        $stockValueQuery = FifoLayer::select('product_id')
            ->selectRaw('SUM((available_quantity) * (CASE WHEN original_quantity != 0 THEN (total_cost / original_quantity) ELSE 0 END)) as actual_cost')
            // ->where('warehouse_id', $purchaseOrder->destination_warehouse_id)
            ->whereIn('product_id', function ($query) use ($fifoLayersTable) {
                $query->select('product_id')->from($fifoLayersTable);
            })
            ->groupBy(['product_id']);

        $unreceivedQuantity = PurchaseOrderLine::select('product_id')
            ->selectRaw('SUM(purchase_order_lines.quantity - received_quantity_subquery.received_quantity) as unreceived_quantity')
            ->joinRelationship('purchaseOrder', function ($query) {
                $query->where('order_status', PurchaseOrder::STATUS_OPEN);
            })
            ->withReceivedQuantity($fifoLayersTable)
            ->groupBy(['product_id']);

        //Update products inventory for specific warehouse
        $productInventoryQuery = InventoryMovement::withInventoryDetails()
            ->groupInventory(['product_id']);

        $productInventoryRecords = DB::table('product_inventory_records')
            ->select('product_inventory_records.*', 'unreceived_quantity_records.*', 'stock_value_records.actual_cost')
            ->withExpression('product_inventory_records', $productInventoryQuery)
            ->withExpression('unreceived_quantity_records', $unreceivedQuantity)
            ->withExpression('stock_value_records', $stockValueQuery)
            ->leftJoin('unreceived_quantity_records', function ($join) {
                $join->on('product_inventory_records.product_id', '=', 'unreceived_quantity_records.product_id');
            })
            ->leftJoin('stock_value_records', function ($join) {
                $join->on('product_inventory_records.product_id', '=', 'stock_value_records.product_id');
            })
            ->get()
            ->map(function ($record) {
                $record = (array) $record;
                $record['inventory_available'] = $record['in_warehouse_quantity'] - $record['in_warehouse_reserved_quantity'];
                $record['inventory_stock_value'] = $record['actual_cost'];

                return $record;
            });

        //Update product inventory
        return ProductInventory::dataFeedBulkImport([
            'debug' => false,
            'data_to_import' => [
                'type' => 'json',
                'data' => json_encode($productInventoryRecords),
            ],
            'insert' => false,
            'update' => true,
            'mappings' => [
                [
                    'expected_column_name' => 'product_id',
                    'data_column_name' => 'product_id',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'inventory_total',
                    'data_column_name' => 'in_warehouse_quantity',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'inventory_reserved',
                    'data_column_name' => 'in_warehouse_reserved_quantity',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'inventory_available',
                    'data_column_name' => 'inventory_available',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'inventory_in_transit',
                    'data_column_name' => 'in_warehouse_transit_quantity',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'inventory_stock_value',
                    'data_column_name' => 'inventory_stock_value',
                    'expected_column_type' => 'decimal',
                ],
            ],
            'default_columns' => [
                [
                    'expected_column_name' => 'created_at',
                    'default_value' => now()->toDateTimeString(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'updated_at',
                    'default_value' => now()->toDateTimeString(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'warehouse_id',
                    'default_value' => 0,
                    'expected_column_type' => 'integer',
                ],
            ],
            'unique_by_columns' => [
                'product_id',
                'warehouse_id',
            ],
        ]);
    }

    private function updateProductListingQuantity($tempTable)
    {
        ProductListing::whereIn('product_id', function ($query) use ($tempTable) {
            $query->select('product_id')->from($tempTable);
        })
            ->leftJoinRelationship('salesChannel.integrationInstance.integration')
            ->select('integration_instances.id')
            ->distinct()
            ->get()
            ->each(function ($record) use ($tempTable) {
                /** @var IntegrationInstance $integrationInstance */
                $integrationInstance = IntegrationInstance::query()->findOrFail($record->id);

                $masterOfStock = $integrationInstance->getMasterOfStock();
                $inventoryRules = json_encode($integrationInstance->getInventoryData()->inventoryModificationRules);

                if ($masterOfStock == ProductListing::MASTER_SKU) {
                    ProductListing::forProductIds($tempTable)
                        ->forProductInventoryRules($integrationInstance)
                        ->where('sales_channel_id', $integrationInstance->salesChannel->id)
                        ->whereNull('master_of_stock')
                          /* @see \CreateGetProductListingQuantityFunction Database Function */
                        ->update(['quantity' => DB::raw("getProductListingQuantity(`inventory_rules`, '$inventoryRules', `pi`.`inventory_available`)")]);
                } else {
                    ProductListing::forProductIds($tempTable)
                        ->where('sales_channel_id', $integrationInstance->salesChannel->id)
                        ->where(function (Builder $builder) {
                            $builder->whereNull('master_of_stock')
                                ->orWhere('master_of_stock', '!=', ProductListing::MASTER_SKU);
                        })
                        ->update(['quantity' => null]);
                }
            });
    }

    /**
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     */
    public function receiveLinesWithDataFeed(
        array $receiptLines,
        array $payload,
        ?bool $addToInventory = true,
        ?PurchaseOrderShipment $shipment = null,
        ?int $userId = null,
    ): PurchaseOrderShipmentReceipt {
        // Aggregate the purchase order receipt lines
        $receiptLines = PurchaseOrderShipmentReceipt::aggregateReceiptLines($receiptLines);

        // We get the purchase order from the purchase order lines
        $purchaseOrder = $this->validateOrderFromReceiptLines($receiptLines);

        // Next, we get the purchase order shipment. If none is provided,
        // we create a new shipment for the receipt lines
        if (! $shipment) {
            $shipment = $this->getShipmentForReceipts($purchaseOrder, $payload);
        }

        /*
         * Receipt lines don't belong in PurchaseOrderShipmentReceipt model.  This does not cause an error in normal
         * operation, but it does cause an error when using factories which bypass the fillable property of the model.
         * For this reason we need to unset the receipt lines from the payload here.
         */
        unset($payload['receipt_lines']);
        // Create Purchase Order Shipment Receipt
        /** @var PurchaseOrderShipmentReceipt $receipt */
        $receipt = $shipment->purchaseOrderShipmentReceipts()->create(array_merge($payload, [
            'user_id' => $userId ?? auth()->user()?->id,
        ]));

        /**
         * We create a shipment and receipt line for each
         * of the receipt lines and receive it into inventory.
         */
        $progress = SKUProgressEvent::sendOn(
            $purchaseOrder->getReceiptBroadcastChannel(),
            6
        );

        //Add UUID for reference
        $receiptLines = collect($receiptLines)->map(function ($line) {
            $line['uid'] = (string) Str::uuid();

            return $line;
        });
        Config::set('database.connections.mysql.strict', false);

        //Shipment Lines
        try {
            $this->createPurchaseShipmentLines($shipment, $receiptLines->toArray());
            $progress->advance();
        } catch (Throwable $e) {
            Log::debug($e->getMessage());
            Log::debug($e->getLine());
            Log::debug($e->getFile());
            throw new ReceivePurchaseOrderLineException('Issue creating purchase shipment lines. '.$e->getMessage(), 1);
        }

        //Shipment Receipt Lines
        try {
            $this->createPurchaseShipmentReceiptLines($receipt, $receiptLines->toArray());
            $progress->advance();
        } catch (Throwable $e) {
            $this->revertPurchaseShipmentLines($receiptLines, $shipment);

            throw new ReceivePurchaseOrderLineException('Issue creating purchase shipment receipt lines.');
        }

        //Fifo layer
        try {
            $fifoLayers = $this->createFifoLayerLines($purchaseOrder, $receipt, $receiptLines->toArray());
            $progress->advance();
        } catch (Throwable $e) {
            $this->revertPurchaseShipmentReceiptLines($receiptLines);
            $this->revertPurchaseShipmentLines($receiptLines, $shipment);

            throw new ReceivePurchaseOrderLineException('Issue creating fifo layers. '.$e->getMessage(), 1);
        }

        try {
            $fifoLayersTable = $fifoLayers['data_feed_table'];
            $productDates = array_fill_keys(
                collect($fifoLayers['data_feed_lines'])->pluck('product_id')->toArray(),
                Helpers::utcStartOfLocalDate($receipt->received_at)
            );
            $this->createInventoryMovements($purchaseOrder, $receipt, $fifoLayers['data_feed_lines']);
            // InventorySnapshot is not currently in use
            //$this->inventorySnapshotRepository->invalidateProductDates($productDates);
            $progress->advance();
        } catch (Throwable $e) {
            $this->revertFifoLayerLines($receiptLines);
            $this->revertPurchaseShipmentReceiptLines($receiptLines);
            $this->revertPurchaseShipmentLines($receiptLines, $shipment);

            throw new ReceivePurchaseOrderLineException('Issue creating inventory movements. '.$e->getMessage(), 1);
        }

        try {
            //Create products inventory
            $tempTable = $this->createProductsInventoryForWarehouse($purchaseOrder, $fifoLayersTable);
            $progress->advance();

            $tempTable = $this->createProductsInventoryForAllWarehouses($fifoLayersTable);
            $this->updateProductListingQuantity($tempTable); //SKU-4850

            $progress->advance();
        } catch (Throwable $e) {
            Artisan::call('sku:inventory:refresh', ['--products' => PurchaseOrderLine::select('product_id')->where('purchase_order_id', $purchaseOrder->id)->get()->pluck('product_id')->all()]);
        }

        Config::set('database.connections.mysql.strict', true);
        DB::purge('mysql');

        ReleaseBackorderQueuesJob::dispatch(collect($fifoLayers['data_feed_lines'])->pluck('product_id')->values()->all());

        /**
         * In case there are no successful shipment lines
         * or receipt lines, we remove the shipment and
         * receipt respectively.
         */
        if ($shipment->purchaseOrderShipmentLines()->count() == 0) {
            $shipment->delete();
        } else {
            $purchaseOrder->shipped($shipment->shipment_date);
        }

        if ($receipt->purchaseOrderShipmentReceiptLines()->count() == 0) {
            $receipt->delete();
        } elseif ($shipment->exists) {
            $shipment->received($receipt->received_at);
        }

        return $receipt;
    }

    /**
     * @throws Throwable
     */
    public function newReceiveLines(
        array $lines,
        array $payload,
        ?PurchaseOrderShipment $shipment = null,
        ?int $userId = null,
    ): PurchaseOrderShipmentReceipt {
        $productsUpdated = [];

        /** @var PurchaseOrderShipmentReceipt $receipt */
        $receipt = DB::transaction(function () use ($lines, $payload, $shipment, $userId, &$productsUpdated) {
            try {
                $purchaseOrder = $this->validateOrderFromReceiptLines($lines);
                customlog('receiving', 'Receiving PO '.$purchaseOrder->purchase_order_number);
                $shipment = $this->purchaseOrderShipmentRepository->findOrCreate($purchaseOrder, @$payload['purchase_order_shipment_id']);

                $receipt = $this->createReceipt($shipment, $payload, $userId);

                $backorderQueueCollection = collect();
                $linesChunks = array_chunk($lines, 100);
                $progress = SKUProgressEvent::sendOn(
                    $shipment->purchaseOrder->getReceiptBroadcastChannel(),
                    count($linesChunks)
                );

                foreach ($linesChunks as $linesChunk) {
                    $this->handleLinesChunk($linesChunk, $receipt, $shipment, $purchaseOrder, $productsUpdated, $backorderQueueCollection);
                    $progress->advance();
                }

                $receipt->load('purchaseOrderShipmentReceiptLines');

                if ($receipt->purchaseOrderShipmentReceiptLines()->count() == 0) {
                    customlog('receiving', 'Deleting receipt because there are no receipt lines');
                    $receipt->delete();
                } elseif ($shipment->exists) {
                    customlog('receiving', 'Marking shipment as received');
                    $shipment->received($receipt->received_at);
                }

                customlog('receiving', 'Automating fulfillment for backorders');
                /*
                 * Automate fulfillment, outside chunking loop, but within transaction
                 * Outside chunking loop because the results may differ if a sales order only has some lines that got released
                 * Inside transaction because it is an important part of the workflow
                 */
                try {
                    $this->automateFulfillmentForBackorders($backorderQueueCollection);
                } catch (Throwable $e) {
                    if (config('app.env') != 'testing') {
                        dd($e->getMessage());
                    }
                }
                customlog('receiving', 'Finished automating fulfillment for backorders');
            } catch (Throwable $e) {
                dd($e);
            }

            return $receipt;
        }, 5);

        customlog('receiving', 'Updating product listing cache');
        // Product listing cache update can be outside transaction
        $this->updateProductListingCache($productsUpdated);

        $productIdsUpdated = array_map(function (Product $product) {
            return $product->id;
        }, $productsUpdated);

        dispatch_sync(new UpdateProductsInventoryAndAvgCost($productIdsUpdated));
        customlog('receiving', 'Updating product listing cache completed');

        return $receipt;
    }

    private function automateFulfillmentForBackorders(Collection $backorderQueueCollection): void
    {
        $uniqueSalesOrders = $backorderQueueCollection->flatMap(function ($backorderQueue) {
            return $backorderQueue->salesOrderLine ? [$backorderQueue->salesOrderLine->salesOrder] : [];
        })->unique('id');

        $uniqueSalesOrders->each(/**
         * @throws Throwable
         */ function (SalesOrder $salesOrder) {
            dispatch(new AutomatedSalesOrderFulfillmentJob($salesOrder))->onQueue('automatedFulfillments');
        });
    }

    private function createReceipt(PurchaseOrderShipment $shipment, array $payload, ?int $userId): PurchaseOrderShipmentReceipt
    {
        return $this->purchaseOrderShipmentReceiptRepository->save(PurchaseOrderShipmentReceiptDto::from([
            'purchase_order_shipment_id' => $shipment->id,
            'user_id' => $userId ?? auth()->user()?->id,
            'received_at' => $payload['received_at'],
        ]));
    }

    public function handleLinesChunk($linesChunk, $receipt, $shipment, $purchaseOrder, &$productsUpdated, Collection &$backorderQueueCollection): void
    {
        customlog('receiving', 'handleLinesChunk - Main logic');
        $purchaseOrderLineId = $linesChunk[0]['purchase_order_line_id'];
        $existingShipmentLines = PurchaseOrderShipmentLine::where('purchase_order_line_id', $purchaseOrderLineId)
            ->where('purchase_order_shipment_id', $shipment->id)
            ->get();
        if ($existingShipmentLines->count() > 0) {
            $newShipmentLines = $existingShipmentLines;
        } else {
            // Prepare the shipment lines and insert them
            $shipmentLines = $this->prepareShipmentLines($linesChunk, $shipment);
            PurchaseOrderShipmentLine::insert($shipmentLines);
            customlog('receiving', count($shipmentLines).' shipment lines inserted');

            // Retrieve the newly inserted shipment lines
            $newShipmentLines = $this->purchaseOrderShipmentLineRepository->getShipmentLinesFromPurchaseOrderLineIds($shipment->id, array_column($shipmentLines, 'purchase_order_line_id'));
        }

        // Prepare the receipt lines and insert them
        $receiptLineCollection = $this->getReceiptLinesCollection($newShipmentLines, $receipt);
        PurchaseOrderShipmentReceiptLine::insert($receiptLineCollection->toArray());
        customlog('receiving', count($receiptLineCollection).' shipment receipt lines inserted');

        // Retrieve the newly inserted receipt lines
        $newShipmentReceiptLines = $this->purchaseOrderShipmentReceiptLineRepository->getReceiptLinesFromShipmentLineIds($receipt->id, $receiptLineCollection->pluck('purchase_order_shipment_line_id')->toArray());

        // Retrieve the purchase order lines with cost values
        $purchaseOrderLines = $this->purchaseOrderLineRepository->getPurchaseOrderLinesWithCostFromIds($shipment->purchaseOrder, $newShipmentLines->pluck('purchase_order_line_id')->toArray());

        // Prepare the purchase order lines
        $purchaseOrderLineCollection = $this->getPurchaseOrderLinesCollection($newShipmentReceiptLines, $purchaseOrderLines);

        // Product Inventory Data for updating cache
        $productInventoryData = [];

        $fifoLayerCollection = $this->getFifoLayerCollection($newShipmentReceiptLines, $purchaseOrderLines, $receipt, $purchaseOrder, $productsUpdated, $productInventoryData);
        FifoLayer::insert($fifoLayerCollection->toArray());
        customlog('receiving', count($fifoLayerCollection).' fifo layers inserted');

        $newFifoLayers = $this->fifoLayerRepository->getFifoLayersForTypeAndIds(PurchaseOrderShipmentReceiptLine::class, $fifoLayerCollection->pluck('link_id')->toArray());

        $inventoryMovementCollection = $this->getInventoryMovementCollection($newShipmentReceiptLines, $newFifoLayers, $purchaseOrder);
        InventoryMovement::insert($inventoryMovementCollection->toArray());
        customlog('receiving', count($inventoryMovementCollection).' inventory movements inserted');

        customlog('receiving', 'handleLinesChunk - updateProductsInventory');
        $this->updateProductsInventory($productInventoryData);

        customlog('receiving', 'handleLinesChunk - releaseBackorderQueues');
        // Release backorder queues
        $backorderQueueCollection = $backorderQueueCollection->merge((new BackorderManager())->releaseBackorderQueues($newShipmentReceiptLines));
        customlog('receiving', 'handleLinesChunk - updating PO Lines');
        batch()->update(new PurchaseOrderLine(), $purchaseOrderLineCollection->toArray(), 'id');
        customlog('receiving', 'handleLinesChunk - updating PO Lines completed');
    }

    private function updateProductsInventory(array $productInventoryData): void
    {
        $tmpTableName = 'tmp_products_inventory_'.\Illuminate\Support\Str::random(10);

        // Create temporary table and copy structure of products_inventory
        DB::statement("CREATE TEMPORARY TABLE $tmpTableName LIKE products_inventory;");

        // Insert data into temporary table
        DB::table($tmpTableName)->insert($productInventoryData);

        // Prepare update query
        $query = <<<SQL
        UPDATE products_inventory as pi
        INNER JOIN $tmpTableName as tmp_products_inventory
            ON pi.product_id = tmp_products_inventory.product_id
            AND pi.warehouse_id = tmp_products_inventory.warehouse_id
        SET 
            pi.inventory_total = pi.inventory_total + tmp_products_inventory.inventory_total,
            pi.inventory_available = pi.inventory_available + tmp_products_inventory.inventory_available,
            pi.inventory_stock_value = pi.inventory_stock_value + tmp_products_inventory.inventory_stock_value
        SQL;

        // Execute update query
        DB::statement($query);

        // Perform select of the missing records
        $recordsToInsert = DB::table($tmpTableName)
            ->select('product_id', 'warehouse_id', 'inventory_total', 'inventory_reserved', 'inventory_in_transit', 'inventory_available', 'inventory_stock_value', 'created_at', 'updated_at')
            ->whereRaw('(product_id, warehouse_id) NOT IN (SELECT product_id, warehouse_id FROM products_inventory)')
            ->get();

        $dataToInsert = $recordsToInsert->map(function ($item) {
            return (array) $item;
        })->toArray();

        ProductInventory::insert($dataToInsert);
    }

    private function prepareShipmentLines($linesChunk, $shipment): array
    {
        return array_map(function ($line) use ($shipment) {
            $line['purchase_order_shipment_id'] = $shipment->id;
            $line['created_at'] = Carbon::now();
            $line['updated_at'] = Carbon::now();

            return $line;
        }, $linesChunk);
    }

    private function getReceiptLinesCollection($newShipmentLines, $receipt): Collection
    {
        $receiptLinesCollection = new Collection();

        $newShipmentLines->each(function (PurchaseOrderShipmentLine $purchaseOrderShipmentLine) use ($receipt, &$receiptLinesCollection) {
            $receiptLinesCollection->add(PurchaseOrderShipmentReceiptLineDto::from([
                'purchase_order_shipment_receipt_id' => $receipt->id,
                'purchase_order_shipment_line_id' => $purchaseOrderShipmentLine->id,
                'purchase_order_line_id' => $purchaseOrderShipmentLine->purchase_order_line_id,
                'quantity' => $purchaseOrderShipmentLine->quantity,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]));
        });

        return $receiptLinesCollection;
    }

    private function getPurchaseOrderLinesCollection(Collection $newShipmentReceiptLines, Collection $purchaseOrderLines): Collection
    {
        $purchaseOrderLinesCollection = new Collection();

        $purchaseOrderLines->each(function (PurchaseOrderLine $purchaseOrderLine) use ($newShipmentReceiptLines, &$purchaseOrderLinesCollection) {
            $quantityReceived = $newShipmentReceiptLines->where('purchase_order_line_id', $purchaseOrderLine->id)->first()->quantity;
            $purchaseOrderLinesCollection->add(PurchaseOrderLineDto::from([
                'id' => $purchaseOrderLine->id,
                'received_quantity' => $quantityReceived + $purchaseOrderLine->received_quantity,
                'updated_at' => Carbon::now(),
            ]));
        });

        return $purchaseOrderLinesCollection;
    }

    private function getFifoLayerCollection($newShipmentReceiptLines, $purchaseOrderLines, $receipt, $purchaseOrder, &$productsUpdated, &$productInventoryData): Collection
    {
        $fifoLayerCollection = new Collection();

        $newShipmentReceiptLines->each(function (PurchaseOrderShipmentReceiptLine $purchaseOrderShipmentReceiptLine) use ($purchaseOrderLines, &$fifoLayerCollection, &$productsUpdated, $receipt, $purchaseOrder, &$productInventoryData) {
            $purchaseOrderLine = $purchaseOrderLines->where('id', $purchaseOrderShipmentReceiptLine->purchaseOrderShipmentLine->purchase_order_line_id)->first();
            $totalCost = ($purchaseOrderLine->total_cost / $purchaseOrderLine->quantity) * $purchaseOrderShipmentReceiptLine->quantity;

            $fifoLayerCollection->add(FifoLayerDto::from([
                'fifo_layer_date' => $receipt->received_at->toDateTimeString(),
                'warehouse_id' => $purchaseOrder->destination_warehouse_id,
                'original_quantity' => $purchaseOrderShipmentReceiptLine->quantity,
                'product_id' => $purchaseOrderLine->product_id,
                'total_cost' => $totalCost,
                'link_id' => $purchaseOrderShipmentReceiptLine->id,
                'link_type' => PurchaseOrderShipmentReceiptLine::class,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]));

            // Data for updated products
            $productsUpdated[] = $purchaseOrderLine->product;
            $this->tallyForProductInventory($productInventoryData, $purchaseOrderLine, $purchaseOrder, $purchaseOrderShipmentReceiptLine, $totalCost);
        });

        return $fifoLayerCollection;
    }

    private function tallyForProductInventory(?array &$productInventoryData, PurchaseOrderLine $purchaseOrderLine, $purchaseOrder, PurchaseOrderShipmentReceiptLine $purchaseOrderShipmentReceiptLine, float $totalCost): void
    {
        $warehouseIds = [$purchaseOrder->destination_warehouse_id, 0];

        foreach ($warehouseIds as $warehouseId) {
            $key = $purchaseOrderLine->product_id.'_'.$warehouseId;

            if (! isset($productInventoryData[$key])) {
                $productInventoryData[$key] = [
                    'product_id' => $purchaseOrderLine->product_id,
                    'warehouse_id' => $warehouseId,
                    'inventory_total' => 0,
                    'inventory_available' => 0,
                    'inventory_stock_value' => 0,
                ];
            }

            $productInventoryData[$key]['inventory_total'] += $purchaseOrderShipmentReceiptLine->quantity;
            $productInventoryData[$key]['inventory_available'] += $purchaseOrderShipmentReceiptLine->quantity;
            $productInventoryData[$key]['inventory_stock_value'] += $totalCost;
        }
    }

    private function getInventoryMovementCollection($newShipmentReceiptLines, $newFifoLayers, $purchaseOrder): Collection
    {
        $inventoryMovementCollection = new Collection();

        $newShipmentReceiptLines->each(function ($line) use ($newFifoLayers, &$inventoryMovementCollection, $purchaseOrder) {
            $fifoLayer = $newFifoLayers->where('link_id', $line->id)->where('link_type', PurchaseOrderShipmentReceiptLine::class)->first();
            $inventoryMovementCollection->add(InventoryMovementDto::from([
                'inventory_movement_date' => $fifoLayer->fifo_layer_date,
                'product_id' => $fifoLayer->product_id,
                'quantity' => $fifoLayer->original_quantity,
                'type' => InventoryMovement::TYPE_PURCHASE_RECEIPT,
                'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
                'warehouse_id' => $fifoLayer->warehouse_id,
                'warehouse_location_id' => $fifoLayer->warehouse->defaultLocation?->id,
                'link_id' => $line->id,
                'link_type' => PurchaseOrderShipmentReceiptLine::class,
                'layer_id' => $fifoLayer->id,
                'layer_type' => FifoLayer::class,
                'reference' => $purchaseOrder->purchase_order_number,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]));
        });

        return $inventoryMovementCollection;
    }

    private function updateProductListingCache($productsUpdated): void
    {
        customlog('syncInventory', 'Receiving: Running updateProductListingCache for '.count($productsUpdated).' products');
        /*
         * Product listing cache update can be outside transaction since we only ever update the sales channel every 15 minutes
         * from the cached data anyway
         */
        $productListingsData = [];
        foreach ($productsUpdated as $product) {
            $product->productListings->each(function (ProductListing $productListing) use (&$productListingsData, $product) {
                if (! isset($productListingsData[$productListing->salesChannel->integration_instance_id])) {
                    $productListingsData[$productListing->salesChannel->integration_instance_id] = [];
                }
                $productListingsData[$productListing->salesChannel->integration_instance_id][] = $product->id;
            });
        }
        customlog('syncInventory', 'Receiving: There are '.count($productListingsData).' integration instances to update product listing cache for');
        foreach ($productListingsData as $integrationInstanceId => $productIds) {
            customlog('syncInventory', 'Receiving: Dispatching CacheProductListingQuantityJob for integration instance '.$integrationInstanceId.' with '.count($productIds).' products');
            dispatch(new GenerateCacheProductListingQuantityJob(IntegrationInstance::find($integrationInstanceId), $productIds));
        }
    }

    /**
     * @throws NotOpenPurchaseOrderException
     */
    protected function handleUnexpectedItemsInReceipt(array $productLines, array $payload)
    {
        // We create adjustments for receipts with adjustment action
        $forAdjustments = array_filter($productLines, function ($line) {
            return ! isset($line['action']) || $line['action'] === UNEXPECTED_RECEIPT_ACTION_CREATE_ADJUSTMENT;
        });
        if (! empty($forAdjustments)) {
            $this->createAdjustmentsForUnexpectedReceipts($forAdjustments, $payload);
        }

        $addToOrder = array_filter($productLines, function ($line) {
            return isset($line['action']) && $line['action'] === UNEXPECTED_RECEIPT_ACTION_ADD_TO_PO;
        });

        if (! empty($addToOrder)) {
            $this->addLinesToPOAndReceive($addToOrder, $payload);
        }
    }

    /**
     * @throws NotOpenPurchaseOrderException
     */
    protected function addLinesToPOAndReceive(array $productLines, array $payload)
    {
        $purchaseOrder = $this->getPurchaseOrderFromPayload($payload);
        $purchaseOrderLines = $purchaseOrder->setPurchaseOrderLines(
            collect($productLines)->map(function ($line) {
                $product = $this->getProductForUnexpectedItem($line);

                return [
                    'product_id' => $product->id,
                    'quantity' => $line['quantity'],
                    'amount' => $line['amount'] ?? $product->average_cost ?: 0,
                    'description' => $line['description'] ?? $product->name,
                ];
            })->toArray(),
            false
        );

        // Receive the lines
        $this->newReceiveLines($purchaseOrderLines, $payload);
    }

    protected function createAdjustmentsForUnexpectedReceipts(array $productLines, array $payload)
    {
        $purchaseOrder = $this->getPurchaseOrderFromPayload($payload);
        collect($productLines)->each(function ($line) use ($purchaseOrder) {
            $product = $this->getProductForUnexpectedItem($line);
            $this->handleAdjustment([
                'adjustment_date' => now(),
                'product_id' => $product->id,
                'warehouse_id' => $purchaseOrder->destination_warehouse_id,
                'quantity' => $line['quantity'],
                'notes' => 'Unexpected items in Purchase Order receipt.',
                'unit_cost' => $product->average_cost ?? 0,
                'link_type' => PurchaseOrder::class,
                'link_id' => $purchaseOrder->id,
            ]);
        });
    }

    /**
     * @throws BindingResolutionException
     */
    protected function handleAdjustment(array $adjustment)
    {
        $adjustment = new InventoryAdjustment($adjustment);
        $adjustment->save();

        // Add fifo layer
        $fifoLayer = new FifoLayer();
        $fifoLayer->fifo_layer_date = $adjustment->adjustment_date;
        $fifoLayer->product_id = $adjustment->product_id;
        $fifoLayer->original_quantity = $adjustment->quantity;
        $fifoLayer->total_cost = $adjustment->quantity * $adjustment->unit_cost;
        $fifoLayer->warehouse_id = $adjustment->warehouse_id;
        $adjustment->fifoLayers()->save($fifoLayer);

        // Add inventory movement
        $inventoryMovement = new InventoryMovement();

        $inventoryMovement->inventory_movement_date = $adjustment->adjustment_date;
        $inventoryMovement->product_id = $adjustment->product_id;
        $inventoryMovement->quantity = $adjustment->quantity;
        $inventoryMovement->type = InventoryMovement::TYPE_ADJUSTMENT;
        $inventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
        $inventoryMovement->warehouse_id = $adjustment->warehouse_id;
        $inventoryMovement->fifo_layer = $fifoLayer->id;
        $adjustment->inventoryMovements()->save($inventoryMovement);

        // Attempt releasing backorder layers if any
        (new ReleaseBackorderQueues($adjustment->product))->execute($fifoLayer);
    }

    protected function getPurchaseOrderFromPayload(array $payload)
    {
        if (! empty($payload['purchase_order_id'])) {
            return PurchaseOrder::with([])->findOrFail($payload['purchase_order_id']);
        }

        if (! empty($payload['purchase_order_shipment_id'])) {
            /** @var PurchaseOrderShipment $shipment */
            $shipment = PurchaseOrderShipment::with([])
                ->findOrFail($payload['purchase_order_shipment_id']);

            return $shipment->purchaseOrder;
        }

        if (! empty($payload['receipt_lines'])) {
            $receiptLine = collect($payload['receipt_lines'])
                ->whereNotNull('purchase_order_line_id')
                ->first();
            if ($receiptLine) {
                /** @var PurchaseOrderLine $firstLine */
                $firstLine = PurchaseOrderLine::with([])->findOrFail($receiptLine['purchase_order_line_id']);

                return $firstLine->purchaseOrder;
            }

            $receiptLine = collect($payload['receipt_lines'])
                ->whereNotNull('purchase_order_shipment_line_id')
                ->first();

            if ($receiptLine) {
                /** @var PurchaseOrderShipmentLine $shipmentLine */
                $shipmentLine = PurchaseOrderShipmentLine::with([])->findOrFail($receiptLine['purchase_order_shipment_line_id']);

                return $shipmentLine->purchaseOrderLine->purchaseOrder;
            }
        }

        throw new InvalidArgumentException('Could not find purchase order from receipt payload.');
    }

    /**
     * @throws InvalidArgumentException|Throwable
     */
    public function updateShipmentReceipt(PurchaseOrderShipmentReceipt $receipt, array $payload): PurchaseOrderShipmentReceipt
    {
        return DB::transaction(function () use ($receipt, $payload) {
            // Update the receipt data
            $receipt->fill($payload);
            $receipt->save();

            // Aggregate the purchase order receipt lines
            $receiptLines = PurchaseOrderShipmentReceipt::aggregateReceiptLines($payload['receipt_lines']);
            // Get the purchase order from the purchase order receipt
            $purchaseOrder = $receipt->purchaseOrderShipment->purchaseOrder;
            $this->checkPurchaseOrderStatus($purchaseOrder);

            // We handle three scenarios: removed lines, additional lines and kept lines
            // Get the existing receipt lines on the receipt.
            // Aggregating the lines binds the purchase order line ids
            $existingLines = PurchaseOrderShipmentReceipt::aggregateReceiptLines($receipt->purchaseOrderShipmentReceiptLines->toArray());
            $existingLineIds = $this->extractPurchaseOrderLineIds($existingLines);

            // Get the lines in the update
            $updateLineIds = $this->extractPurchaseOrderLineIds($receiptLines);

            // We get the kept, additional and removed line ids
            $keptLineIds = array_intersect($existingLineIds, $updateLineIds);

            $removedLIneIds = array_diff($existingLineIds, $keptLineIds);

            // We handle removed lines
            $this->handleRemovedLinesFromReceipt($receipt, $removedLIneIds);

            // Next, we handle additional lines
            $additionalLineIds = array_diff($updateLineIds, $keptLineIds);
            $warehouseId = $payload['warehouse_id'] ?? $purchaseOrder->destination_warehouse_id;
            $this->handleAdditionalReceiptLines($receipt, $additionalLineIds, $receiptLines, $warehouseId);

            // We now handle updated lines
            $this->modifyShipmentReceiptQuantityForLines($receipt, $keptLineIds, $receiptLines);

            // Now, we set the receipt status of the shipment
            $receipt->purchaseOrderShipment->received($receipt->received_at);

            // Update inbound cache.
            (new UpdateProductsInventoryAndAvgCost(
                PurchaseOrderLine::query()
                    ->whereIn('id', $additionalLineIds + $removedLIneIds + $keptLineIds)
                    ->pluck('product_id')
                    ->all()
            ))->handle();

            return $receipt;
        });
    }

    public function deleteReceipt(PurchaseOrderShipmentReceipt $receipt, array $receiptLines = []): ?bool
    {
        /**
         * When receipt lines are provided, it means
         * the user wants to delete only those receipt
         * lines, so we return the other lines.
         */
        if (empty($receiptLines)) {
            $receipt->delete();
        } else {
            (new BulkInventoryManager())->bulkDeletePositiveInventoryEvents($receipt->purchaseOrderShipmentReceiptLines()
                ->whereIn('id', $receiptLines['ids'])
                ->get());
        }

        $receipt->purchaseOrderShipment?->purchaseOrder?->refreshReceiptStatus();

        return true;
    }

    /**
     * @param  Arrayable|mixed  $receiptLines
     *
     * @throws InvalidArgumentException
     */
    private function modifyShipmentReceiptQuantityForLines(PurchaseOrderShipmentReceipt $receipt, array $keptLineIds, $receiptLines)
    {
        $keptLines = $this->buildReceiptLinesByPurchaseOrderLines($receipt, $keptLineIds);
        $keptLines->each(function (PurchaseOrderShipmentReceiptLine $receiptLine) use ($receiptLines) {
            // For dropship purchase receipts, updates can only be done
            // through the accompanying sales order fulfillment.
            // TODO: Handle dropship fulfillment

            $updatedLine = collect($receiptLines)->where('purchase_order_line_id', $receiptLine->purchaseOrderShipmentLine->purchase_order_line_id)->first();

            /** Since shipments are created on the fly with receipts, if updated receipt quantity exceeds shipment quantity,
             * we will update the shipment quantity as well. However, this will be aborted if the new receipt quantity
             * exceeds the quantity on the purchase order line.
             */
            if ($updatedLine['quantity'] > $receiptLine->purchaseOrderShipmentLine->unreceived_quantity + $receiptLine->quantity) { // Add back in current quantity received
                $orderLine = $receiptLine->purchaseOrderShipmentLine->purchaseOrderLine;

                $unreceivedLineQuantity = $orderLine->quantity - $orderLine->received_quantity;

                if ($updatedLine['quantity'] > $unreceivedLineQuantity + $receiptLine->quantity) {
                    throw new InvalidArgumentException($updatedLine['quantity'].' is more than unreceived shipment for line: '.$orderLine->id);
                } else {
                    // The update won't exceed the total quantity on the purchase order, we simply update the shipment quantity
                    $receiptLine->purchaseOrderShipmentLine->quantity = $updatedLine['quantity'];
                    $receiptLine->purchaseOrderShipmentLine->save();
                }
            } else {
                // Update the shipment quantity
                $receiptLine->purchaseOrderShipmentLine->update(['quantity' => $updatedLine['quantity']]);
            }

            $offset = $updatedLine['quantity'] - $receiptLine->quantity;

            if ($offset == 0) {
                return;
            }

            $purchaseOrderLine = $receiptLine->purchaseOrderShipmentLine->purchaseOrderLine;
            if ($purchaseOrderLine->product_id) {
                /**
                 * Handle the inventory events via the
                 * inventory manager.
                 */
                $manager = new InventoryManager(
                    $receiptLine->purchaseOrderShipmentLine->purchaseOrderLine->purchaseOrder->destination_warehouse_id,
                    $purchaseOrderLine->product
                );

                if ($offset > 0) {
                    $manager->increasePositiveEventQty($offset, $receiptLine);
                } elseif ($offset < 0) {
                    $manager->reducePositiveEventQty(abs($offset), $receiptLine);
                }

                // Update received quantity cache
                $receiptLine->reduceReceivedQuantityCacheOnOrderLine(-$offset);
            }

            // Update quantity
            $receiptLine->quantity = $updatedLine['quantity'];
            $receiptLine->save();
        });

        /**
         * Incase there is any unallocated unreceived quantity on the
         * purchase order line, not yet allocated to a backorder queue,
         * we attempt to use it to cover queues.
         */
        dispatch(new SyncBackorderQueueCoveragesJob($keptLines->get()->pluck('purchase_order_shipment_line.purchase_order_line_id')->toArray()));
    }

    private function handleAdditionalReceiptLines(
        PurchaseOrderShipmentReceipt $receipt,
        array $additionalReceiptLines,
        array $receiptLines,
        $warehouseId
    ) {
        if (empty($additionalReceiptLines)) {
            return;
        }
        // Create the receipt lines
        $createdLines = $receipt->purchaseOrderShipmentReceiptLines()->createMany(
            collect($receiptLines)->whereIn('purchase_order_line_id', $additionalReceiptLines)->toArray()
        );

        // Add receipt lines to inventory
        $receipt->addLinesToInventory($createdLines, $warehouseId);
    }

    private function handleRemovedLinesFromReceipt(PurchaseOrderShipmentReceipt $receipt, array $removedLineIds)
    {
        $removedLines = $this->buildReceiptLinesByPurchaseOrderLines($receipt, $removedLineIds);
        $removedLines->each(function (PurchaseOrderShipmentReceiptLine $receiptLine) {
            // If line isn't a product, we don't need to process this.
            $purchaseOrderLine = $receiptLine->purchaseOrderShipmentLine->purchaseOrderLine;
            if (! $purchaseOrderLine->product_id || ! $purchaseOrderLine->purchaseOrder->destination_warehouse_id) {
                return;
            }
            $receiptLine->reduceReceivedQuantityCacheOnOrderLine();
        });

        (new BulkInventoryManager())->bulkDeletePositiveInventoryEvents($removedLines->get());

        dispatch(new SyncBackorderQueueCoveragesJob($removedLines->get()->pluck('purchase_order_shipment_line.purchase_order_line_id')->toArray()));
    }

    /**
     * @return HasMany|mixed
     */
    private function buildReceiptLinesByPurchaseOrderLines(PurchaseOrderShipmentReceipt $receipt, array $orderLineIds): HasMany
    {
        return $receipt->purchaseOrderShipmentReceiptLines()
            ->whereHas('purchaseOrderShipmentLine', function (Builder $builder) use ($orderLineIds) {
                return $builder->whereHas('purchaseOrderLine', function (Builder $builder) use ($orderLineIds) {
                    return $builder->whereIn('id', $orderLineIds);
                });
            });
    }

    private function extractPurchaseOrderLineIds(array $receiptLines): array
    {
        return array_map(function ($line) {
            return $line['purchase_order_line_id'];
        }, $receiptLines);
    }

    /**
     * Adds the shipment line ids for each receipt line.
     */
    private function bindShipmentLines(PurchaseOrderShipment $shipment, array $receiptLines): array
    {
        foreach ($receiptLines as $key => $receiptLine) {
            $receiptLines[$key]['purchase_order_shipment_line_id'] = $shipment->purchaseOrderShipmentLines()
                ->where('purchase_order_line_id', $receiptLine['purchase_order_line_id'])
                ->first()->id;
        }

        return $receiptLines;
    }

    private function getShipmentForReceipts(PurchaseOrder $purchaseOrder, array $payload): PurchaseOrderShipment
    {
        // We attempt to find the shipment (if one is provided) and create one on the fly otherwise.
        /** @var PurchaseOrderShipment $purchaseOrderShipment */
        $purchaseOrderShipment = PurchaseOrderShipment::with(['purchaseOrder'])->find($payload['purchase_order_shipment_id'] ?? null);
        if (! $purchaseOrderShipment) {
            $purchaseOrderShipment = $purchaseOrder->purchaseOrderShipments()
                ->create([
                    'purchase_order_id' => $purchaseOrder->id,
                    'shipment_date' => now(),
                    'fulfilled_shipping_method_id' => $purchaseOrder->requested_shipping_method_id,
                ]);
        }

        return $purchaseOrderShipment;
    }

    /**
     * @throws NotOpenPurchaseOrderException
     */
    private function validateOrderFromReceiptLines(array $receiptLines): PurchaseOrder
    {
        if (empty($receiptLines)) {
            throw new InvalidArgumentException('Receipt lines cannot be empty.');
        }
        // Get the purchase order from the first receipt line.
        $purchaseOrder = PurchaseOrderLine::with(['purchaseOrder'])->findOrFail($receiptLines[0]['purchase_order_line_id'])->purchaseOrder;

        // We check the status of the purchase order
        $this->checkPurchaseOrderStatus($purchaseOrder);

        return $purchaseOrder;
    }

    private function receivingForShipment(array $payload): bool
    {
        return ! empty($payload['purchase_order_shipment_id']);
    }

    private function receivingEntireOrder(array $payload): bool
    {
        return ! empty($payload['purchase_order_id']);
    }

    private function hasReceiptLines(array $payload): bool
    {
        return ! empty($payload['receipt_lines']);
    }

    private function hasUnexpectedItems(array $payload): bool
    {
        return ! empty($payload['unexpected_items']);
    }

    /**
     * @return PurchaseOrder|Model
     */
    private function completelyReceiveForOrder($purchaseOrderId, array $data)
    {
        // Get the purchase order and receive all lines.
        /** @var PurchaseOrder $purchaseOrder */
        $purchaseOrder = PurchaseOrder::with([])->findOrFail($purchaseOrderId);

        $warehouseId = $data['warehouse_id'] ?? $purchaseOrder->destination_warehouse_id;
        if (! empty($warehouseId)) {
            $data['warehouse_id'] = $warehouseId;
        }

        if (! $purchaseOrder->fully_shipped) {
            // We ship all the lines first
            if (empty($data['shipment_date'])) {
                $data['shipment_date'] = now();
            }
            $shipment = $purchaseOrder->shipAllLines($data);

            // Mark the Purchase Order as shipped
            $purchaseOrder->shipped($shipment->shipment_date);
        }

        // Receive un received shipments
        $purchaseOrder->purchaseOrderShipments()
            ->whereNull('fully_received_at')
            ->with(['purchaseOrderShipmentLines', 'purchaseOrderShipmentReceipts'])
            ->each(function (PurchaseOrderShipment $shipment) use ($data) {
                // We fully receive the shipment
                $this->completelyReceiveForShipment($shipment, $data);
            });

        return $purchaseOrder;
    }

    /**
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     */
    private function completelyReceiveForShipment(
        $shipment,
        array $data,
        bool $addToInventory = true
    ): PurchaseOrderShipmentReceipt {
        if (is_numeric($shipment)) {
            /** @var PurchaseOrderShipment $shipment */
            $shipment = PurchaseOrderShipment::with([])->findOrFail($shipment);
        }

        // Warehouse id must be provided or the purchase order must have a destination warehouse
        $warehouseId = $data['warehouse_id'] ?? $shipment->purchaseOrder->destination_warehouse_id;
        if (! empty($warehouseId)) {
            $data['warehouse_id'] = $warehouseId;
        }

        $receiptLines = [];
        $shipment->purchaseOrderShipmentLines->each(function (PurchaseOrderShipmentLine $shipmentLine) use (&$receiptLines) {
            if (! $shipmentLine->purchaseOrderLine->fully_received) {
                // Add the receipt line
                $receiptLines[] = [
                    'purchase_order_line_id' => $shipmentLine->purchase_order_line_id,
                    'purchase_order_shipment_line_id' => $shipmentLine->id,
                    'quantity' => $shipmentLine->unreceived_quantity,
                ];
            }
        });

        if (count($receiptLines) > 20) {
            return $this->receiveLinesWithDataFeed($receiptLines, $data, true, $shipment);
        } else {
            return $this->receiveLines($receiptLines, $data, true, $shipment);
        }
    }

    public function addLinesToExistingShipmentReceipt(PurchaseOrderShipmentReceipt $purchaseOrderReceipt, array $lines): PurchaseOrderShipmentReceipt
    {
        $shipment = $purchaseOrderReceipt->purchaseOrderShipment;
        $purchaseOrder = $shipment->purchaseOrder;
        $productsUpdated = [];
        $backorderQueueCollection = collect();

        $this->handleLinesChunk($lines, $purchaseOrderReceipt, $purchaseOrderReceipt->purchaseOrderShipment, $purchaseOrder, $productsUpdated, $backorderQueueCollection);

        return $purchaseOrderReceipt->refresh();
    }
}
