<?php

namespace App\Services\PurchaseOrder;

use App\Helpers;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\Warehouse;
use App\Validator;
use InvalidArgumentException;

class PurchaseOrderValidator
{
    const PURCHASE_ORDER_ACTION_CREATE = 'create';

    const PURCHASE_ORDER_ACTION_UPDATE = 'update';

    const PURCHASE_ORDER_ACTIONS = [
        self::PURCHASE_ORDER_ACTION_UPDATE,
        self::PURCHASE_ORDER_ACTION_CREATE,
    ];

    const APPROVAL_STATUS_PENDING = 'pending';

    const APPROVAL_STATUS_APPROVED = 'approved';

    const APPROVAL_STATUS = [self::APPROVAL_STATUS_PENDING, self::APPROVAL_STATUS_APPROVED];

    const STATUS_DRAFT = 'draft';

    private $action = self::PURCHASE_ORDER_ACTION_CREATE;

    /**
     * @var PurchaseOrder
     */
    private $purchaseOrder;

    private function creating(): bool
    {
        return $this->action === self::PURCHASE_ORDER_ACTION_CREATE;
    }

    private function updating(): bool
    {
        return $this->action === self::PURCHASE_ORDER_ACTION_UPDATE;
    }

    public function withAction(string $action, ?PurchaseOrder $purchaseOrder = null): self
    {
        if (! in_array($action, self::PURCHASE_ORDER_ACTIONS)) {
            throw new InvalidArgumentException('Unknown action: '.$action);
        }

        if ($action === self::PURCHASE_ORDER_ACTION_UPDATE && ! $purchaseOrder) {
            throw new InvalidArgumentException('Provide sales order to be updated.');
        }
        $this->action = $action;
        $this->purchaseOrder = $purchaseOrder;

        return $this;
    }

    public function rules(): array
    {
        $rules = [
            'purchase_order_date' => 'required|date',
            'other_date' => 'nullable|date',
            'purchase_order_number' => 'nullable|max:255|unique:purchase_orders,purchase_order_number,'.request()->route()?->parameter('purchase_order')?->id,
            'submission_format' => 'nullable|in:'.implode(',', PurchaseOrder::SUBMISSION_FORMATS),
            'submission_status' => 'nullable|in:'.implode(',', PurchaseOrder::SUBMISSION_STATUS),
            'last_submitted_at' => 'nullable|datetime',
            'approval_status' => 'nullable|in:'.implode(',', self::APPROVAL_STATUS),
            'order_status' => 'nullable|in:'.self::STATUS_DRAFT,
            'payment_term_id' => 'nullable|exists:payment_terms,id',
            'incoterm_id' => 'nullable|exists:incoterms,id',
            'store_id' => 'nullable|exists:stores,id',
            'shipping_method_id' => 'nullable|exists:shipping_methods,id',
            'requested_shipping_method_id' => 'nullable|exists:shipping_methods,id', // it's same shipping_method_id
            'requested_shipping_method' => 'nullable|max:255',
            'supplier_id' => 'required_without:supplier_name|exists:suppliers,id',
            'supplier_name' => 'required_without:supplier_id|exists:suppliers,name',
            'supplier_warehouse_id' => 'nullable|exists:warehouses,id',
            'destination_warehouse_id' => 'nullable|exists:warehouses,id',
            'currency_id' => 'required_without:currency_code|exists:currencies,id',
            'currency_code' => 'required_without:currency_id|exists:currencies,code',
            'estimated_delivery_date' => 'nullable|date',
            'supplier_notes' => 'nullable',
            'shipment_id' => 'nullable',
            'tracking_number' => 'nullable',
            'is_tax_included' => 'nullable|boolean',
            'tax_rate_id' => 'nullable|numeric',
            'tags' => 'nullable|array',

            // Purchase Order Lines

            'purchase_order_lines' => 'nullable|array',
            'purchase_order_lines.*.description' => 'nullable|max:255',
            'purchase_order_lines.*.is_deleted' => 'nullable|boolean',
            'purchase_order_lines.*.product_id' => 'nullable|exists:products,id',
            'purchase_order_lines.*.sku' => 'nullable|exists:products,sku',
            'purchase_order_lines.*.quantity' => 'required|numeric|min:0|lt:100000',
            'purchase_order_lines.*.amount' => 'required|numeric|min:0|lt:100000',
            'purchase_order_lines.*.tax' => 'nullable|numeric|lt:100000',
            'purchase_order_lines.*.tax_rate_id' => 'nullable|numeric',
            'purchase_order_lines.*.discount' => 'nullable|numeric|lt:100000',
            'purchase_order_lines.*.estimated_delivery_date' => 'nullable|date',
            'purchase_order_lines.*.nominal_code_id' => 'nullable|exists:nominal_codes,id',
            'purchase_order_lines.*.line_reference' => 'nullable|unique:purchase_order_lines,line_reference',
            'purchase_order_lines.*.linked_backorders' => 'nullable|array',
            'purchase_order_lines.*.linked_backorders.*.id' => 'required|exists:backorder_queues,id',
            'purchase_order_lines.*.linked_backorders.*.quantity' => 'required|numeric|gt:0',
            'purchase_order_lines.*.tax_rate' => 'nullable|numeric',

            // Is purchase order created from forecasting
            'forecasting_products' => 'nullable|array',
            'forecasting_products.*.product_id' => 'required|exists:products,id',
            'forecasting_products.*.quantity' => 'required|numeric|min:1|lt:100000',
            'forecasting_products.*.amount' => 'required|numeric|min:0|lt:100000',
        ];

        if ($this->updating()) {
            $rules['po_number'] = $rules['purchase_order_number'];
            $rules['purchase_order_date'] = 'sometimes|'.$rules['purchase_order_date'];
            $rules['supplier_id'] = 'sometimes|'.$rules['supplier_id'];
            $rules['supplier_name'] = 'sometimes|'.$rules['supplier_name'];
            $rules['currency_id'] = 'sometimes|'.$rules['currency_id'];
            $rules['currency_code'] = 'sometimes|'.$rules['currency_code'];
            $rules['purchase_order_lines.*.id'] = 'nullable|exists:purchase_order_lines,id';
            $rules['tracking_number'] = 'sometimes|'.$rules['tracking_number'];
        }

        return $rules;
    }

    public function withValidator(Validator $validator)
    {
        $attributes = $validator->attributes();

        if ($this->creating() && isset($attributes['destination_warehouse_id'])) {
            $warehouse = Warehouse::with([])->find($attributes['destination_warehouse_id']);
            if ($warehouse->type == Warehouse::TYPE_AMAZON_FBA && ! isset($attributes['shipment_id'])) {
                $validator->addFailure('shipment_id', 'Required');
            }
        }

        if (isset($attributes['purchase_order_lines'])) {
            foreach ($attributes['purchase_order_lines'] as $index => $line) {
                if (isset($line['product_id'])) {
                    $product = Product::find($line['product_id']);
                    if ($product && $product->type !== Product::TYPE_STANDARD) {
                        $validator->errors()->add("purchase_order_lines.{$index}.product_id", 'Only standard products can be purchased.');
                    }
                }

                // Check linked backorder queue quantities
                if (isset($line['linked_backorders'])) {
                    $totalQuantity = collect($line['linked_backorders'])->sum('quantity');
                    if ($totalQuantity > $line['quantity']) {
                        $validator->errors()->add("purchase_order_lines.{$index}.product_id", 'Total linked backorders quantity cannot exceed purchase order line quantity.');
                    }
                }
            }
        }

        // check the selected supplier have a default warehouse and the selected supplier warehouse belongs to the supplier
        if (! empty($attributes['supplier_id'])) {
            $supplier = Supplier::with(['warehouses'])->findOrFail($attributes['supplier_id']);
        } elseif (! empty($attributes['supplier_name'])) {
            $supplier = Supplier::with(['warehouses'])->where('name', $attributes['supplier_name'])->firstOrFail();
        } else {
            $supplier = $this->purchaseOrder->supplier;
        }

        if (isset($attributes['purchase_order_lines']) && count($attributes['purchase_order_lines'])) {
            $this->validateSupplierProducts($attributes['purchase_order_lines'], $supplier, $validator);
            $this->validateReceivedInvoicedLines($attributes['purchase_order_lines'], $validator);
        }

        if ($this->updating() && (isset($attributes['supplier_id']) || isset($attributes['supplier_name'])) && ! isset($attributes['purchase_order_lines'])) {
            $lines = $this->purchaseOrder->purchaseOrderLines;

            $this->validateSupplierProducts($lines, $supplier, $validator);
        }

        if (empty($attributes['supplier_warehouse_id'])) {
            if (empty($supplier->default_warehouse_id)) {
                $validator->addFailure('supplier_id', 'MustHaveDefaultWarehouse');
            }
        } else {
            if (! $supplier->warehouses->firstWhere('id', $attributes['supplier_warehouse_id'])) {
                $validator->addFailure('supplier_warehouse_id', 'DoesNotBelongToSupplier');
            }
        }

        if ($this->creating()) {
            // check store
            if (empty($attributes['store_id']) && empty($supplier->default_store_id) && empty(Helpers::setting(Setting::KEY_PO_DEFAULT_STORE))) {
                $validator->addFailure('store_id', 'RequiredStore');
            }
        }

        return $validator;
    }

    public function messages()
    {
        return [
            'must_be_unique_by_supplier' => __('messages.purchase_order.po_number_used_by_supplier'),
            'does_not_belong_to_supplier' => __('messages.purchase_order.warehouse_not_belong_to_supplier'),
            'must_have_default_warehouse' => __('messages.supplier.must_have_default_warehouse'),
            'required_store' => __('messages.purchase_order.required_store'),
            'not_associated_to_supplier' => __('messages.purchase_order.not_associated_to_supplier'),
            'quantity_less_than_received' => __('messages.purchase_order.quantity_less_than_received'),
            'quantity_less_than_invoiced' => __('messages.purchase_order.quantity_less_than_invoiced'),
            'can_not_delete_received_lines' => __('messages.purchase_order.can_not_delete_received_lines'),
            'can_not_delete_invoiced_lines' => __('messages.purchase_order.can_not_delete_invoiced_lines'),
            'order_status.in' => __('messages.purchase_order.only_draft_orders'),
            'purchase_order_lines.*.sku.exists' => __('messages.purchase_order.product_not_exists_validation'),
        ];
    }

    public function validateSupplierProducts($lines, $supplier, Validator $validator): void
    {
        $ids = collect($lines)->pluck('product_id')->filter();
        $skus = collect($lines)->pluck('sku')->filter();
        if ($skus->isNotEmpty()) {
            $productIds = Product::with([])->whereIn('sku', $skus)->pluck('id', 'sku');
            $ids = $ids->merge($productIds->values());
        }
        $inventory = SupplierProduct::with([])->where('supplier_id', $supplier->id)->whereIn('product_id', $ids->unique()->values())->get();
        foreach ($lines as $index => $line) {
            $key = 'product_id';
            if (! isset($line['product_id'])) {
                if (isset($line['sku'])) {
                    $line['product_id'] = $productIds[$line['sku']];
                    $key = 'sku';
                } else {
                    continue;
                }
            }

            $supplierInventory = $inventory->firstWhere('product_id', $line['product_id']);
            if (! $supplierInventory) {
                $validator->addFailure("purchase_order_lines.{$index}.{$key}", 'NotAssociatedToSupplier');
            }
        }
    }

    private function validateReceivedInvoicedLines(array $orderLines, Validator $validator)
    {
        if ($this->updating() && request()->isMethod('put') && $this->purchaseOrder->order_status != PurchaseOrder::STATUS_DRAFT) {
            $this->purchaseOrder->loadMissing('purchaseOrderLines.purchaseOrderShipmentLines');

            $existingPurchaseOrderLines = [];
            foreach ($orderLines as $index => $orderLine) {
                $purchaseOrderLine = PurchaseOrderLine::findByStoreRequest($this->purchaseOrder->id, $orderLine);
                // check updated quantity, received/invoiced
                if ($purchaseOrderLine && $purchaseOrderLine->product_id && isset($orderLine['quantity'])) {
                    // purchaseOrderLine from the relation to prevent reload purchaseOrderShipmentLines relation again.
                    /** @var PurchaseOrderLine $purchaseOrderLine */
                    $purchaseOrderLine = $this->purchaseOrder->purchaseOrderLines->firstWhere('id', $purchaseOrderLine->id);

                    // check received quantity
                    if ($purchaseOrderLine->received_quantity > $orderLine['quantity']) {
                        $validator->addFailure("purchase_order_lines.{$index}.quantity", 'QuantityLessThanReceived');
                    }
                    // check invoiced quantity
                    if ($purchaseOrderLine->invoice_total > $orderLine['quantity']) {
                        $validator->addFailure("purchase_order_lines.{$index}.quantity", 'QuantityLessThanInvoiced');
                    }
                }
                $existingPurchaseOrderLines[] = $purchaseOrderLine->id ?? null;
            }

            // check deleted lines, invoiced/received
            $linesThatWillDeleted = $this->purchaseOrder->purchaseOrderLines->whereNotIn('id', $existingPurchaseOrderLines)->where('product_id', '!=', null);
            if ($linesThatWillDeleted->where('received_quantity', '>', 0)->isNotEmpty()) {
                $validator->addFailure('purchase_order_lines', 'CanNotDeleteReceivedLines');
            }
            if ($linesThatWillDeleted->where('invoice_total', '>', 0)->isNotEmpty()) {
                $validator->addFailure('purchase_order_lines', 'CanNotDeleteInvoicedLines');
            }
        }
    }
}
