<?php

namespace App\Services\PurchaseOrder\PurchaseOrderBuilder;

use App\Models\BackorderQueue;
use App\Models\BackorderQueueCoverage;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductComponent;
use App\Models\ProductInventory;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\Warehouse;
use App\Services\InventoryForecasting\MakesForecastSalesData;
use Carbon\Carbon;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Entities\AmazonProduct;

/**
 * Class PurchaseOrderBuilderRepository.
 */
class PurchaseOrderBuilderRepository
{
    use MakesForecastSalesData;

    public function getBackordersAsInventoryAvailableWithSalesFilters(
        Supplier $supplier,
        int $warehouseId,
        array $productFilters,
        array $salesFilters = []
    ): Collection {
        $query = $this->getSupplierProductsQuery(
            supplier: $supplier,
            warehouseId: $warehouseId,
            filters: $productFilters
        );

        if (! empty($salesFilters)) {
            // With sales order filters, we ensure that both
            // backorder quantity (as negative inventory available) and
            // backorder coverages (as inventory inbound) are based on the
            // sales order filters.

            $withBackorderQueueSubQuery = SalesOrderLine::with(['salesOrder'])
                ->selectRaw('sales_order_lines.product_id, sales_order_lines.sales_order_id, backorder_queues.sales_order_line_id, backorder_queues.id, backorder_queues.shortage_quantity')
                ->join('backorder_queues', 'backorder_queues.sales_order_line_id', 'sales_order_lines.id')
                ->where('warehouse_id', $warehouseId)
                ->whereHas('salesOrder', function (Builder $q) use ($salesFilters) {
                    $q->filter($salesFilters);
                });

            $coveragesSubQuery = BackorderQueueCoverage::query()
                ->selectRaw('backorder_queue_id, coalesce(sum(unreleased_quantity), 0) unreleased_quantity')
                ->where('unreleased_quantity', '>', 0)
                ->groupBy('backorder_queue_id');
            $query
                ->selectRaw('boq.sales_order_line_id, boq.id backorder_queue_id, sales_orders.id sales_order_id, sales_orders.sales_order_number, -coalesce(sum(boq.shortage_quantity), 0) inventory_available, coalesce(sum(backorder_queue_coverages.unreleased_quantity), 0) inventory_inbound')
                ->joinSub($withBackorderQueueSubQuery, 'boq', 'boq.product_id', 'products.id')
                ->join('sales_orders', 'sales_orders.id', 'boq.sales_order_id')
                ->leftJoinSub($coveragesSubQuery, 'backorder_queue_coverages', 'backorder_queue_coverages.backorder_queue_id', 'boq.id')
                ->groupBy(['products.id', 'boq.sales_order_line_id']);
        }

        return $this->makeSupplierProductsResults($query);
    }

    private function makeSupplierProductsResults(Builder $builder): Collection
    {
        return $builder
            ->groupBy(['products.id'])
            ->get();
    }

    public function getSupplierProductsQuery($supplier, int $warehouseId, array $filters): Builder
    {
        $inventoryQuery = ProductInventory::query()
            ->selectRaw('product_id inventory_product_id, coalesce(sum(inventory_available), 0) inventory_available')
            ->where('warehouse_id', $warehouseId)
            ->groupBy('product_id');

        $inboundQuery = PurchaseOrderLine::query()
            ->selectRaw(
                'product_id inbound_product_id, coalesce(sum(purchase_order_lines.quantity), 0) - coalesce(sum(purchase_order_shipment_receipt_lines.quantity), 0) inventory_inbound'
            )
            ->joinRelationship('purchaseOrder', function ($q) use ($warehouseId) {
                return $q->where('order_status', PurchaseOrder::STATUS_OPEN)
                    ->where('receipt_status', '!=', PurchaseOrder::RECEIPT_STATUS_RECEIVED)
                    ->where('destination_warehouse_id', $warehouseId);
            })
            ->leftJoinRelationship('purchaseOrderShipmentLines.purchaseOrderShipmentReceiptLines')
            ->whereNotNull('product_id')
            ->groupBy('product_id');

        $parentsQuery = ProductComponent::query()
            ->selectRaw('component_product_id product_id, COALESCE(backorder_queues.shortage_quantity * product_components.quantity, 0) parent_backordered_quantity')
            ->join('sales_order_lines', 'sales_order_lines.product_id', 'product_components.parent_product_id')
            ->join('backorder_queues', 'backorder_queues.sales_order_line_id', 'sales_order_lines.id');

        return Product::query()
            ->selectRaw('products.id, products.sku, products.name, barcode, inventory_available, inventory_inbound, parent_backordered_quantity')
            ->joinSub($inventoryQuery, 'inventory', 'inventory.inventory_product_id', 'products.id')
            ->leftJoinSub($inboundQuery, 'inbound', 'inbound.inbound_product_id', 'products.id')
            ->leftJoinSub($parentsQuery, 'parents', 'parents.product_id', 'products.id')
            ->joinRelationship('supplierProducts.supplier')
            ->filter($filters)
            ->where('type', '!=', Product::TYPE_KIT) // Kit cannot be purchased
            ->where('suppliers.id', $supplier->id)
            ->limit(PurchaseOrderBuilder::MAX_PRODUCTS_COUNT)
            ->groupBy(['products.id']);
    }


    /**
     * This method is used to get the sales history data for a supplier.
     * It returns a query builder that can be used to get the sales history data.
     *
     * @param  int  $supplierId
     * @param  int|null  $warehouseId
     * @param  Carbon|null  $historyStartDate
     * @param  Carbon|null  $historyEndDate
     * @param  array  $productFilters
     * @param  array  $salesHistoryFilters
     * @return false|\Illuminate\Database\Query\Builder
     */
    private function getSalesHistoryQuery(
        int $supplierId,
        ?int $warehouseId = null,
        ?Carbon $historyStartDate = null,
        ?Carbon $historyEndDate = null,
        array $productFilters = [],
        array $salesHistoryFilters = []
    ): false|\Illuminate\Database\Query\Builder {

        $salesOrdersQuery = SalesOrder::query()
            ->where('order_status', '!=', SalesOrder::STATUS_DRAFT)
            ->filter($salesHistoryFilters);

        $productsQuery = Product::query()
            ->where('type', '!=', Product::TYPE_BLEMISHED)
            ->filter($productFilters);

        $nonKitProductsQuery = Product::query()
            ->whereNotIn('type', [Product::TYPE_KIT, Product::TYPE_BLEMISHED])
            ->filter($productFilters);

        // Kit sales query
        $kitSalesQuery = SalesOrderLine::join('product_components as pc', 'sales_order_lines.product_id', '=', 'pc.parent_product_id')
            ->joinSub($salesOrdersQuery, 'so', 'so.id', 'sales_order_lines.sales_order_id')
            ->joinSub($productsQuery, 'p', 'p.id', 'pc.component_product_id')
            ->join('supplier_products as sp', 'sp.product_id', '=', 'pc.component_product_id')
            ->when($warehouseId, function ($q) use ($warehouseId) {
                $q->where('sales_order_lines.warehouse_id', $warehouseId);
            })
            ->when($historyStartDate, function ($q) use ($historyStartDate) {
                $q->where('so.order_date', '>=', $historyStartDate->setTimezone('UTC'));
            })
            ->when($historyEndDate, function ($q) use ($historyEndDate) {
                $q->where('so.order_date', '<=', $historyEndDate->setTimezone('UTC'));
            })
            ->where('sp.supplier_id', $supplierId)
            ->select([
                'sales_order_lines.warehouse_id',
                'sales_order_lines.sales_order_id',
                'pc.component_product_id as product_id',
                'so.sales_order_number',
                DB::raw('sales_order_lines.quantity * pc.quantity as quantity')
            ]);

        // Non-kit sales query
        $nonKitSalesQuery = SalesOrderLine::joinSub($salesOrdersQuery, 'so', 'so.id', 'sales_order_lines.sales_order_id')
            ->joinSub($nonKitProductsQuery, 'p', 'p.id', 'sales_order_lines.product_id')
            ->join('supplier_products as sp', 'sp.product_id', '=', 'sales_order_lines.product_id')
            ->whereNotIn('sales_order_lines.product_id', function ($query) {
                $query->select('parent_product_id')->from('product_components');
            })
            ->when($warehouseId, function ($q) use ($warehouseId) {
                $q->where('sales_order_lines.warehouse_id', $warehouseId);
            })
            ->when($historyStartDate, function ($q) use ($historyStartDate) {
                $q->where('so.order_date', '>=', $historyStartDate->setTimezone('UTC'));
            })
            ->when($historyEndDate, function ($q) use ($historyEndDate) {
                $q->where('so.order_date', '<=', $historyEndDate->setTimezone('UTC'));
            })
            ->where('sp.supplier_id', $supplierId)
            ->select([
                'sales_order_lines.warehouse_id',
                'sales_order_lines.sales_order_id',
                'sales_order_lines.product_id',
                'so.sales_order_number',
                'sales_order_lines.quantity'
            ]);

        // Combine the queries using union all
        $allSalesQuery = $kitSalesQuery->unionAll($nonKitSalesQuery);

        $daysBetweenDates = $historyStartDate && $historyEndDate ? $historyStartDate->diffInDays($historyEndDate) : 0;
        if($daysBetweenDates === 0){
            // No days of stock history, so we return empty results
            return false;
        }

        // Final query with grouping and aggregation
        return DB::table(DB::raw("({$allSalesQuery->toSql()}) as all_sales"))
            ->mergeBindings($allSalesQuery->getQuery())
            ->select([
                'product_id',
                DB::raw('GROUP_CONCAT(`sales_order_number` SEPARATOR ",") as sales_order_numbers'),
                DB::raw("SUM(quantity) / $daysBetweenDates as quantity")
            ])
            ->groupBy('product_id');
    }


    protected function getInventoryQuery(?int $warehouseId = null): \Illuminate\Database\Query\Builder
    {

        $warehousesQuery = Warehouse::query()
            ->select('id', 'type', 'integration_instance_id')
            ->when($warehouseId, function ($q) use ($warehouseId) {
                $q->where('id', $warehouseId);
            });

        $inTransitQuery = ProductInventory::query()
            ->selectRaw('
            product_id, 
            cast(coalesce(sum(inventory_in_transit), 0) as signed) as inventory_in_transit,
            cast(0 as unsigned) as inventory_available,
            cast(0 as unsigned) as inventory_inbound
            ')
            ->joinSub($warehousesQuery, 'warehouses', 'warehouses.id', 'products_inventory.warehouse_id')
            ->where('warehouse_id', '!=', 0)
            ->whereNotIn('warehouses.type', [Warehouse::TYPE_AMAZON_FBA, Warehouse::TYPE_SUPPLIER])
            ->when($warehouseId, function ($q) use ($warehouseId) {
                $q->where('products_inventory.warehouse_id', $warehouseId);
            })
            ->groupBy('product_id');


        $availableInventoryQuery = InventoryMovement::query()
            ->selectRaw('
            product_id,
            cast(0 as unsigned) as inventory_in_transit,
            cast(coalesce(sum(quantity), 0) as signed) as inventory_available,
            cast(0 as unsigned) as inventory_inbound        
            ')
            ->joinSub($warehousesQuery, 'warehouses', 'warehouses.id', 'inventory_movements.warehouse_id')
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->whereNotIn('warehouses.type', [Warehouse::TYPE_AMAZON_FBA, Warehouse::TYPE_SUPPLIER])
            ->when($warehouseId, function ($q) use ($warehouseId) {
                $q->where('inventory_movements.warehouse_id', $warehouseId);
            })
            ->groupBy('product_id');

        $inboundQuery = PurchaseOrderLine::query()
            ->selectRaw('
            product_id, 
            cast(0 as unsigned) as inventory_in_transit,
            cast(0 as unsigned) as inventory_available,
            cast(coalesce(sum(quantity), 0) - coalesce(sum(received_quantity), 0) as signed) as inventory_inbound
            ')
            ->join('purchase_orders', 'purchase_orders.id', 'purchase_order_lines.purchase_order_id')
            ->joinSub($warehousesQuery, 'warehouses', 'warehouses.id', 'purchase_orders.destination_warehouse_id')
            ->where('purchase_orders.order_status', PurchaseOrder::STATUS_OPEN)
            ->whereNotIn('warehouses.type', [Warehouse::TYPE_AMAZON_FBA, Warehouse::TYPE_SUPPLIER])
            ->when($warehouseId, function ($q) use ($warehouseId) {
                $q->where('destination_warehouse_id', $warehouseId);
            })
            ->groupBy('product_id');

        // Combine the subqueries using unions
        $combinedQuery = $availableInventoryQuery
            ->unionAll($inTransitQuery)
            ->unionAll($inboundQuery);

        // Create the final query to aggregate the results
        $inventoryQuery = DB::table(DB::raw("({$combinedQuery->toSql()}) as combined"))
            ->mergeBindings($combinedQuery->getQuery())
            ->select([
                'product_id',
                DB::raw('COALESCE(sum(inventory_available), 0) as inventory_available'),
                DB::raw('COALESCE(sum(inventory_inbound), 0) as inventory_inbound'),
                DB::raw('COALESCE(sum(inventory_in_transit), 0) as inventory_in_transit')
            ])
            ->groupBy('product_id');


        // Amazon FBA Warehouses
        $fbaWarehouses = Warehouse::query()
            ->selectRaw('
                products.id as product_id,
                cast(coalesce(sum(afri.afn_fulfillable_quantity + afri.afn_reserved_quantity), 0) as signed) as inventory_available,
                cast(coalesce(sum(afri.afn_inbound_working_quantity + afri.afn_inbound_shipped_quantity + afri.afn_inbound_receiving_quantity), 0) as signed) as inventory_inbound,
                cast(0 as unsigned) as inventory_in_transit
            ')
            ->join('amazon_fba_report_inventory as afri', 'afri.integration_instance_id', 'warehouses.integration_instance_id')
            ->join('amazon_products', function(JoinClause $q){
                $q->on('amazon_products.integration_instance_id', 'warehouses.integration_instance_id')
                    ->whereColumn('amazon_products.seller_sku', 'afri.sku');
            })
            ->join('product_listings', function(JoinClause $q){
                $q->on('product_listings.document_id', 'amazon_products.id')
                    ->where('product_listings.document_type', AmazonProduct::class);
            })
            ->join('products', 'products.id', 'product_listings.product_id')
            ->where('warehouses.type', Warehouse::TYPE_AMAZON_FBA)
            ->when($warehouseId, function ($q) use ($warehouseId) {
                $q->where('warehouses.id', $warehouseId);
            })
            ->groupBy('products.id');

        // Combine the queries using union all
        $query = $inventoryQuery->unionAll($fbaWarehouses);

        return DB::table(DB::raw("({$query->toSql()}) as inventory"))
            ->mergeBindings($query)
            ->select([
                'product_id',
                DB::raw('cast(COALESCE(sum(inventory_available), 0) as signed) as inventory_available'),
                DB::raw('cast(COALESCE(sum(inventory_inbound), 0) as unsigned) as inventory_inbound'),
                DB::raw('cast(COALESCE(sum(inventory_in_transit), 0) as signed) as inventory_in_transit')
            ])
            ->groupBy('product_id');
    }


    public function getAverageDailySalesWithInventory(
        int $supplierId,
        ?int $warehouseId = null,
        ?Carbon $historyStartDate = null,
        ?Carbon $historyEndDate = null,
        array $productFilters = [],
        array $salesHistoryFilters = []
    ): Arrayable|Collection {

        // Get the sales history query
        $salesHistory = $this->getSalesHistoryQuery(
            $supplierId,
            $warehouseId,
            $historyStartDate,
            $historyEndDate,
            $productFilters,
            $salesHistoryFilters
        );

        // If no sales history, return empty collection
        if(!$salesHistory){
            return collect();
        }

        // Inventory query
        $inventoryQuery = $this->getInventoryQuery($warehouseId);

        // Return sales history with inventory data
        return DB::query()
            ->selectRaw('
            p.product_id product_id, 
            p.sales_order_numbers,
            COALESCE(p.quantity, 0) as quantity, 
            COALESCE(inventory.inventory_available, 0) as inventory_available,
            COALESCE(inventory.inventory_inbound, 0) as inventory_inbound,
            COALESCE(inventory.inventory_in_transit, 0) as inventory_in_transit'
            )
            ->fromSub($salesHistory, 'p')
            ->leftJoinSub($inventoryQuery, 'inventory', function(JoinClause $q) use ($warehouseId) {
                $q->on('inventory.product_id', 'p.product_id');
            })
            ->having('quantity', '>', 0)
            ->groupBy('p.product_id')
            ->get();
    }

    public function getSupplierProducts($supplier, int $warehouseId, array $filters): array|Collection
    {
        return $this->getSupplierProductsQuery($supplier, $warehouseId, $filters)->get();
    }

    public function getAverageDailySales(
        Product $product,
        ?Carbon $startDate = null,
        ?Carbon $endDate = null,
        array $filters = []
    ): float {
        $builder = $this->buildHistoricalSales(
            $product,
            $filters,
            $startDate,
            $endDate
        );

        $daysBetweenDates = $startDate && $endDate ? $startDate->diffInDays($endDate) : 1;

        return $builder->get()->sum(function ($row) {
            return $row->quantity * $row->multiplier;
        }) / ($daysBetweenDates != 0 ? $daysBetweenDates : 1);
    }

    public function getUnallocatedBackorderQueueQuantity(Product $product, int $supplierId, int $warehouseId)
    {
        return
      (int) DB::table('queues')
          ->selectRaw('COALESCE(sum(outstanding), 0) as count')
          ->fromSub(function (\Illuminate\Database\Query\Builder $builder) use ($supplierId, $product, $warehouseId) {
              $builder->selectRaw('backorder_queues.*, COALESCE(sum(bqc.covered_quantity - bqc.released_quantity), 0) as coverages, COALESCE(sum(backorder_queues.backordered_quantity - backorder_queues.released_quantity), 0) as outstanding')
                  ->from('backorder_queues')
                  ->leftJoin('sales_order_lines as sol', 'backorder_queues.sales_order_line_id', 'sol.id')
                  ->leftJoin('backorder_queue_coverages as bqc', 'backorder_queues.id', 'bqc.backorder_queue_id')
                  ->where('backorder_queues.supplier_id', $supplierId)
                  ->where('sol.product_id', $product->id)
                  ->where('sol.warehouse_id', $warehouseId)
                  ->whereColumn('backorder_queues.backordered_quantity', '>', 'backorder_queues.released_quantity')
                  ->whereNotNull('backorder_queues.priority')
                  ->groupBy(['backorder_queues.id'])
                  ->havingRaw('outstanding > coverages');
          }, 'queues')->first()->count;
    }

}
