<?php

namespace App\Services\PurchaseOrder\PurchaseOrderBuilder;

use App\Models\SupplierProduct;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\Builders\SalesBasedPurchaseOrderBuilder;

class PurchaseOrderBuilderLine
{
    /**
     * @var SupplierProduct
     */
    protected SupplierProduct $supplierProduct;

    /**
     * @var int
     */
    protected int $quantity;

    /**
     * @var ?int
     */
    protected ?int $warehouseId;

    /**
     * @var array
     */
    protected array $options = [];

    /**
     * @var int|null
     */
    private ?int $salesOrderLineId = null;

    /**
     * @var int|null
     */
    private ?int $salesOrderId = null;

    /**
     * @var int|null
     */
    private ?int $backorderQueueId = null;

    /**
     * @var string|null
     */
    private ?string $salesOrderNumber = null;

    /**
     * PurchaseOrderBuilderLine constructor.
     */
    public function __construct(SupplierProduct $supplierProduct, int $quantity, ?int $warehouseId = null, array $options = [])
    {
        $this->supplierProduct = $supplierProduct;
        $this->quantity = $quantity;
        $this->warehouseId = $warehouseId;
        $this->options = $options;
    }

    /**
     * @return SupplierProduct
     */
    public function getSupplierProduct(): SupplierProduct
    {
        return $this->supplierProduct;
    }

    /**
     * @return int
     */
    public function getQuantity(): int
    {
        return $this->quantity;
    }

    /**
     * @return array
     */
    public function getCalculation(): array
    {
        return [
            'sample_days' => $this->options['sample_days'] ?? 0,
            'num_sales' => $this->options['num_sales'] ?? 0,
            'target_days_of_stock' => $this->options['target_days_of_stock'] ?? null,
            'leadtime' => $this->options['leadtime'] ?? 0,
            'daily_average_consumption' => ($this->options['sample_days'] ?? 0) != 0 ? ($this->options['num_sales'] ?? 0) / $this->options['sample_days'] : 0,
            'num_backordered' => $this->options['num_backordered'] ?? 0,
            'num_ordered' => $this->options['num_ordered'] ?? 0,
            'num_stock' => $this->options['num_stock'] ?? 0,
            'rounding_strategy' => $this->options['rounding_method'] ?? SalesBasedPurchaseOrderBuilder::ROUNDING_METHOD_HALF_UP,
            'quantity_needed' => $this->options['quantity_needed'] ?? 0,
            'quantity_calculated' => $this->options['quantity_calculated'] ?? 0,
            'num_inbound' => $this->options['num_inbound'] ?? 0,
            'num_in_transit' => $this->options['num_in_transit'] ?? 0,
            'sales_order_numbers' => $this->options['sales_order_numbers'] ?? [],
        ];
    }

    /**
     * @param  int|null  $id
     * @param  int|null  $salesOrderId
     * @param  string|null  $salesOrderNumber
     * @param  int|null  $backorderQueueId
     * @return $this
     */
    public function withSalesOrderLine(
        ?int $id = null,
        ?int $salesOrderId = null,
        ?string $salesOrderNumber = null,
        ?int $backorderQueueId = null
    ): self {
        if (! $id || ! $salesOrderId || ! $salesOrderNumber || ! $backorderQueueId) {
            return $this;
        }

        $this->salesOrderLineId = $id;
        $this->salesOrderId = $salesOrderId;
        $this->salesOrderNumber = $salesOrderNumber;
        $this->backorderQueueId = $backorderQueueId;

        return $this;
    }

    /**
     * @return int|null
     */
    public function getSalesOrderLineId(): ?int
    {
        return $this->salesOrderLineId;
    }

    /**
     * @return int|null
     */
    public function getSalesOrderId(): ?int
    {
        return $this->salesOrderId;
    }

    /**
     * @return ?int
     */
    public function getWarehouseId(): ?int
    {
        return $this->warehouseId;
    }

    /**
     * @return string|null
     */
    public function getSalesOrderNumber(): ?string
    {
        return $this->salesOrderNumber;
    }

    /**
     * @return int|null
     */
    public function getBackorderQueueId(): ?int
    {
        return $this->backorderQueueId;
    }
}
