<?php

namespace App\Services\PurchaseOrder\PurchaseOrderBuilder;

use App\Models\Product;
use App\Models\Supplier;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\Exceptions\LargePurchaseOrderException;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Collection;

abstract class PurchaseOrderBuilder
{
    const MAX_PRODUCTS_COUNT = 20000;

    /**
     * @var Supplier
     */
    protected $supplier;

    /**
     * @var int
     */
    protected $destinationWarehouseId;

    /**
     * @var array
     */
    protected $productFilters = [];

    /**
     * @var Product[]|Collection|mixed
     */
    protected $products;

    /**
     * @var PurchaseOrderBuilderRepository
     */
    protected $purchaseOrders;

    /**
     * PurchaseOrderBuilder constructor.
     *
     * @param  ?int  $destinationWarehouseId
     *
     * @throws BindingResolutionException
     */
    public function __construct(Supplier $supplier, ?int $destinationWarehouseId = null, array $productFilters = [])
    {
        $this->supplier = $supplier;
        $this->productFilters = $productFilters;
        $this->destinationWarehouseId = $destinationWarehouseId;
        $this->purchaseOrders = app()->make(PurchaseOrderBuilderRepository::class);
    }

    public function getSupplier(): Supplier
    {
        return $this->supplier;
    }

    public function setSupplier(Supplier $supplier): void
    {
        $this->supplier = $supplier;
    }

    public function getProductFilters(): array
    {
        return $this->productFilters;
    }

    public function setProductFilters(array $productFilters): void
    {
        $this->productFilters = $productFilters;
    }

    /**
     * @return Product[]|Collection|mixed
     */
    public function getProducts()
    {
        return $this->products;
    }

    protected function getSupplierProducts(): Collection
    {
        return $this->purchaseOrders->getSupplierProducts(
            $this->supplier,
            $this->destinationWarehouseId,
            $this->getProductFilters()
        );
    }

    /**
     * @throws LargePurchaseOrderException
     */
    public function build(): \Illuminate\Support\Collection
    {
        // We fetch the products by the given filters
        $this->products = $this->getSupplierProducts();

        $results = collect($this->buildOrder());
        if (($count = $results->count()) > self::MAX_PRODUCTS_COUNT) {
            throw new LargePurchaseOrderException($count);
        }

        return $results;
    }

    /**
     * @return float|int|mixed
     */
    protected function getUnallocatedBackorderQueueQuantity(Product $product)
    {
        return $this->purchaseOrders->getUnallocatedBackorderQueueQuantity(
            $product,
            $this->supplier->id,
            $this->destinationWarehouseId
        );
    }

    /**
     * @return float|int|mixed
     */
    protected function getBackorderQueueQuantity(Product $product)
    {
        /*
         * Here we return the negative product inventory, maxed out at zero, which should effectively return the qty to
         * cover backorders.
         *
         * We do it this way instead of querying backorders because a) it is more efficient and b) sometimes backorder
         * queue coverages can have bad data and relying on the cached inventory available is less error prone
         *
         * Also, this approach does not limit qty to fulfill backorders to just the supplier in question, but that is ok,
         * because from a business logic perspective a user wants to simply fulfill everything that is needed from the
         * supplier they are building the po for.
         */
        return max(0, -$product->productInventory->where('warehouse_id', $this->destinationWarehouseId)->sum('inventory_available'));
    }

    /**
     * Builds the purchase order.
     */
    abstract protected function buildOrder(): array;
}
