<?php

namespace App\Services\PurchaseOrder\PurchaseOrderBuilder\Builders;

use App\Models\Product;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\PurchaseOrderBuilder;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\PurchaseOrderBuilderLine;
use Illuminate\Contracts\Container\BindingResolutionException;

/**
 * Class TargetStockLevelPurchaseOrderBuilder.
 */
class TargetStockLevelPurchaseOrderBuilder extends PurchaseOrderBuilder
{
    protected int|float $quantity = 0;

    protected bool $useMOQ = false;

    /**
     * @throws BindingResolutionException
     */
    public function __construct(
        int $quantity,
        Supplier $supplier,
        int $destinationWarehouseId,
        array $productFilters = [],
        bool $useMOQ = false
    ) {
        $this->quantity = abs($quantity);
        $this->useMOQ = $useMOQ;
        parent::__construct($supplier, $destinationWarehouseId, $productFilters);
    }

    /**
     * {@inheritDoc}
     */
    protected function buildOrder(): array
    {
        // When targeting stock level, no sales history is needed.
        // We simply ensure that each stock has in excess of any
        // unallocated backorder queue quantity by the provided quantity
        // after adjusting for available and in-transit quantity
        $results = [];
        /** @var Product $product */
        foreach ($this->getProducts() as $current) {
            $inventoryAdjustment = max(0, $current->inventory_available ?: 0) + max(0, $current->inventory_inbound ?: 0);

            /** @var SupplierProduct $supplierProduct */
            $supplierProduct = $current->supplierProducts->where('supplier_id', $this->supplier->id)->first();

            $backordered = max(0, -($current->inventory_available ?: 0)) + $current->parent_backordered_quantity ?? 0;

            // Only add positive quantities
            $quantityCalculated = max(0, $backordered + $this->quantity - $inventoryAdjustment);

            $minimumOrderQuantity = 0;
            /*
             * Without the quantityCalcuated > 0 condition, this brings products that have 0 quantity calculated up to
             * this point to 1. This is not the desired behavior, so we only do this if the quantity is greater than 0.
             * MOQ is intended to apply a minimum to the quantities for products already intended to be ordered.
             *
             * All supplier products are present up to this point (the ones that don't need ordering will have qty 0),
             * since in the PurchaseOrderBuilderRepository::getSupplierProducts() method, we are using a left join on
             * inventory movements.
             */
            if ($this->useMOQ && $quantityCalculated > 0) {
                $minimumOrderQuantity = $supplierProduct?->productMinimumOrderQuantity ?: 0;
            }

            $quantityNeeded = max($minimumOrderQuantity, $backordered + $this->quantity - $inventoryAdjustment);

            if ($quantityNeeded <= 0) {
                continue;
            }

            $results[] = (new PurchaseOrderBuilderLine(
                $supplierProduct,
                $quantityNeeded,
                $this->destinationWarehouseId,
                [
                    'num_backordered' => $backordered,
                    'num_ordered' => $inventoryAdjustment,
                    'num_stock' => $inventory->inventory_available ?? 0,
                    'quantity_calculated' => $quantityCalculated,
                    'quantity_needed' => $quantityNeeded,
                ]
            ))->withSalesOrderLine(
                id: $current->sales_order_line_id,
                salesOrderId: $current->sales_order_id,
                salesOrderNumber: $current->sales_order_number,
                backorderQueueId: $current->backorder_queue_id,
            );
        }

        return $results;
    }
}
